#include <string.h> /* NULL, movedata() */
#include <stdarg.h> /* va_list, va_start(), va_arg(), va_end() */
#include <dos.h> /* outportb(), inportb(), disable(), enable() */

/* this must agree with the GDT layout in STARTUP.ASM */
#define	LINEAR_SEL	0x08

#define	TIMER_VECT	0x20
#define	SYSCALL_VECT	0x30
#define	SYS_CPUTS	0

/* vector type and macros for portability  */
#define	INTERRUPT	/* nothing */

#define	SAVE_VECTOR(vec,num)		save_vector(vec, num)
#define	INSTALL_HANDLER(vec,num,fn)	install_handler(vec, num, (handler_t)fn)
#define	RESTORE_VECTOR(vec,num)		restore_vector(vec, num)

#ifdef __cplusplus
extern "C"
{
#endif

/* code in STARTUP.ASM depends on the layout of this structure */
typedef struct
{
	unsigned handler_adr;
} vector_t;

/* IMPORTS:
from STARTUP.ASM */
extern unsigned long _virt_to_phys;

typedef void (*handler_t)();

void save_vector(vector_t *vec, unsigned num);
void install_handler(vector_t *vec, unsigned num, handler_t handler);
void restore_vector(vector_t *vec, unsigned num);

/* layout of this structure must match the order of registers
pushed and popped by the exception handlers in STARTUP.ASM */
typedef struct
{
/* pushed by pusha */
	unsigned edi, esi, ebp, esp, ebx, edx, ecx, eax;
/* pushed separately */
	unsigned ds, es, fs, gs;
	unsigned which_int, err_code;
/* pushed by exception. Exception may also push err_code.
user_esp and user_ss are pushed only if a privilege change occurs. */
	unsigned eip, cs, eflags, user_esp, user_ss;
} regs_t;

#ifdef __cplusplus
}
#endif

static unsigned short *_fb_adr;
static unsigned _crtc_io_adr, _attrib, _csr_x, _csr_y, _vc_width, _vc_height;
/*****************************************************************************
*****************************************************************************/
static void memsetw(unsigned short *dst, unsigned val, size_t count)
{
	for(; count != 0; count--)
	{
		*dst = val;
		dst++;
	}
}
/*****************************************************************************
*****************************************************************************/
static void scroll(void)
{
	unsigned short blank, temp;

	blank = 0x20 | ((unsigned short)_attrib << 8);
/* scroll up */
	if(_csr_y >= _vc_height)
	{
		temp = _csr_y - _vc_height + 1;
		memmove(_fb_adr, _fb_adr + temp * _vc_width,
			(_vc_height - temp) * _vc_width * 2);
/* blank the bottom line of the screen */
		memsetw(_fb_adr + (_vc_height - temp) * _vc_width,
			blank, _vc_width);
		_csr_y = _vc_height - 1;
	}
}
/*****************************************************************************
*****************************************************************************/
static void move_csr(void)
{
	unsigned short off;

	off = _csr_y * _vc_width + _csr_x;
	outportb(_crtc_io_adr + 0, 14);
	outportb(_crtc_io_adr + 1, off >> 8);
	outportb(_crtc_io_adr + 0, 15);
	outportb(_crtc_io_adr + 1, off);
}
/*****************************************************************************
*****************************************************************************/
static void putch(unsigned char c)
{
	unsigned short att;

	att = (unsigned short)_attrib << 8;
/* backspace */
	if(c == 0x08)
	{
		if(_csr_x != 0)
			_csr_x--;
	}
/* tab */
	else if(c == 0x09)
		_csr_x = (_csr_x + 8) & ~(8 - 1);
/* carriage return */
	else if(c == '\r')	/* 0x0D */
		_csr_x = 0;
/* line feed */
//	else if(c == '\n')	/* 0x0A */
//		_csr_y++;
/* CR/LF */
	else if(c == '\n')	/* ### - 0x0A again */
	{
		_csr_x = 0;
		_csr_y++;
	}
/* printable ASCII */
	else if(c >= ' ')
	{
		unsigned short *where;

		where = _fb_adr + _csr_y * _vc_width + _csr_x;
		*where = c | att;
		_csr_x++;
	}
	if(_csr_x >= _vc_width)
	{
		_csr_x = 0;
		_csr_y++;
	}
	scroll();
	move_csr();
}
/*****************************************************************************
*****************************************************************************/
static void cputs(char *str)
{
	while(*str != '\0')
	{
		putch(*str);
		str++;
	}
}
/*****************************************************************************
*****************************************************************************/
static int kprintf_help(char c, void **ptr)
{
	putch(c);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
/* flags used in processing format string */
#define		PR_LJ	0x01	/* left justify */
#define		PR_CA	0x02	/* use A-F instead of a-f for hex */
#define		PR_SG	0x04	/* signed numeric conversion (%d vs. %u) */
#define		PR_32	0x08	/* long (32-bit) numeric conversion */
#define		PR_16	0x10	/* short (16-bit) numeric conversion */
#define		PR_WS	0x20	/* PR_SG set and num was < 0 */
#define		PR_LZ	0x40	/* pad left with '0' instead of ' ' */
#define		PR_FP	0x80	/* pointers are far */

/* largest number handled is 2^32-1, lowest radix handled is 8.
2^32-1 in base 8 has 11 digits (add 5 for trailing NUL and for slop) */
#define		PR_BUFLEN	16

typedef int (*fnptr_t)(char c, void **helper);

static int do_printf(const char *fmt, va_list args, fnptr_t fn, void **ptr)
{
	unsigned char state, flags, radix, actual_wd;
	unsigned short count, given_wd;
	char *where, buf[PR_BUFLEN];
	long num;

	state = flags = count = given_wd = 0;
/* begin scanning format specifier list */
	for(; *fmt; fmt++)
	{
		switch(state)
		{
/* STATE 0: AWAITING % */
		case 0:
			if(*fmt != '%')	/* not %... */
			{
				fn(*fmt, ptr);	/* ...just echo it */
				count++;
				break;
			}
/* found %, get next char and advance state to check if next char is a flag */
			state++;
			fmt++;
			/* FALL THROUGH */
/* STATE 1: AWAITING FLAGS (%-0) */
		case 1:
			if(*fmt == '%')	/* %% */
			{
				fn(*fmt, ptr);
				count++;
				state = flags = given_wd = 0;
				break;
			}
			if(*fmt == '-')
			{
				if(flags & PR_LJ)/* %-- is illegal */
					state = flags = given_wd = 0;
				else
					flags |= PR_LJ;
				break;
			}
/* not a flag char: advance state to check if it's field width */
			state++;
/* check now for '%0...' */
			if(*fmt == '0')
			{
				flags |= PR_LZ;
				fmt++;
			}
			/* FALL THROUGH */
/* STATE 2: AWAITING (NUMERIC) FIELD WIDTH */
		case 2:
			if(*fmt >= '0' && *fmt <= '9')
			{
				given_wd = 10 * given_wd +
					(*fmt - '0');
				break;
			}
/* not field width: advance state to check if it's a modifier */
			state++;
			/* FALL THROUGH */
/* STATE 3: AWAITING MODIFIER CHARS (FNlh) */
		case 3:
			if(*fmt == 'F')
			{
				flags |= PR_FP;
				break;
			}
			if(*fmt == 'N')
				break;
			if(*fmt == 'l')
			{
				flags |= PR_32;
				break;
			}
			if(*fmt == 'h')
			{
				flags |= PR_16;
				break;
			}
/* not modifier: advance state to check if it's a conversion char */
			state++;
			/* FALL THROUGH */
/* STATE 4: AWAITING CONVERSION CHARS (Xxpndiuocs) */
		case 4:
			where = buf + PR_BUFLEN - 1;
			*where = '\0';
			switch(*fmt)
			{
			case 'X':
				flags |= PR_CA;
				/* FALL THROUGH */
/* xxx - far pointers (%Fp, %Fn) not yet supported */
			case 'x':
			case 'p':
			case 'n':
				radix = 16;
				goto DO_NUM;
			case 'd':
			case 'i':
				flags |= PR_SG;
				/* FALL THROUGH */
			case 'u':
				radix = 10;
				goto DO_NUM;
			case 'o':
				radix = 8;
/* load the value to be printed. l=long=32 bits: */
DO_NUM:				if(flags & PR_32)
					num = va_arg(args, unsigned long);
/* h=short=16 bits (signed or unsigned) */
				else if(flags & PR_16)
				{
					if(flags & PR_SG)
						num = va_arg(args, short);
					else
						num = va_arg(args, unsigned short);
				}
/* no h nor l: sizeof(int) bits (signed or unsigned) */
				else
				{
					if(flags & PR_SG)
						num = va_arg(args, int);
					else
						num = va_arg(args, unsigned int);
				}
/* take care of sign */
				if(flags & PR_SG)
				{
					if(num < 0)
					{
						flags |= PR_WS;
						num = -num;
					}
				}
/* convert binary to octal/decimal/hex ASCII
OK, I found my mistake. The math here is _always_ unsigned */
				do
				{
					unsigned long temp;

					temp = (unsigned long)num % radix;
					where--;
					if(temp < 10)
						*where = temp + '0';
					else if(flags & PR_CA)
						*where = temp - 10 + 'A';
					else
						*where = temp - 10 + 'a';
					num = (unsigned long)num / radix;
				}
				while(num != 0);
				goto EMIT;
			case 'c':
/* disallow pad-left-with-zeroes for %c */
				flags &= ~PR_LZ;
				where--;
				*where = (char)va_arg(args, char);
				actual_wd = 1;
				goto EMIT2;
			case 's':
/* disallow pad-left-with-zeroes for %s */
				flags &= ~PR_LZ;
				where = va_arg(args, char *);
EMIT:
				actual_wd = strlen(where);
				if(flags & PR_WS)
					actual_wd++;
/* if we pad left with ZEROES, do the sign now */
				if((flags & (PR_WS | PR_LZ)) ==
					(PR_WS | PR_LZ))
				{
					fn('-', ptr);
					count++;
				}
/* pad on left with spaces or zeroes (for right justify) */
EMIT2:				if((flags & PR_LJ) == 0)
				{
					while(given_wd > actual_wd)
					{
						fn(flags & PR_LZ ? '0' :
							' ', ptr);
						count++;
						given_wd--;
					}
				}
/* if we pad left with SPACES, do the sign now */
				if((flags & (PR_WS | PR_LZ)) == PR_WS)
				{
					fn('-', ptr);
					count++;
				}
/* emit string/char/converted number */
				while(*where != '\0')
				{
					fn(*where++, ptr);
					count++;
				}
/* pad on right with spaces (for left justify) */
				if(given_wd < actual_wd)
					given_wd = 0;
				else given_wd -= actual_wd;
				for(; given_wd; given_wd--)
				{
					fn(' ', ptr);
					count++;
				}
				break;
			default:
				break;
			}
		default:
			state = flags = given_wd = 0;
			break;
		}
	}
	return count;
}
/*****************************************************************************
*****************************************************************************/
void kprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	(void)do_printf(fmt, args, kprintf_help, NULL);
	va_end(args);
}
/*****************************************************************************
*****************************************************************************/
static void panic(const char *fmt, ...)
{
	va_list args;

	disable();
	va_start(args, fmt);
	(void)do_printf(fmt, args, kprintf_help, NULL);
	while(1)
		/* freeze */;
}
/*****************************************************************************
increment char in upper left corner of screen on every timer tick
*****************************************************************************/
static void timer_int(void)
{
	(*(unsigned char *)_fb_adr)++;
/* reset 8259 interrupt chip */
	outportb(0x20, 0x20);
}
/*****************************************************************************
*****************************************************************************/
static void syscall_int(volatile regs_t regs)
{
	switch(regs.eax)
	{
		case SYS_CPUTS:
			cputs((char *)regs.esi);
			break;
	}
}
/*****************************************************************************
*****************************************************************************/
static void fault(volatile regs_t regs)
{
	static const char *msg[] =
	{
		"divide error", "debug exception", "NMI", "INT3",
		"INTO", "BOUND exception", "invalid opcode", "no coprocessor",
		"double fault", "coprocessor segment overrun",
			"bad TSS", "segment not present",
		"stack fault", "GPF", "page fault", "??",
		"coprocessor error", "alignment check", "??", "??",
		"??", "??", "??", "??",
		"??", "??", "??", "??",
		"??", "??", "??", "??"
	};

	kprintf("Exception #%u ", regs.which_int);
	if(regs.which_int <= sizeof(msg) / sizeof(char *))
		kprintf("(%s) ", msg[regs.which_int]);
	panic("in kernel mode. System halted\n");
}
/*****************************************************************************
*****************************************************************************/
static void init_8259s(void)
{
	outportb(0x20, 0x11); /* ICW1 */
	outportb(0xA0, 0x11);

	outportb(0x21, 0x20); /* ICW2: route IRQs 0...7 to INTs 20h...27h */
	outportb(0xA1, 0x28); /* ...IRQs 8...15 to INTs 28h...2Fh */

	outportb(0x21, 0x04); /* ICW3 */
	outportb(0xA1, 0x02);

	outportb(0x21, 0x01); /* ICW4 */
	outportb(0xA1, 0x01);
/* enable IRQ0 (timer) only */
	outportb(0x21, ~0x01);
	outportb(0xA1, ~0x00);
}
/*****************************************************************************
*****************************************************************************/
#define	VGA_MISC_READ	0x3CC

int main(void)
{
	static const char *msg = "     Press RESET button to end\n\r";
/**/
	vector_t old_syscall_vector;
	unsigned i;

/* set up video */
	if((inportb(VGA_MISC_READ) & 0x01) != 0)
	{
		_fb_adr = (unsigned short *)(0xB8000L - _virt_to_phys);
		_crtc_io_adr = 0x3D4;
	}
	else
	{
		_fb_adr = (unsigned short *)(0xB0000L - _virt_to_phys);
		_crtc_io_adr = 0x3B4;
	}
	_attrib = 0x1F; /* bright white on blue */
	_vc_width = 80;
	_vc_height = 25;
/* */
	kprintf("Attempting syscall with do-nothing handler...\n");
	__asm__("int %0"
		:
		: "i"(SYSCALL_VECT),
		"a"(SYS_CPUTS),
		"S"((unsigned)msg)
		);
	SAVE_VECTOR(&old_syscall_vector, SYSCALL_VECT);
/* */
	kprintf("Installing exception handlers...\n");
	for(i = 0; i <= SYSCALL_VECT; i++)
		INSTALL_HANDLER(NULL, i, fault);
	kprintf("Installing syscall handler...\n");
	INSTALL_HANDLER(&old_syscall_vector, SYSCALL_VECT, syscall_int);
/* */
	kprintf("Attempting syscall with C language handler...\n");
	__asm__("int %0"
		:
		: "i"(SYSCALL_VECT),
		"a"(SYS_CPUTS),
		"S"((unsigned)msg)
		);
/* */
	kprintf("Restoring do-nothing syscall handler...\n");
	RESTORE_VECTOR(&old_syscall_vector, SYSCALL_VECT);
	kprintf("Attempting syscall with restored do-nothing handler...\n");
	__asm__("int %0"
		:
		: "i"(SYSCALL_VECT),
		"a"(SYS_CPUTS),
		"S"((unsigned)msg)
		);
/* */
	kprintf("Installing timer interrupt handler...\n");
	INSTALL_HANDLER(NULL, TIMER_VECT, timer_int);
	kprintf("Reprogramming 8259 chips...\n");
	init_8259s();
	kprintf("Enabling timer interrupt...\n");
	enable();
/* freeze */
	while(1)
		/* nothing */;
	return 0;
}
