;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; First-stage bootloader for FAT12 (DOS) floppy disk
; Chris Giese <geezer@execpc.com>, http://www.execpc.com/~geezer/os
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

SS_ADR	equ	10000h

SS_ORG	equ	100h

%define	SS_NAME	"LOADER  BIN"

;
; "No user-serviceable parts beyond this point" :)
; Actually, you can modify it if you want.
;

FAT12_EOF	equ	0FF8h

; first-stage address
%ifdef DOS
%define FS_ADR		100h	; .COM file, for testing purposes only
%else
%define FS_ADR		7C00h
%endif

ADR_STACK	equ	(FS_ADR + 400h)

; one-sector directory buffer. FAT sectors may be as large as 4K
ADR_DIRBUF	equ	(ADR_STACK)

; two-sector FAT buffer -- two sectors because FAT12
; entries are 12 bits and may straddle a sector boundary
ADR_FATBUF	equ	(FS_ADR + 1400h)

	ORG FS_ADR
start:

; define memory used for scratchpad variables. These are initialized
; at run-time, so there's no need to waste first-stage memory on them.
; Put them just below 'start'. If this code is built as a .COM file,
; these variables may overwrite the command tail in the PSP -- no loss.

; (1 byte) drive we booted from; 0=A, 80h=C
boot_drive	EQU (start - 1)

; (2 bytes) sector where root directory starts
root_start	EQU (boot_drive - 2)

; (2 bytes) sector where the actual disk data starts
data_start	EQU (root_start - 2)

; (2 bytes) number of 16-byte paragraphs per sector
para_per_sector	EQU (data_start - 2)

; (2 bytes) number of 16-byte paragraphs per cluster
para_per_cluster EQU (para_per_sector - 2)

	jmp short over		; skip over BPB
	nop

; minimal, default BIOS Parameter Block (BPB)

oem_id:			; 03h	not used by this code
	db "GEEZER", 0, 0

bytes_per_sector:	; 0Bh
	dw 512

sectors_per_cluster:	; 0Dh
	db 1

fat_start:
num_reserved_sectors:	; 0Eh
	dw 1

num_fats:		; 10h
	db 2

num_root_dir_ents:	; 11h
	dw 224

total_sectors:		; 13h	not used by this code
	dw 2880		; 2880 for 1.44 meg, 3360 for 1.68 meg
 ;	dw 3360

media_id:		; 15h	not used by this code
	db 0F0h

sectors_per_fat:	; 16h
	dw 9

sectors_per_track:	; 18h
	dw 18		; 18 for 1.44 meg, 21 for 1.68 meg
 ;	dw 21

heads:			; 1Ah
	dw 2

hidden_sectors:		; 1Ch
	dd 0

total_sectors_large:	; 20h	not used by this code
	dd 0

over:
%ifdef DOS
	xor dl,dl
%else
; evidently some buggy BIOSes jump to 07C0:0000, so fix that now
	jmp 0:over2
%endif
over2:
; ...and some BIOSes don't zero DS, so do that as well
	mov ax,cs
	mov ds,ax
	mov ss,ax				; zero SS, too
	mov sp,ADR_STACK
	mov bp,over				; for relative addressing
	cld					; string operations go up

; save [boot_drive] from DL. BIOS sets DL=0 if
; booting from drive A, DL=80h if booting from drive C
	mov [bp - (over - boot_drive)],dl

; compute first sector of root directory
	mov al,[num_fats]			; number of FATs
	cbw					; "mov ah,0"
	mul word [bp - (over - sectors_per_fat)] ; multiply by sectors/FAT
	add ax,[bp - (over - fat_start)]	; plus reserved sectors
	mov [root_start],ax

; compute first sector of disk data area
	mov bx,ax
	mov ax,[num_root_dir_ents]		; entries in root dir
	mov dl,32				; * bytes/entry (assume DH=0)
	mul dx					; == bytes in root dir
	div word [bp - (over - bytes_per_sector)] ; / bytes per sector
	add ax,bx				; = sectors
	mov [data_start],ax

; compute number of 16-byte paragraphs per disk sector
	mov dx,[bp - (over - bytes_per_sector)]
	mov cl,4
	shr dx,cl
	mov [bp - (over - para_per_sector)],dx

; compute number of 16-byte paragraphs per FAT cluster
	mov al,[sectors_per_cluster]
	cbw
	mul dx					; DX still=paragraphs/sector
	mov dx,ax
	mov [bp - (over - para_per_cluster)],dx

; OK, everything is set up for 'find_file', 'walk_fat', and 'read_sectors'
; Find the file named at 'second_stage:' in the root directory
	mov si,second_stage
	call find_file
	jc err					; disk error

; if second-stage file not found, display blinking 'F'
	mov ax,9F46h
	jne err2

; get conventional memory size
	int 12h

; subtract starting second stage address to get available mem
; xxx - fix this for .COM file
	sub ax,(SS_ADR >> 10)

; convert from K to bytes
	mov dx,1024
	mul dx

; if second stage file is too big...
	sub ax,[es:di + 28]
	sbb dx,[es:di + 30]

; ...display a blinking 'M'
	mov ax,9F4Dh
	jc err2

; found second-stage, load it
%ifdef DOS
	mov di,ds
	add di,(SS_ADR >> 4)
%else
	mov di,(SS_ADR >> 4)
%endif
load_2nd:
	mov es,di

; convert cluster BX to sector value in DX:AX, and get next cluster in BX
	call walk_fat
	jc err

; read an entire cluster
	xor ch,ch
	mov cl,[bp - (over - sectors_per_cluster)]
	call read_sectors
	jc err
	add di,[bp - (over - para_per_cluster)]	; advance mem ptr 1 cluster
	cmp bx,FAT12_EOF			; EOF cluster value
	jb load_2nd

; turn off floppy motor
	mov dx,3F2h
	xor al,al
	out dx,al

; jump to second stage that is ORGed to address "SS_ORG"
%ifdef DOS
	mov ax,ds
	add ax,((SS_ADR - SS_ORG) >> 4)
%else
	mov ax,((SS_ADR - SS_ORG) >> 4)
%endif
	mov ds,ax
	mov es,ax

	push ax
	push word SS_ORG
	retf

; disk read error; display blinking 'R'
err:
	mov ax,9F52h
err2:
	mov bx,0B800h	; xxx - assumes color emulation
	mov es,bx
	xor bx,bx
	mov [es:bx],ax

	mov ax,0E07h	; *** BEEEP ***
	int 10h

	mov ah,0	; await key pressed
	int 16h

%ifdef DOS
	mov ax,4C01h	; DOS terminate
	int 21h
%else
	int 19h		; re-start the boot process
%endif

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			find_file
; action:		searches for file in root dir
; in:			11-char all-caps filename at SI, [root_start] set
; out (disk error):	CY=1
; out (file not found):	CY=0, ZF=0
; out (found it):	CY=0, ZF=1, BX=starting cluster of file,
;			ES:DI is left pointing to the directory entry
; modifies:		AX, BX, CX, DX, DI, ES
; minimum CPU:		8088
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

find_file:
		mov ax,[root_start]		; 1st sector in root dir
%ifdef DOS
		mov bx,ds			; where to load it
		add bx,(ADR_DIRBUF >> 4)
%else
		mov bx,(ADR_DIRBUF >> 4)
%endif
		mov es,bx
find_file_1:
		mov cx,1			; one sector at a time

		call read_sectors
		jc find_file_5			; disk read error
		xor di,di
find_file_2:
		test [es:di],byte 0FFh		; 0 byte = end of root dir
		je find_file_3
		mov cl,11			; 'read_sectors' zeroed CH
		push si
		push di				; compare filename to dirent
			rep cmpsb
		pop di
		pop si
		je find_file_4

; if filename comparision failed, advance 32 bytes to next dir entry
		add di,byte 32
		cmp di,[bp - (over - bytes_per_sector)]
		jb find_file_2
		inc ax
		cmp ax,[bp - (over - data_start)]
		jb find_file_1			; go to next sector of root

find_file_3:
; did not find the file: return with CY=0, ZF=0
		or al,1
find_file_4:
; found the file
		mov bx,[es:di + 26]		; get first cluster of file
find_file_5:
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			walk_fat
; action:		converts cluster value to sector,
;			and gets next cluster of file from FAT
; in:			BX=cluster
; out (disk error):	CY=1
; out (success):	CY=0, AX=first sector of cluster, BX=next cluster
; modifies:		AX, BX, DX, SI
; minimum CPU:		8088
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

walk_fat:
	push es
	push cx
		mov ax,bx

; cluster 2 is the first data cluster
		dec ax
		dec ax

; convert from clusters to sectors
		mov dh,0
		mov dl,[bp - (over - sectors_per_cluster)]
		mul dx
		add ax,[bp - (over - data_start)]
		adc dx,byte 0

; AX is return value; save it
		push ax

%ifdef DOS
			mov ax,ds
			add ax,(ADR_FATBUF >> 4)
%else
			mov ax,(ADR_FATBUF >> 4)
%endif
			mov es,ax

; FAT12 entries are 12 bits, bytes are 8 bits. Ratio is 3 / 2,
; so multiply cluster by 3, and divide by 2 later.
			mov ax,bx
			shl ax,1		; entry * 2
			add ax,bx		; + entry = entry * 3

			mov bx,ax
; BX:0		=use high or low 12 bits of 16-bit value read from FAT
; BX:9...BX:1	=byte offset into FAT sector (9 assumes 512-byte sectors)
; BX:?...BX:10	=which sector of FAT to load

; figure out which FAT sectors to load
			shr ax,1		; entry / 2
			xor dx,dx
			div word [bp - (over - bytes_per_sector)]

; remainder is byte offset into FAT; put it in SI
			mov si,dx

; quotient in AX is FAT sector: add FAT starting sector
			add ax,[bp - (over - fat_start)]

; check the FAT buffer to see if this sector is already loaded
			cmp ax,[bp - (over - curr_sector)]
			je walk_fat_1
			mov [curr_sector],ax

; read the target FAT sector plus the sector after it
; (in case the 12-bit FAT entry straddles the two sectors)
			xor dx,dx
			mov cx,2
			call read_sectors
			jc walk_fat_4
; get 16 bits from FAT
walk_fat_1:
			mov ax,[es:si]

; look at BX:0 to see if we want the high 12 bits or the low 12 bits
			shr bx,1
			jc walk_fat_2
			and ax,0FFFh		; CY=1: use low 12 bits
			jmp short walk_fat_3
walk_fat_2:
			mov cl,4
			shr ax,cl		; CY=0: use high 12 bits
walk_fat_3:
			mov bx,ax

; clear CY bit to signal success
			xor dx,dx
walk_fat_4:
		pop ax

	pop cx
	pop es
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			read_sectors
; action:		reads one or more disk sectors into memory
; in:			ES:0=address of memory where sectors should be read
;			AX=first sector to read
;			CX=number of sectors to read
; out (disk error):	CY=1
; out (success):	CY=0
; modifies:		DX, CX
; minimum CPU:		8088
; notes:		on return from this function, CX=0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

read_sectors:
	push es
	push bx
	push ax
		xor bx,bx
read_sectors_1:
		push cx
		push ax

; S = sector		= LBA % sectors_per_track + 1
; H = head		= LBA / sectors_per_track % heads
; C = cylinder (track)  = LBA / sectors_per_track / heads
;
; divide LBA sector number (in AX) by 8-bit sectors_per_track
			mov cx,[bp - (over - sectors_per_track)]
			div cl

; remainder (in AH) is sector-1; store in CL and add 1
			mov cl,ah
			inc cl

; divide quotient from above (in AL) by heads
			xor ah,ah
			mov dx,[bp - (over - heads)]	; DH=heads
			div dl

; remainder (in AH) is head; quotient (in AL) is cylinder (track)
; put everything in the proper register for INT 13h

			mov dh,ah	; DH=head
			mov dl,[bp - (over - boot_drive)]

			mov ch,al	; CH=cylinder 7:0
			and cl,3Fh	; CL 7:6 = cylinder 9:8 = 0
; now: read one sector
			mov ax,0201h
			int 13h
			jnc read_sectors_3

; disk error; recalibrate/reset drive
			mov ah,0
			int 13h
			jc read_sectors_2
; try the read again. If it fails a second time, give up.
			mov ax,0201h
			int 13h
			jnc read_sectors_3
read_sectors_2:
		pop ax
		pop cx
		jmp short read_sectors_5

; advance memory pointer
read_sectors_3:
			mov ax,es
			add ax,[bp - (over - para_per_sector)]
			mov es,ax
		pop ax
		pop cx

; increment AX to advance to next sector
		inc ax
		loop read_sectors_1

; clear CY bit to signal success
		clc
read_sectors_5:
	pop ax
	pop bx
	pop es
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 1st-stage data
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

curr_sector:	; which sector is in the FAT buffer
	dw -1

; name of the second stage bootloader
second_stage:
	db SS_NAME

; pad with NOPs to offset 510
	times (510 + $$ - $) nop

; 2-byte magic bootsector signature
	db 55h, 0AAh
