;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; coffload.nsm - DJGPP/COFF kernel loader
; Copyright (C) 2000 - Christopher Giese <geezer@execpc.com>
; http://www.execpc.com/~geezer/os/
;
; Assemble with NASM:
;	nasm -f bin -o coffload.com coffload.nsm
; then attach a COFF kernel:
;	copy /b coffload.com + krnl.cof  krnl.com
;
; krnl.cof can be compiled with DJGPP, but requires special linking:
;	- must be exactly three sections (.text, .data, .bss)
;	- the sections must be in the order shown
;	- COFF file must have 28-byte a.out header
;		(these three conditions are normally the case with DJGPP)
;	- the sections must all reside in the same 4 Mbyte virtual
;		address space
;	- each section must start on a page (4 Kbyte) boundary
;
; Bugs/to do:
;	- what does CMOS return for extended memory size if more than 64M?
;	- ConvMemSize not set if kernel will fit in extended memory
;	- maybe enable A20 only if kernel going above 1 meg
;	- use VCPI for pmode switch if EMM386 loaded?
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	ORG 100h				; .COM file

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Check for Virtual 8086 mode. Code from Freedows '98 ldr_asm.asm
; Copyright (C) 1997 Joachim Breitsprecher <j.breitsprecher@schwaben.de>
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov cl,1			; error 1: CPU in V86 mode
	smsw ax
	test al,1			; look at PE bit of MSW (CR0)
	jne die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Check for 32-bit CPU (386SX or better)
; I (Chris Giese) have not tested this code with 8088/'286 systems
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov cl,2			; error 2: not a 32-bit CPU

	pushf
		pushf			; FLAGS -> AX
		pop ax
		xor ax,7000h		; try changing b14:b12
		push ax			; AX -> FLAGS
		popf
		pushf			; FLAGS -> BX
		pop bx
	popf
	cmp bx,ax			; 80286 clears b14:b12 to 000
	jne die				; 80(1)86/88 sets b15:b12 to 1111

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Validate attached COFF image
; COFF info from /djgpp/include/coff.h of DJGPP
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov cl,3			; error 3: bad/missing COFF kernel

	cmp word [_coff_magic],014Ch
	jne die
	cmp word [_num_sects],3
	jne die
	cmp word [_opt_hdr_size],28
	jne die
	test word [_coff_flags],2
	je die

	cmp word [_aout_magic],010Bh
	jne die

	cmp dword [_code_name],".tex"
	jne die
	cmp dword [_data_name],".dat"
	jne die
	cmp dword [_bss_name],".bss"
	je sex_ok			; that's short for "sections" :)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Exit to DOS with error code and message
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

die:	push cx				; save error number
		lea si,[_msgs]		; point to error messages
		cld
die1:		inc si
		cmp byte [si - 1],0	; skip to specific msg per CL
		jne die1
		loop die1
		mov ah,0Eh		; INT 10h: teletype output
		xor bx,bx		; video page 0
		jmp die3
die2:		int 10h
die3:		lodsb
		or al,al
		jne die2
	pop ax				; error number -> AL
	mov ah,4Ch			; DOS terminate
	int 21h

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; More COFF validation: make sure all sections in same 4 Mbyte address space
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

sex_ok:	mov eax,[_code_virt]
	mov edx,eax
	shr edx,22			; page directory index -> DX

	add eax,[_code_size]
	dec eax
	shr eax,22
	cmp ax,dx
	jne die

	mov eax,[_data_virt]
	mov ebx,eax
	shr ebx,22
	cmp bx,dx
	jne die

	add eax,[_data_size]
	dec eax
	shr eax,22
	cmp ax,dx
	jne die

	mov eax,[_bss_virt]
	mov ebx,eax
	shr ebx,22
	cmp bx,dx
	jne die

	add eax,[_bss_size]
	dec eax
	shr eax,22
	cmp ax,dx
	jne die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Make sure sections start on page (4 Kbyte) boundaries
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov eax,[_code_virt]
	or eax,[_data_virt]
	or eax,[_bss_virt]
	test ax,0FFFh
	jne near die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Compute kernel size
; Note that the BSS includes the initial kernel stack and page tables
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ebx,[_c_size]
	add ebx,[_d_size]
	add ebx,[_b_size]
	mov [_krnl_size],ebx

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Find top of RAM: read extended memory size from CMOS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	xor eax,eax
	mov al,18h
	out 70h,al
	in al,71h
	mov ah,al
	mov al,17h
	out 70h,al
	in al,71h
	shl eax,10			; Kbytes -> bytes
	je no_ext_mem
	add eax,100000h			; extended memory size -> _ram_top
	jmp ext_mem_ok

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; No extended memory; read conventional memory size from CMOS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

no_ext_mem:
	mov al,16h
	out 70h,al
	in al,71h
	mov ah,al
	mov al,15h
	out 70h,al
	in al,71h
	shl eax,10			; Kbytes -> bytes
ext_mem_ok:
	mov [_ram_top],eax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; See if there's enough extended memory to load the kernel at 1 meg
;
; Why 1 meg? You want to reserve conventional memory (< 1 meg) for things
; like ISA DMA. You also want the kernel below 4 meg, so only the first
; 4 meg of RAM need be mapped (need only one page table).
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov cl,4			; error 4: not enough RAM
	mov edi,100000h			; load kernel here if there's room
	mov eax,[_krnl_size]
	add eax,edi
	cmp eax,[_ram_top]
	ja near die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Zeroing EFLAGS sets IF=0, which disables interrupts. It also sets
; IOPL=00, so only Ring 0 programs can do I/O, and zeroes the NT bit,
; so IRET does a normal IRET instead of a task-switch.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	push dword 0
	popfd

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Enable A20 line. I tried the code from Freedows '98 ldr_asm.asm
; but it didn't work, so use Linux arch/i386/boot/setup.s
;
; To copy the kernel to extended memory, we need page-alignment.
; HIMEM.SYS/XMS can't do this, so we turn on A20 and use unreal mode to
; access extended memory directly. If the extended memory is in use (e.g.
; by SMARTDRV) and we later try to return to DOS, the system will crash.
;
; xxx - this need be done only if extended memory is used
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	call kbd
	mov al,0D1h
	out 64h,al
	call kbd
	mov al,0DFh
	out 60h,al
	call kbd

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Point _gdt_ptr to the GDT.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	xor ebx,ebx
	mov bx,ds
	shl ebx,4
	add [_gdt_ptr + 2],ebx

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Enable unreal mode. This is identical to real mode with one exception:
; 32-bit addresses no longer cause pseudo-GPF (interrupt 0Dh).
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	push ds
		lgdt [_gdt_ptr]
		mov eax,cr0
		or al,1
		mov cr0,eax		; partial switch to pmode
		mov ax,DATA_SEL		; selector to segment w/ 4G limit
		mov ds,ax
		mov es,ax		; set seg limits in descriptor caches
		mov fs,ax
		mov gs,ax		; (do NOT change CS nor SS)
		mov eax,cr0
		and al,0FEh
		mov cr0,eax		; back to (un)real mode
		nop
	pop ds				; segment regs back to old values,
	xor ax,ax			; but now 32-bit addresses are OK
	mov es,ax
	mov fs,ax
	mov gs,ax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Make sure A20 is really on
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov cl,5			; error 5: can't enable A20

	mov ax,[es:0]
	push ax				; save word at 0
		not ax			; word at 1 Meg = ~word at 0
		mov [es:dword 100000h],ax
		mov ax,[es:dword 100000h] ; read it back
		cmp ax,[es:0]		; fail if word at 1 Meg == word at 0
	pop word [es:0]			; restore word at 0
	je near die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Copy code from COFF image to memory.
; The D bit was cleared with the rest of EFLAGS, above,
; so these copies go up.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	push edi			; save kernel base for later
		mov esi,_end		; point to attached COFF file
		add esi,[_code_off]	; point to code in COFF image
		mov ecx,[_c_size]
		a32
		rep movsb

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Copy data from COFF image to memory
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

		mov esi,_end		; point to attached COFF file
		add esi,[_data_off]	; point to data in COFF image
		mov ecx,[_d_size]
		a32
		rep movsb

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Zero the BSS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

		push edi		; save BSS base
			xor al,al
			mov ecx,[_b_size]
			a32
			rep stosb
		pop edi			; EDI -> BSS base
	pop esi				; ESI -> loaded kernel (code)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Switch to 32-bit protected mode. When the kernel starts:
;
; - ESI points to physical address of loaded kernel code
; - EAX contains pmode data-segment selector, ready to be loaded into
;	DS, SS, etc.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov eax,[EntryPt]		; virtual entry point
	sub eax,[_code_virt]
	add eax,esi			; physical entry point
	mov [_entry],eax
	mov eax,cr0			; GDTR has already been loaded
	or al,1				; turn on pmode
	mov cr0,eax
	mov eax,DATA_SEL
	jmp dword far [_entry]		; go!

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Subroutines:
; await keyboard controller
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

kbd0:	jmp short $+2
	in al,60h
kbd:	jmp short $+2
	in al,64h
	test al,1
	jnz kbd0
	test al,2
	jnz kbd
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; data
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

_msgs:	db 0
	db "CPU is in Virtual 8086 mode", 13, 10, 0
	db "Not a 32-bit CPU", 13, 10, 0
	db "Invalid/missing kernel COFF file attached to this "
	db	".COM file", 13, 10, 0
	db "Not enough contiguous RAM to load kernel", 13, 10, 0
	db "Could not enable A20 gate", 13, 10, 0

_krnl_size:
	dd 0

_ram_top:
	dd 0

_entry:	dd 0				; far address of pmode entry point
	dw CODE_SEL

_gdt_ptr:
	dw _gdt_end - _gdt - 1		; GDT limit
	dd _gdt				; linear, physical adr of GDT

_gdt:	dw 0				; limit 15:0
	dw 0				; base 15:0
	db 0				; base 23:16
	db 0				; type
	db 0				; limit 19:16, flags
	db 0				; base 31:24
CODE_SEL	equ	$-_gdt
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah		; present, ring 0, code, non-conforming, readable
	db 0CFh		; page-granular, 32-bit
	db 0
DATA_SEL	equ	$-_gdt
	dw 0FFFFh
	dw 0
	db 0
	db 92h		; present, ring 0, data, expand-up, writable
	db 0CFh		; page-granular, 32-bit
	db 0
_gdt_end:

_end:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; .COM portion of file ends here; attached COFF file begins
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; COFF header (20 bytes)
_coff_magic	equ $			; must be 014Ch
_num_sects	equ _coff_magic + 2	; must be 3 (.text, .data, .bss)
TimeDate	equ _num_sects + 2
SymTabOff	equ TimeDate + 4
NumSyms		equ SymTabOff + 4
_opt_hdr_size	equ NumSyms + 4		; must be 28
_coff_flags	equ _opt_hdr_size + 2	; b1 must be set
COFFHdrEnd	equ _coff_flags + 2

; a.out ("optional") header (28 bytes)
_aout_magic	equ COFFHdrEnd		; must be 010Bh
Version		equ _aout_magic + 2
_c_size		equ Version + 2
_d_size		equ _c_size + 4
_b_size		equ _d_size + 4
EntryPt		equ _b_size + 4
CVirtAdr	equ EntryPt + 4
DVirtAdr	equ CVirtAdr + 4
AoutHdrEnd	equ DVirtAdr + 4

; section headers (40 bytes each)
_code_name:	equ AoutHdrEnd
CodePhysAdr:	equ _code_name + 8
_code_virt:	equ CodePhysAdr + 4
_code_size:	equ _code_virt + 4
_code_off:	equ _code_size + 4
CodeRelocOff:	equ _code_off + 4
CodeLineNumOff: equ CodeRelocOff + 4
CodeNumRelocs:	equ CodeLineNumOff + 4
CodeNumLineNums:equ CodeNumRelocs + 2
CodeFlags:	equ CodeNumLineNums + 2
CodeHdrEnd:	equ CodeFlags + 4

_data_name:	equ CodeHdrEnd
DataPhysAdr:	equ _data_name + 8
_data_virt:	equ DataPhysAdr + 4
_data_size:	equ _data_virt + 4
_data_off:	equ _data_size + 4
DataRelocOff:	equ _data_off + 4
DataLineNumOff:	equ DataRelocOff + 4
DataNumRelocs:	equ DataLineNumOff + 4
DataNumLineNums:equ DataNumRelocs + 2
DataFlags:	equ DataNumLineNums + 2
DataHdrEnd	equ DataFlags + 4

_bss_name:	equ DataHdrEnd
BSSPhysAdr:	equ _bss_name + 8
_bss_virt:	equ BSSPhysAdr + 4
_bss_size:	equ _bss_virt + 4
BSSOff:		equ _bss_size + 4
BSSRelocOff:	equ BSSOff + 4
BSSLineNumOff:	equ BSSRelocOff + 4
BSSNumRelocs:	equ BSSLineNumOff + 4
BSSNumLineNums:	equ BSSNumRelocs + 2
BSSFlags:	equ BSSNumLineNums + 2
BSSHdrEnd:	equ BSSFlags + 4
