/*****************************************************************************
VFS

EXPORTS:
int stat(char *path, struct stat *statbuf);

DIR *opendir(char *path);
void rewinddir(DIR *dir);
struct dirent *readdir(DIR *dir);
int closedir(DIR *dir);

int open(const char *path, unsigned access);
long lseek(unsigned handle, long offset, int whence);
off_t read(unsigned handle, void HUGE *buf, off_t count);
int close(unsigned handle);
*****************************************************************************/
#include <sys/stat.h> /* S_IFDIR, struct stat */
#include <stdlib.h> /* calloc(), free() */
#include <string.h> /* strcpy() */
#include <dirent.h>
#include <errno.h>
#include <io.h> /* SEEK_... */
#include "defs.h"

/* IMPORTS
from MOUNTS.C */
int get_mount(const char **path_p, mount_t **mount_p);

#define	MAX_FILE	20

static file_t _files[MAX_FILE];
/*****************************************************************************
*****************************************************************************/
static int find_free_file(file_t **file_ptr, unsigned *handle_ptr)
{
	unsigned handle;
	file_t *file;

/* find an unused file_t in _files[] */
	file = _files + 0;
	for(handle = 0; handle < MAX_FILE; handle++)
	{
		if(!file->is_open)
			break;
		file++;
	}
	if(handle >= MAX_FILE)
		return -EMFILE;
/* zero the file_t */
	memset(file, 0, sizeof(file_t));
/* ...except for is_open */
	file->is_open = 1;
	(*file_ptr) = file;
	(*handle_ptr) = handle;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int do_open(file_t *file, const char *path)
{
	const char *slash;
	int err, done;

/* remove the mount point from path and set file->mount */
	err = get_mount(&path, &file->mount);
	if(err != 0)
		return err;
/* make sure FS-specific find() works */
	if(file->mount == NULL || file->mount->fs_info == NULL ||
		file->mount->fs_info->find == NULL)
			return -ENOIMP;
	err = file->mount->fs_info->find(file, "/", 1);
	if(err != 0)
		return err;
/* if end of path, we are done */
	if(path[0] == '\0')
		return 0;
	done = 0;
	do
	{
/* pick out one name in path */
		slash = strchr(path, '/');
		if(slash == NULL)
		{
			slash = path + strlen(path);
			done = 1;
		}
/* look for file "path" in (sub)directory "file"
if found, close (sub)directory "file" and re-open it as "path"
i.e. "morph" the directory entry into the new current directory */
		err = file->mount->fs_info->find(file, path, slash - path);
		if(err != 0)
			return err;
/* advance to next name+ext pair in pathname
+1 to skip '/' */
		path = slash + 1;
	} while(!done);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int stat(char *path, struct stat *statbuf)
{
	file_t file;
	drv_t *drv;
	int err;

	memset(&file, 0, sizeof(file_t));
	err = do_open(&file, path);
	if(err < 0)
		return err;
	drv = file.mount->drv;
	statbuf->st_dev = drv->dev_num;
	statbuf->st_mode = file.mode;
	statbuf->st_size = file.size;
	return 0;
}
/*****************************************************************************
*** WARNING ***
Only one directory stream can be open at a time
*****************************************************************************/
DIR *opendir(char *path)
{
	static DIR ret_val;
/**/
	unsigned handle;
	int err;

/* allocate a file_t */
	err = find_free_file(&ret_val.file, &handle);
	if(err != 0)
		return NULL;
/* open the directory */
	err = do_open(ret_val.file, path);
	if(err < 0)
		return NULL;
	return &ret_val;
}
/*****************************************************************************
*****************************************************************************/
void rewinddir(DIR *dir)
{
	dir->file->pos = 0;
}
/*****************************************************************************
*****************************************************************************/
struct dirent *readdir(DIR *dir)
{
	int err;

/* make sure it's a directory */
	if((dir->file->mode & S_IFDIR) == 0)
		return NULL;
/* make sure FS-specific do_readdir() works */
	if(dir->file->mount == NULL ||
		dir->file->mount->fs_info == NULL ||
			dir->file->mount->fs_info->readdir == NULL)
				return NULL;
	err = dir->file->mount->fs_info->readdir(dir->file, &dir->ent);
	if(err != 0)
		return NULL;
	return &dir->ent;
}
/*****************************************************************************
*****************************************************************************/
int closedir(DIR *dir)
{
	dir->file->is_open = 0;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int open(const char *path, unsigned access)
{
	unsigned handle;
	file_t *file;
	int err;

/* allocate a file_t */
	err = find_free_file(&file, &handle);
	if(err != 0)
		return err;
/* open the file */
	err = do_open(file, path);
	if(err < 0)
		return err;
	return handle;
}
/*****************************************************************************
*****************************************************************************/
long lseek(unsigned handle, long offset, int whence)
{
	file_t *file;

/* validate handle */
	if(handle >= MAX_FILE)
		return -EBADF;
	file = _files + handle;
	switch(whence)
	{
		case SEEK_SET:
			/* nothing */
			break;
		case SEEK_CUR:
			offset += file->pos;
			break;
		case SEEK_END:
			offset = (file->size - 1) - offset;
			break;
		default:
			return -EINVAL;
	}
	if(offset < 0)
		offset = 0;
	else if(offset >= file->size - 1)
		offset = file->size - 1;
	file->pos = offset;
	return offset;
}
/*****************************************************************************
*****************************************************************************/
off_t read(unsigned handle, void HUGE *buf, off_t count)
{
	file_t *file;

/* validate handle */
	if(handle >= MAX_FILE)
		return -EBADF;
	file = _files + handle;
/* make sure FS-specific read() works */
	if(file->mount == NULL || file->mount->fs_info == NULL ||
		file->mount->fs_info->read == NULL)
			return -ENOIMP;
	return file->mount->fs_info->read(file, buf, count);
}
/*****************************************************************************
*****************************************************************************/
int close(unsigned handle)
{
	file_t *file;

/* validate handle */
	if(handle >= MAX_FILE)
		return -EBADF;
	file = _files + handle;
/* close it */
	file->is_open = 0;
	return 0;
}
