/*****************************************************************************
xxx - support RDSK and ELF files
xxx - load RDSK files into extended memory if too little conventional memory
xxx - let user change "Physical load address" and "Store boot data in BSS"
xxx - fat_mount() should get hard drive geometry -- somehow
	INT 13h AH=08h should be OK for hard disk (not for floppy, though)
(later)
xxx - let user browse different disks/partitions
xxx - use XMS to access extended memory if HIMEM.SYS loaded
xxx - use VCPI to switch to pmode if EMM386 loaded
xxx - support DOS .EXE files, DOS .COM files, and chainloaded bootsectors
xxx - support binary kernel (let user change load adr, entry point, etc.)
xxx - support ext2 filesystem
*****************************************************************************/
/* portability! These assume sizeof(short)==2 and sizeof(long)==4
They also assume a little-endian CPU like x86.
Big-endian CPUs will need byte-swapping functions here. */
#define	read_le16(X)	*(unsigned short *)(X)
#define	read_le32(X)	*(unsigned long *)(X)

/* some functions return -1 when I don't know what else to return */
#define	ERR_EOF		-2	/* end of file */
#define	ERR_IO		-3	/* disk read error */
#define	ERR_MEM		-4	/* out of memory */
#define	ERR_SW		-5	/* software error ("can't happen") */
#define	ERR_FILE	-6	/* invalid file format */
#define	ERR_FILESYSTEM	-7

#define	SEEK_SET	0
#define	SEEK_CUR	1
#define	SEEK_END	2

/* "An inode is an [unique] integer associated with a file on the filesystem."
For FAT, we use the first cluster of the file as the inode */
typedef unsigned short	inode_t;

typedef unsigned long	sector_t;

typedef unsigned long	pos_t;

typedef struct
{
	unsigned char int13_dev;
	unsigned short sects, heads, bytes_per_sector;
} dev_t;

typedef struct
{
	pos_t pos;
	struct _mount *mount;
	inode_t inode;
	unsigned long file_size;
	unsigned is_dir : 1;
	char name[16]; /* xxx - 16 is arbitrary...maybe use malloc()? */
} file_t;

typedef struct
{
	int (*open_root)(file_t *root_dir, struct _mount *mount);
	int (*readdir)(file_t *dir, file_t *file);
	int (*read)(file_t *file, unsigned char *buf, unsigned len);
/*	void *info; */
	unsigned char info[64]; /* xxx - use malloc() */
} fsinfo_t;

typedef struct _mount
{
	dev_t *dev;
	fsinfo_t fsinfo;
	sector_t partition_start;
	inode_t curr_dir;
} mount_t;

/* non-standard (not ANSI nor UNIX) I/O functions, but they were
eaiser to write than open(), readdir(), etc. (and more efficient,
since they deal directly with inodes, rather than path names) */
static int my_seek(file_t *file, long offset, int whence);
static int my_read(file_t *file, void *buf, unsigned len);
static int my_close(file_t *file);

static void cprintf(const char *fmt, ...);

static mount_t _mount;
/*////////////////////////////////////////////////////////////////////////////
	READ-ONLY DISK CODE AND CACHE
////////////////////////////////////////////////////////////////////////////*/
#include <bios.h> /* biosdisk() */

#define	MAX_CACHE	32	/* 32 sectors == 16K */
#define	BPS		512	/* bytes per sector */

typedef struct
{
	sector_t sector;
	unsigned char blk[BPS];
} cache_t;
/*****************************************************************************
*****************************************************************************/
static int read_sector(dev_t *dev, sector_t sector, unsigned char **blk)
{
	static unsigned char init, evict;
	static cache_t cache[MAX_CACHE];
/* */
	unsigned short c, h, s, temp;
	unsigned char tries;

	if(!init)
	{
		init = 1;
		for(temp = 0; temp < MAX_CACHE; temp++)
			cache[temp].sector = -1uL;
	}
/* see if this sector is cached */
	for(temp = 0; temp < MAX_CACHE; temp++)
	{
		if(cache[temp].sector == sector)
		{
			(*blk) = cache[temp].blk;
			return 0;
		}
	}
/* not cached, find a free buffer for it */
	for(temp = 0; temp < MAX_CACHE; temp++)
	{
		if(cache[temp].sector == -1uL)
			break;
	}
/* no free buffer, kick out someone else */
	if(temp >= MAX_CACHE)
	{
		temp = evict;
		evict++;
		if(evict >= MAX_CACHE)
			evict = 0;
	}
/* load it */
	cache[temp].sector = sector;
	(*blk) = cache[temp].blk;
/* we can load sector 0 even if we don't know the disk geometry
(which is good, because FAT uses sector 0 to store floppy geometry info) */
	if(sector == 0)
	{
		s = 1;
		h = 0;
		c = 0;
	}
	else
	{
		s = sector % dev->sects + 1;
		h = (sector / dev->sects) % dev->heads;
		c = (sector / dev->sects) / dev->heads;
	}
/* make 3 attempts */
	for(tries = 3; tries != 0; tries--)
	{
		temp = biosdisk(/* _DISK_READ */2, dev->int13_dev,
			h, c, s, 1, *blk);
/* biosdisk() does not return what the Turbo C online help says.
It returns the AH value from INT 13h, not AX
		temp >>= 8; */
		if(temp == 0 || temp == 0x11)
			return 0;
/* reset FDC if error */
		(void)biosdisk(/* _DISK_RESET */0, dev->int13_dev,
			0, 0, 0, 0, 0);
	}
	cprintf("\n\rread_sector: INT 13h disk error 0x%02X, CHS=%u:%u:%u, "
		"dev=0x%02X, *blk=%p\n\r", temp, c, h, s,
		dev->int13_dev, *blk);
	return ERR_IO;
}
/*///////////////////////////////////////////////////////////////////////////
	FAT12/16 FILESYSTEM
////////////////////////////////////////////////////////////////////////////*/
/* NULL, memset, memcmp, memcpy, strchr, strlen, strcpy, strupr, strchr */
#include <string.h>

#define	min(a,b)	(((a) < (b)) ? (a) : (b))

/* bytes per FAT directory entry */
#define	FAT_DIRENT_LEN		32
/* FAT entries 0 and 1 are reserved: */
#define	MAGIC_FAT_CLUSTER	0
#define	MAGIC_ROOT_CLUSTER	1

typedef enum
{
	FAT12, FAT16, FAT32
} fat_type_t;

typedef struct
{
	unsigned short max_cluster;
	unsigned char sectors_per_cluster;
	sector_t fat_start, root_start, data_start;
	fat_type_t fat_type;
} fat_t;
/*****************************************************************************
e.g. "foo.i" -> "FOO     I  "
'dst' must be >=12 bytes
*****************************************************************************/
static int fat_convert_name_to_fat(char *dst, char *src)
{
	unsigned len;
	char *dot;

/* put src in FAT format */
	memset(dst, ' ', 11);
	dst[11] = '\0';
	dot = strchr(src, '.');

/* xxx - why? trouble with Turbo C 2.0 installation? */
#define NULL 0

/* there is an extension */
	if(dot != NULL)
	{
/* long filenames not supported */
		len = dot - src;
		if(len > 8)
			return ERR_FILESYSTEM;
/* copy filename */
		memcpy(dst, src, len);
		dst[len] = '\0';
/* long extension not supported */
		len = strlen(dot) - 1;
		if(len > 3)
			return ERR_FILESYSTEM;
/* copy extension */
		memcpy(dst + 8, dot + 1, len);
	}
/* no extension */
	else
	{
/* long filenames not supported */
		len = strlen(src);
		if(len > 8)
			return ERR_FILESYSTEM;
/* copy filename */
		strcpy(dst, src);
	}
/* make it upper case */
	strupr(dst);
	return 0;
}
/*****************************************************************************
e.g. "README  TXT" -> "readme.txt"
'dst' must be >=12 bytes
*****************************************************************************/
static void fat_convert_name_from_fat(char *dst, char *src)
{
	unsigned char i;
	char *d = dst;

	for(i = 0; i < 8; i++)
	{
		if(src[i] == ' ')
			break;
		*d = src[i];
		d++;
	}
	if(src[8] != ' ')
	{
		*d = '.';
		d++;
		for(i = 8; i < 11; i++)
		{
			if(src[i] == ' ')
				break;
			*d = src[i];
			d++;
		}
	}
	*d = '\0';
/* make it lower case */
	strlwr(dst);
}
/*****************************************************************************
*****************************************************************************/
static int fat_open_root(file_t *root_dir, mount_t *mount)
{
	root_dir->pos = 0;
	root_dir->mount = mount;
	root_dir->inode = MAGIC_ROOT_CLUSTER;
	root_dir->file_size = -1uL; /* xxx - get root dir size */
	root_dir->is_dir = 1;
	strcpy(root_dir->name, "/");
	return 0;
}
/*****************************************************************************
FAT file attribute bits:
Bit(s)	Description	(Table 01401)
 0	read-only
 1	hidden
 2	system
 3	volume label or LFN
 4	directory
 5	archive
 6	?
 7	if set, file is shareable under Novell NetWare
*****************************************************************************/
static int fat_readdir(file_t *dir, file_t *file)
{
	char dirent[FAT_DIRENT_LEN];
	int err;

	while(1)
	{
/* read one 32-byte FAT directory entry */
		err = my_read(dir, dirent, FAT_DIRENT_LEN);
		if(err != FAT_DIRENT_LEN)
		{
/* short read means end of dir */
			if(err >= 0)
				return ERR_EOF;
			return err;
		}
/* "virgin" dir entry means end of dir, for both root dir and subdir */
		if(dirent[0] == 0)
			return ERR_EOF;
/* deleted file */
		if(dirent[0] == 5 || dirent[0] == '\xE5')
			continue;
/* volume label or LFN */
		if((dirent[11] & 0x08) == 0)
			break;
	}
/* found it! */
	file->pos = 0;
	file->mount = dir->mount;
	file->inode = read_le16(dirent + 26);
	file->file_size = read_le32(dirent + 28);
	if(dirent[11] & 0x10)
		file->is_dir = 1;
	else
		file->is_dir = 0;
	fat_convert_name_from_fat(file->name, dirent);
/* inode==0 is actually a pointer to the root directory */
	if(file->inode == 0)
	{
		if(file->is_dir)
			file->inode = MAGIC_ROOT_CLUSTER;
/* ...but only for directories */
		else
			return -1;
	}
/* xxx - corrupt filesystem; not software error */
	else if(file->inode < 2)
		return ERR_SW;
/* I guess FAT doesn't store the correct size of subdirectories
in the directory entry */
	if(file->file_size == 0)
	{
		if(file->is_dir)
			file->file_size = -1uL;
/* else it's a zero-length file, which is cool */
	}
	return 0;
}
/*****************************************************************************
convert 'cluster' to 'sector', then advance to next cluster in FAT chain
and store next cluster at 'cluster'
*****************************************************************************/
static int fat_walk(file_t *file, sector_t *sector, unsigned short *cluster)
{
	unsigned char buf[2];
	unsigned short entry;
	file_t the_fat;
	sector_t temp;
	fat_t *fat;
	int err;

/* xxx - init the_fat properly */
	fat = (fat_t *)(file->mount->fsinfo.info);
	the_fat.mount = file->mount;
	the_fat.inode = MAGIC_FAT_CLUSTER;
	the_fat.file_size = -1uL;
/* must be cluster within data area of disk */
	temp = (*cluster);
	if(temp < 2)
		return ERR_SW; /* can't (shouldn't) happen */
	temp -= 2;
/* convert cluster to sector */
	temp *= fat->sectors_per_cluster;
	temp += fat->data_start;
	(*sector) = temp;
/* now convert cluster to byte offset into FAT */
	temp = (*cluster);
	if(fat->fat_type == FAT12)
	{
/* 12 bits per FAT entry, so byte offset into FAT is 3/2 * temp */
		temp *= 3;
		the_fat.pos = temp >> 1;
/* read 2-byte entry */
		err = my_read(&the_fat, buf, 2);
		if(err < 0)
			return err;
		entry = read_le16(buf);
/* top 12 bits or bottom 12 bits? */
		if(temp & 1)
			entry >>= 4;
		else
			entry &= 0x0FFF;
	}
	else if(fat->fat_type == FAT16)
	{
/* 16 bits per FAT entry */
		temp <<= 1;
		the_fat.pos = temp;
/* read 2-byte entry */
		err = my_read(&the_fat, buf, 2);
		if(err < 0)
			return err;
		entry = read_le16(buf);
	}
	else
	{
		cprintf("fat_walk: FAT32 not yet supported\n\r");
		return ERR_SW;
	}
/* that's what we want! */
	(*cluster) = entry;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int fat_read_sector(file_t *file, sector_t rel_sector,
		unsigned char **blk)
{
	unsigned short rel_cluster, abs_cluster;
	sector_t abs_sector;
	fsinfo_t *fsinfo;
	mount_t *mount;
	dev_t *dev;
	fat_t *fat;
	int err;

	mount = file->mount;
	dev = mount->dev;
	fsinfo = &mount->fsinfo;
	fat = (fat_t *)fsinfo->info;
	abs_cluster = file->inode;
/* starting cluster == 0: read from FAT */
	if(abs_cluster == MAGIC_FAT_CLUSTER)
	{
		abs_sector = fat->fat_start + rel_sector;
		if(abs_sector >= fat->root_start)
			return ERR_EOF;
	}
/* starting cluster == 1: read from root directory */
	else if(abs_cluster == MAGIC_ROOT_CLUSTER)
	{
		abs_sector = fat->root_start + rel_sector;
		if(abs_sector >= fat->data_start)
			return ERR_EOF;
	}
/* starting cluster >= 2: normal read from data area of disk */
	else
	{
/* cluster within file */
		rel_cluster = rel_sector / fat->sectors_per_cluster;
/* sector within cluster */
		rel_sector %= fat->sectors_per_cluster;
/* chase clusters, so we go from relative cluster (cluster-within-file)... */
		for(;;)
		{
			if(abs_cluster > fat->max_cluster)
				return ERR_EOF;
			err = fat_walk(file, &abs_sector, &abs_cluster);
			if(err != 0)
				return err;
			if(rel_cluster == 0)
				break;
			rel_cluster--;
		}
/* ...to absolute cluster (cluster-within-disk). fat_walk() also
converted cluster to sector, so add things together to get the
absolute sector number (finally!) */
		abs_sector += rel_sector;
	}
/* load it */
	return read_sector(dev, mount->partition_start + abs_sector, blk);
}
/*****************************************************************************
*****************************************************************************/
static int fat_read(file_t *file, unsigned char *buf, unsigned want)
{
	unsigned short byte_in_sector;
	sector_t rel_sector;
	unsigned got, count;
	unsigned char *blk;
	mount_t *mount;
	dev_t *dev;
	int err;

	if(want == 0)
		return 0;
	mount = file->mount;
	dev = mount->dev;
	count = 0;
	do
	{
		rel_sector = file->pos;
/* byte within sector */
		byte_in_sector = rel_sector % dev->bytes_per_sector;
/* sector within file */
		rel_sector /= dev->bytes_per_sector;
/* read the sector */
		err = fat_read_sector(file, rel_sector, &blk);
		if(err < 0)
			return err;
/* how many bytes can we read from this sector? */
		got = dev->bytes_per_sector - byte_in_sector;
/* nearing end of file? */
		if(file->pos + got > file->file_size)
			got = file->file_size - file->pos;
		if(got == 0)
			break;
/* how many will we actually read from it? */
		got = min(got, want);
/* read them */
		memcpy(buf, blk + byte_in_sector, got);
/* done with this sector
maybe I will need a function like this later?
		uncache_sector(file, sector, blk); */
/* advance pointers */
		file->pos += got;
		buf += got;
		want -= got;
		count += got;
/* done? */
	} while(want != 0);
	return count;
}
/*****************************************************************************
*****************************************************************************/
/* #include <stdlib.h> malloc() */

static int fat_mount(mount_t *mount, dev_t *dev, unsigned char part)
{
	unsigned char *blk, *ptab_rec; /* partition table record */
	unsigned long total_sectors;
	fat_type_t fat_type;
	fsinfo_t *fsinfo;
	fat_t *fat;
	int err;

	mount->curr_dir = MAGIC_ROOT_CLUSTER;
	dev->bytes_per_sector = BPS;
/* floppy */
	if(dev->int13_dev < 0x80)
	{
		fat_type = FAT12;
/* read sector 0 of floppy (boot sector) */
		mount->partition_start = 0;
		err = read_sector(dev, mount->partition_start, &blk);
		if(err != 0)
			return err;
/* make sure it's a FAT disk */
		if((blk[0] != 0xEB && blk[0] != 0xE9) || /* JMP (SHORT) */
			read_le16(blk + 0x0B) != 512 || /* bytes/sector */
			blk[0x0D] != 1 ||		/* sectors/cluster */
			blk[0x15] != 0xF0 ||		/* media ID */
			blk[0x1A] != 2)			/* heads */
		{
NOT:			cprintf("Partition %u on drive 0x%02X is "
				"not a FAT12 or FAT16 partition\n\r",
				part, dev->int13_dev);
			return -1;
		}
/* read disk geometry from BIOS Parameter Block (BPB) of FAT boot sector */
		dev->sects = read_le16(blk + 0x18);
		dev->heads = read_le16(blk + 0x1A);
	}
/* hard disk */
	else
	{
		if(part > 3)
		{
			cprintf("fat_mount: partition number must be 0-3\n\r");
			return -1;
		}
/* read sector 0 of hard disk (MBR; partition table) */
		err = read_sector(dev, 0, &blk);
		if(err != 0)
			return err;
/* xxx - finish initializing dev */
dev->sects = 63;
dev->heads = 255;
/* point to 16-byte partition table record */
		ptab_rec = blk + 446 + 16 * part;
		switch(ptab_rec[4])
		{
			case 1:
				fat_type = FAT12;
				break;
			case 4:
				fat_type = FAT16; /* up to 32 meg */
				break;
			case 6:
				fat_type = FAT16; /* DOS 3.31+, >32 meg */
				break;
			default:
				goto NOT;
		}
		mount->partition_start = read_le32(ptab_rec + 8);
/* read sector 0 of partition (boot sector) */
		err = read_sector(dev, mount->partition_start, &blk);
		if(err != 0)
			return err;
/* make sure it's a FAT disk */
		if((blk[0] != 0xEB && blk[0] != 0xE9) || /* JMP (SHORT) */
			read_le16(blk + 0x0B) != 512)	/* bytes/sector */
				goto NOT;
	}
/*	fat = malloc(sizeof(fat_t));
	if(fat == NULL)
		return ERR_MEM; */
	mount->dev = dev;
/* init mount->fsinfo */
	fsinfo = &mount->fsinfo;
fat = (fat_t *)fsinfo->info;

	fsinfo->open_root = fat_open_root;
	fsinfo->readdir = fat_readdir;
	fsinfo->read = fat_read;
/* init mount->fsinfo->info
	fsinfo->info = fat; */
memcpy(fsinfo->info, fat, sizeof(fat));
	fat = (fat_t *)fsinfo->info;
	fat->sectors_per_cluster = blk[13];
	fat->fat_start = read_le16(blk + 14);	/* reserved_sectors */
	fat->root_start = fat->fat_start +
		blk[16] *			/* num_fats */
		read_le16(blk + 22);		/* sectors_per_fat */
	fat->data_start = fat->root_start +
		(FAT_DIRENT_LEN *		/* bytes_per_dir_ent */
		read_le16(blk + 17)) /		/* num_root_dir_ents */
			BPS;			/* bytes_per_sector */
	total_sectors = read_le16(blk + 19);
	if(total_sectors == 0)
		total_sectors = read_le32(blk + 32);
	fat->max_cluster = total_sectors /
		fat->sectors_per_cluster -
		fat->data_start - 1;
	fat->fat_type = fat_type;
#if 0
cprintf("disk CHS=??:%u:%u\n\r", dev->heads, dev->sects);

cprintf("drive 0x%02X, partition %u: %s partition starting at sector %lu\n\r",
dev->int13_dev, part, fat->fat_type == FAT12 ? "FAT12" : "FAT16",
mount->partition_start);

cprintf("%u sector(s)/cluster, ", fat->sectors_per_cluster);
cprintf("%u FATs, ", blk[0x10]);
cprintf("root at sector %lu, ", fat->root_start);
cprintf("data at sector %lu\n\r", fat->data_start);

cprintf("FAT(s) at sector %lu, ", fat->fat_start);
cprintf("%u entries in root dir, ", read_le16(blk + 0x11));
cprintf("%lu total sectors ", total_sectors);
if(total_sectors >= 16384)
	cprintf("(%luM)\n\r\n\r", total_sectors / 2048);
else
	cprintf("(%luK)\n\r\n\r", total_sectors / 2);
#endif
	return 0;
}
/*////////////////////////////////////////////////////////////////////////////
	VFS
////////////////////////////////////////////////////////////////////////////*/
/*****************************************************************************
read directory entry from directory 'dir'
stores entry at 'file' (we use file_t instead of a DIR type)
*****************************************************************************/
static int my_readdir(file_t *dir, file_t *file)
{
	fsinfo_t *fsinfo;
	mount_t *mount;

	mount = dir->mount;
	fsinfo = &mount->fsinfo;
	return fsinfo->readdir(dir, file);
}
/*****************************************************************************
find file 'name' in already-opened directory 'dir';
stores result at 'file' if success
*****************************************************************************/
static int my_find(file_t *dir, file_t *file, char *name)
{
	int err;

/* seek to beginning of dir */
	err = my_seek(dir, 0, SEEK_SET);
	if(err != 0)
		return err;
/* read one 32-byte directory entry */
	do
	{
		err = my_readdir(dir, file);
		if(err != 0)
			return err;
	} while(strcmp(name, file->name));
	return 0;
}
/*****************************************************************************
convert path from relative to absolute, if necessary?
remove superfluous components of path (e.g. /./ )?
figure out mount point based on absolute path?
*****************************************************************************/
static int my_open(file_t *file, char *path)
{
	char done = 0, *slash;
	file_t search_dir;
	fsinfo_t *fsinfo;
	mount_t *mount;
	int err;

	file->mount = mount = &_mount;
	fsinfo = &mount->fsinfo;
	if(path[0] == '/')
	{
/* skip first '/' */
		path++;
/* open root dir */
		err = fsinfo->open_root(&search_dir, mount);
		if(err != 0)
			return err;
/* done already? */
		if(path[0] == '\0')
		{
			memcpy(file, &search_dir, sizeof(file_t));
			return 0;
		}
	}
/* xxx - relative pathnames are not yet handled */
	else
		return -1;
	done = 0;
	do
	{
/* pick out one name in path */
		slash = strchr(path, '/');
		if(slash == NULL)
		{
			slash = path + strlen(path);
			done = 1;
		}
		*slash = '\0';
/* find file/dir of this name in search_dir */
		err = my_find(&search_dir, file, path);
		if(err != 0)
		{
			my_close(&search_dir);
			return err;
		}
		if(done)
			break;
/* CD to subdirectory */
		memcpy(&search_dir, file, sizeof(file_t));
/* advance to next name+ext pair in pathname
+1 to skip '/' */
		path = slash + 1;
	} while(path[0] != '\0');
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int my_read(file_t *file, void *buf, unsigned len)
{
	fsinfo_t *fsinfo;
	mount_t *mount;

	mount = file->mount;
	fsinfo = &mount->fsinfo;
	return fsinfo->read(file, buf, len);
}
/*****************************************************************************
*****************************************************************************/
static int my_seek(file_t *file, long offset, int whence)
{
	switch(whence)
	{
		case SEEK_SET:
			/* nothing */
			break;
		case SEEK_CUR:
			offset += file->pos;
			break;
		case SEEK_END:
			offset = (file->file_size - 1) - offset;
			break;
		default:
			return -1;
	}
	if(offset < 0)
		offset = 0;
	else if(offset >= file->file_size - 1)
		offset = file->file_size - 1;
	file->pos = offset;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
#pragma argsused
static int my_close(file_t *file)
{
	return 0;
}
/*////////////////////////////////////////////////////////////////////////////
	CONSOLE ROUTINES
////////////////////////////////////////////////////////////////////////////*/
#include <stdarg.h> /* va_list, va_start(), va_end() */
#include <string.h> /* strlen() */
#include <dos.h> /* struct REGPACK, intr() */
/* note that Turbo C 2.0 intr() is buggy (doesn't load register BP properly),
so I've written my own version in LOADERA.ASM */

int vsprintf(char *, const char *, void *);

static unsigned char _attrib = 0x07; /* white on black */
/*****************************************************************************
*****************************************************************************/
static void gotoxy(unsigned char x, unsigned char y)
{
	struct REGPACK regs;

	regs.r_dx = y;
	regs.r_dx <<= 8;
	regs.r_dx |= x;
	regs.r_bx = 0;		/* BH=video page 0 */
	regs.r_ax = 0x0200;	/* AH=subfunction 2 (move csr) */
	intr(0x10, &regs);
}
/*****************************************************************************
*****************************************************************************/
static void textattr(int a)
{
	_attrib = a;
}
/*****************************************************************************
*****************************************************************************/
static void cprintf(const char *fmt, ...)
{
	struct REGPACK regs;
	char buf[200];
	va_list args;

	va_start(args, fmt);
	vsprintf(buf, fmt, args);
	va_end(args);

	regs.r_bx = 0;		/* BH=video page 0 */
	regs.r_ax = 0x0300; /* AH=subfunction 3: get cursor pos to DH,DL */
	intr(0x10, &regs);

	regs.r_es = _DS;
	regs.r_bp = (unsigned)buf;
	regs.r_cx = strlen(buf);
	regs.r_bx = _attrib;	/* BH=video page 0, BL=attribute */
	regs.r_ax = 0x1301;	/* AH=subfunction 13h, AL=write mode 1 */
	intr(0x10, &regs);
}
/*****************************************************************************
*****************************************************************************/
static int getch(void)
{
	static short prev = -1;
	struct REGPACK regs;
	int ret_val;

	if(prev != -1)
	{
		ret_val = prev;
		prev = -1;
	}
	else
	{
		regs.r_ax = 0; /* AH=subfunction 0 (get key) */
		intr(0x16, &regs);
		ret_val = regs.r_ax & 0xFF;
		if(ret_val == 0)
		{
			regs.r_ax >>= 8;
			prev = regs.r_ax;
		}
	}
	return ret_val;
}
/*****************************************************************************
*****************************************************************************/
static void erase_screen(unsigned char x, unsigned char y,
		unsigned char wd, unsigned char ht)
{
	struct REGPACK regs;

	regs.r_ax = 0x0600;	/* AH=subfunction 6 (scroll), AH=0 (erase) */
	regs.r_bx = _attrib;
	regs.r_bx <<= 8;	/* BH=attribute */
	regs.r_cx = y;
	regs.r_cx <<= 8;	/* CH=row */
	regs.r_cx |= x;		/* CL=col */
	regs.r_dx = (y + ht - 1);
	regs.r_dx <<= 8;	/* DH=height */
	regs.r_dx |= (x + wd - 1);/* DL=width */
	intr(0x10, &regs);
}
/*////////////////////////////////////////////////////////////////////////////
	KERNEL FILES
////////////////////////////////////////////////////////////////////////////*/
#define	MAX_SECT	4

typedef struct		/* section in executable file */
{
	char sect_name[12];
	unsigned long adr, size, off;
/* section flags */
	unsigned bss : 1;
	unsigned read : 1;
	unsigned write : 1;
	unsigned exec : 1;
} sect_t;

typedef struct		/* entire executable file */
{
	unsigned long entry_pt;
	unsigned short num_sections;
	sect_t section[MAX_SECT];
	char format_name[32];
	unsigned pmode : 1;
	unsigned cosmos : 1;
} exec_t;
/*****************************************************************************
returns -1 if file is not COFF,
returns ERR_FILE if invalid COFF file,
*****************************************************************************/
/* COFF file header */
#define	COFF_FILE_MAGIC		0	/* 0x014C */
#define	COFF_FILE_SECT_NUM	2	/* number of sections */
#define	COFF_FILE_TIMEDATE	4	/* time and date stamp */
#define	COFF_FILE_SYMTAB_OFF	8	/* file offset of symbol table */
#define	COFF_FILE_SYMTAB_NUM	12	/* number of symtab entries */
#define	COFF_FILE_OPTHDR_SIZE	16	/* "optional" (aout) header size */
#define	COFF_FILE_FLAGS		18
/* 20 bytes to here */
#define	COFF_FILE_HDRLEN	20

#define	COFF_AOUT_MAGIC		20	/* 0x010B */
#define	COFF_AOUT_VER		22
#define	COFF_AOUT_CODE_SIZE	24
#define	COFF_AOUT_DATA_SIZE	28
#define	COFF_AOUT_BSS_SIZE	32
#define	COFF_AOUT_ENTRY		36	/* initial EIP */
#define	COFF_AOUT_CODE_BASE	40
#define	COFF_AOUT_DATA_BASE	44
/* 48 bytes to here */
#define	AOUT_FILE_HDRLEN	48

/* COFF section header */
#define	COFF_SECT_NAME		0	/* ".text", ".data", etc. */
#define	COFF_SECT_PADR		8	/* physical adr */
#define	COFF_SECT_VADR		12	/* virtual adr */
#define	COFF_SECT_SIZE		16
#define	COFF_SECT_OFF		20	/* file offset of section */
#define	COFF_SECT_RELOC_OFF	24	/* file offset of relocations */
#define	COFF_SECT_LINENUM_OFF	28	/* file offset of line number info */
#define	COFF_SECT_RELOC_NUM	32	/* number of relocations */
#define	COFF_SECT_LINENUM_NUM	34	/* number of line numbers */
#define	COFF_SECT_FLAGS		36
/* 40 bytes long */
#define	COFF_SECT_HDRLEN	40

static int read_coff_file(exec_t *exec, file_t *file, unsigned long offset)
{
	unsigned char file_hdr[AOUT_FILE_HDRLEN], sect_hdr[COFF_SECT_HDRLEN];
	unsigned short aout_hdrlen, i;
	sect_t *sect;
	int temp;

	(void)my_seek(file, offset, SEEK_SET);
/* read file header and a.out header */
	temp = my_read(file, file_hdr, AOUT_FILE_HDRLEN);
	if(temp < 0)
		return temp;
/* short file is not a COFF file
return -1 instead of ERR_FILE */
	if(temp != AOUT_FILE_HDRLEN)
		return -1;
	if(read_le16(file_hdr + COFF_FILE_MAGIC) != 0x014C)
		return -1;
/* OK, it's COFF. Check for 3 sections (.text, .data, .bss) */
	exec->num_sections = read_le16(file_hdr + COFF_FILE_SECT_NUM);
	if(exec->num_sections != 3)
	{
ERR:		cprintf("invalid COFF file");
/* oops, I thought it was COFF... */
		return ERR_FILE;
	}
/* check flags for F_EXEC */
	if((read_le16(file_hdr + COFF_FILE_FLAGS) & 0x0002) == 0)
		goto ERR;
	if(read_le16(file_hdr + COFF_AOUT_MAGIC) != 0x010B)
		goto ERR;
	aout_hdrlen = read_le16(file_hdr + COFF_FILE_OPTHDR_SIZE);
	exec->entry_pt = read_le32(file_hdr + COFF_AOUT_ENTRY);
/* skip to first section */
	my_seek(file, offset + COFF_FILE_HDRLEN + aout_hdrlen, SEEK_SET);
	sect = exec->section;
	for(i = 0; i < exec->num_sections; i++)
	{
/* read section header */
		temp = my_read(file, sect_hdr, COFF_SECT_HDRLEN);
		if(temp != COFF_SECT_HDRLEN)
			goto ERR;
/* code; STYP_TEXT */
		if(!memcmp(sect_hdr + COFF_SECT_NAME, ".text", 5) &&
			(read_le32(sect_hdr + COFF_SECT_FLAGS) & 0xE0) == 0x20)
		{
			sect->exec = 1;
			sect->size = read_le32(sect_hdr + COFF_SECT_SIZE);
		}
/* data; STYP_DATA */
		else if(!memcmp(sect_hdr + COFF_SECT_NAME, ".data", 5) &&
			(read_le32(sect_hdr + COFF_SECT_FLAGS) & 0xE0) == 0x40)
		{
			sect->write = 1;
			sect->size = read_le32(sect_hdr + COFF_SECT_SIZE);
		}
/* BSS; STYP_BSS */
		else if(!memcmp(sect_hdr + COFF_SECT_NAME, ".bss", 5) &&
			(read_le32(sect_hdr + COFF_SECT_FLAGS) & 0xE0) == 0x80)
		{
			sect->bss = 1;
			sect->write = 1;
			sect->size = read_le32(file_hdr + COFF_AOUT_BSS_SIZE);
/*			sect->size = read_le32(sect_hdr + COFF_SECT_SIZE);*/
		}
/* anything else */
		else
		{
			cprintf("bad section '%-8.8s' in COFF file",
				(char *)sect_hdr + COFF_SECT_NAME);
			return ERR_FILE;
		}
		sect->read = 1;
		strcpy(sect->sect_name, (char *)sect_hdr + COFF_SECT_NAME);
		sect->off = read_le32(sect_hdr + COFF_SECT_OFF);
/*		sect->adr = read_le32(sect_hdr + COFF_SECT_PADR); */
		sect->adr = read_le32(sect_hdr + COFF_SECT_VADR);
		sect++;
	}
	strcpy(exec->format_name, "DJGPP COFF");
	exec->pmode = 1;
	return 0;
}
/*****************************************************************************
returns -1 if file is not PE,
returns ERR_FILE if invalid PE file,
*****************************************************************************/
static int read_pe_file(exec_t *exec, file_t *file, unsigned long offset)
{
	unsigned long new_exe_offset;
	unsigned char buf[64];
	int temp;

	(void)my_seek(file, offset, SEEK_SET);
/* read DOS "MZ" EXE file header */
	temp = my_read(file, buf, 64);
	if(temp < 0)
		return temp;
/* short file is not a PE file
return -1 instead of temp_FILE */
	if(temp != 64)
		return -1;
	if(buf[0] != 'M' || buf[1] != 'Z')
		return -1;
/* skip to new-style EXE header and read it */
	new_exe_offset = read_le32(buf + 60);
	temp = my_seek(file, offset + new_exe_offset, SEEK_SET);
	if(temp < 0)
		return temp;
	temp = my_read(file, buf, 4);
	if(temp < 0)
		return temp;
	if(temp != 4)
/* could be DOS .EXE
		return ERR_FILE; */
		return -1;
	if(buf[0] != 'P' || buf[1] != 'E')
		return -1;
/* beyond the 4-byte "PE" signature, it's just like COFF */
	temp = read_coff_file(exec, file, offset + new_exe_offset + 4);
	strcpy(exec->format_name, "Win32 PE");
	return temp;
}
/*////////////////////////////////////////////////////////////////////////////
	MAIN
////////////////////////////////////////////////////////////////////////////*/
#define	WIDTH		80
#define	HEIGHT		25

#define	BANNER_HT	4

#define	LISTBOX_X	0
#define	LISTBOX_Y	(1 + BANNER_HT)

#define	LISTBOX_HT	15
#define	LISTBOX_COLS	3
#define	LISTBOX_COLWD	13
#define	LISTBOX_WD	(LISTBOX_COLS * LISTBOX_COLWD)

#define	LISTBOX_FILES	(LISTBOX_HT * LISTBOX_COLS)

#define	INFO_X		(WIDTH / 2 + 1)
#define	INFO_Y		LISTBOX_Y
#define	INFO_WD		(WIDTH / 2 - 1)
#define	INFO_HT		(HEIGHT - INFO_Y)

#define	MSG_X		0
#define	MSG_Y		(LISTBOX_Y + LISTBOX_HT + 1)
#define	MSG_WD		(WIDTH - MSG_X)
#define	MSG_HT		(HEIGHT - MSG_Y)

#define	LOADBUF_SIZE	4096

/* IMPORTS
from LIBA.ASM */
int enable_pmode(int enable_a20, unsigned long phys_entry_pt);

char _got_32bit_cpu = 1;
static unsigned char _windows, _load_buffer[LOADBUF_SIZE];
unsigned long _ext_mem_size = 1048576L, _conv_mem_size = 655360L;
/*****************************************************************************
*****************************************************************************/
static int copy_extended_memory(unsigned long dst_adr,
		unsigned long src_adr, unsigned short count)
{
/* global descriptor table, from Table 00499 of Ralf Brown's list */
	unsigned char gdt[] =
	{
/* 0 */		0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS */
/* 8 */		0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS */
/* page-granular 32-bit data segments with limit 4 Gbytes - 1 */
/* 10h */	0xFF, 0xFF, 0, 0, 0, 0x93, 0xCF, 0,/* src seg */
/* 18h */	0xFF, 0xFF, 0, 0, 0, 0x93, 0xCF, 0,/* dst seg */
/* 20h */	0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS for CS */
/* 28h */	0,    0,    0, 0, 0, 0,    0,    0/* used by BIOS for SS */
	};
	struct REGPACK regs;

/* fill in src segment descriptor */
	gdt[0x12] = src_adr;
	src_adr >>= 8;
	gdt[0x13] = src_adr;
	src_adr >>= 8;
	gdt[0x14] = src_adr;
	src_adr >>= 8;
	gdt[0x17] = src_adr;
/* fill in dst segment descriptor */
	gdt[0x1A] = dst_adr;
	dst_adr >>= 8;
	gdt[0x1B] = dst_adr;
	dst_adr >>= 8;
	gdt[0x1C] = dst_adr;
	dst_adr >>= 8;
	gdt[0x1F] = dst_adr;
/* call INT 15h AH=87h to copy */
	regs.r_ax = 0x8700;
	count >>= 1;	/* words! */
	regs.r_cx = count;
	regs.r_es = _SS;/* ES:SI -> GDT */
	regs.r_si = (unsigned)gdt;
	intr(0x15, &regs);
/* return status byte from AH
Borland compiler bug strikes again! put shifts on their own line of code
http://www.execpc.com/~geezer/embed/bugs.htm
	return regs.r_ax >> 8; */
	regs.r_ax >>= 8;
	if(regs.r_ax != 0)
	{
		cprintf("error from 0x%X in copy_extended_memory\n",
			regs.r_ax);
		return -1;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static unsigned long get_linear_adr(void far *foo)
{
	unsigned long ret_val;

	ret_val = FP_SEG(foo);
	ret_val <<= 4;
	ret_val += FP_OFF(foo);
	return ret_val;
}
/*****************************************************************************
*****************************************************************************/
static void find_extremes(exec_t *exec, unsigned long *lowest,
		unsigned long *highest)
{
	unsigned short temp;
	unsigned long high;

	*lowest = -1uL;
	*highest = 0uL;
	for(temp = 0; temp < exec->num_sections; temp++)
	{
		if(exec->section[temp].adr < *lowest)
			*lowest = exec->section[temp].adr;
		high = exec->section[temp].adr + exec->section[temp].size;
		if(high > *highest)
			*highest = high;
	}
}
/*****************************************************************************
*****************************************************************************/
static int zero_bss(exec_t *exec, unsigned long start, unsigned long end)
{
	unsigned long adr;

	memset(_load_buffer, 0, LOADBUF_SIZE);
/* put Cosmos-compatible boot data in the BSS */
	if(exec->cosmos)
	{
		*(unsigned long *)(_load_buffer + 0) = _conv_mem_size;
		*(unsigned long *)(_load_buffer + 4) = _ext_mem_size;
/* location of loaded RDSK image
		*(unsigned long *)(_load_buffer + 8) =
			get_linear_adr(_conv_mem); */
	}
	for(adr = start; adr < end; adr += LOADBUF_SIZE)
	{
		if(copy_extended_memory(adr, get_linear_adr(_load_buffer),
			LOADBUF_SIZE) != 0)
				return -1;
		memset(_load_buffer, 0, LOADBUF_SIZE);
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int load_section(file_t *file, unsigned long start, unsigned long end)
{
	unsigned long adr;
	int temp;

/* load it: */
	for(adr = start; adr < end; adr += LOADBUF_SIZE)
	{
/* load a block */
		temp = my_read(file, _load_buffer, LOADBUF_SIZE);
		if(temp == 0)
			break;
		else if(temp < 0)
		{
			cprintf("error reading disk/file");
			return temp;
		}
/* copy it to where it belongs */
		if(copy_extended_memory(adr, get_linear_adr(_load_buffer),
			LOADBUF_SIZE) != 0)
				return -1;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int run_pmode_kernel(file_t *file, exec_t *exec, unsigned long phys)
{
	unsigned long lowest, highest, virt_to_phys, start, end;
	unsigned short temp;
	sect_t *sect;

/* find lowest and highest virtual addresses */
	find_extremes(exec, &lowest, &highest);
/* compute virt_to_phys */
	virt_to_phys = phys - lowest;
/* for each section: */
	sect = exec->section + 0;
	for(temp = 0; temp < exec->num_sections; temp++)
	{
/* figure out where to load it */
		start = virt_to_phys + sect->adr;
		end = start + sect->size;
/* BSS */
		if(sect->bss)
		{
			if(zero_bss(exec, start, end) != 0)
				return -1;
		}
/* preloaded section other than BSS
		else if(_preload_adr != 0)
		{
			if(copy_extended_memory(start, sect->off +
				_preload_adr, sect->size) != 0)
					return -1;
		} */
/* non-BSS section not yet loaded */
		else
		{
			(void)my_seek(file, sect->off, SEEK_SET);
			if(load_section(file, start, end) != 0)
				return -1;
		}
		sect++;
	}
/* turn of floppy drive motor(s) */
	outportb(0x3F2, 0);
/* enable pmode and jump to kernel */
	temp = enable_pmode(virt_to_phys + highest >= 0x100000L,
		virt_to_phys + exec->entry_pt);
	cprintf("error from enable_pmode: ");
	if(temp == 1)
		cprintf("32-bit CPU required");
	else if(temp == 2)
		cprintf("could not enable A20 gate");
	else
		cprintf("unknown error %d", temp);
	return -1;
}
/*****************************************************************************
*****************************************************************************/
static int load_kernel(file_t *file)
{
	unsigned long lowest, highest, phys = 0x110000L;
	unsigned char csr_y;
	int err, temp;
	sect_t *sect;
	exec_t exec;

/* display file name */
	textattr(0x78);		/* dark gray on white */
	csr_y = INFO_Y;
	gotoxy(INFO_X, csr_y++);
	cprintf("File name:    %-12.12s", file->name);
	gotoxy(INFO_X, csr_y++);

/* identify file */
	memset(&exec, 0, sizeof(exec));
	err = read_coff_file(&exec, file, 0);
	if(err == -1)
		err = read_pe_file(&exec, file, 0);
	if(err < 0)
	{
		textattr(0x4E);
		cprintf("unknown file format");
ERR:
		textattr(0x78);
		goto OUT;
	}
/* display info */
	cprintf("File format:  %s", exec.format_name);

	gotoxy(INFO_X, csr_y++);
	cprintf("Enable pmode: %s", exec.pmode ? "yes" : "no");

	gotoxy(INFO_X, csr_y++);
	if(exec.pmode && !_got_32bit_cpu)
	{
		textattr(0x4E); /* yellow on red */
		cprintf("32-bit CPU required");
		err = -1;
		goto ERR;
	}
	cprintf("Virtual entry point:    %8lX", exec.entry_pt);
	find_extremes(&exec, &lowest, &highest);

/* usually, the .text section is at the lowest address,
so it's the virtual load (start) address */
	gotoxy(INFO_X, csr_y++);
	cprintf("Virtual load address:   %8lX", lowest);

/* maybe the sections are not contiguous (i.e. there are gaps
between them), so the extent of the kernel is not neccessarily
the sum of the section sizes */
	gotoxy(INFO_X, csr_y++);
	cprintf("Kernel extent (size):   %8lX", highest - lowest);

	gotoxy(INFO_X, csr_y++);
	cprintf("Physical load address:  ");
	textattr(0x0F);	/* bright white on black */
	cprintf("%8.8lX", phys);
	textattr(0x78);

	if(phys >= 0x100000L)
	{
		if(phys + (highest - lowest) > (_ext_mem_size + 0x100000L))
		{
			textattr(0x4E);
			cprintf("not enough extended memory");
			err = ERR_MEM;
			goto ERR;
		}
	}
	else
	{
		if(phys + (highest - lowest) > _conv_mem_size)
		{
			textattr(0x4E);
			cprintf("not enough conventional memory");
			err = ERR_MEM;
			goto ERR;
		}
	}
	gotoxy(INFO_X, csr_y++);
	cprintf("Store boot data in BSS: ");
	textattr(0x0F);	/* bright white on black */
	cprintf("%-3s", exec.cosmos ? "yes" : "no ");
	textattr(0x78);

	csr_y++;
	if(exec.num_sections != 0)
	{
		gotoxy(INFO_X, csr_y++);
		cprintf("Section  Section  Section  Section");
		gotoxy(INFO_X, csr_y++);
		cprintf("name     address  offset   size");
		gotoxy(INFO_X, csr_y++);
		cprintf("-------- -------- -------- --------");
	}
	sect = exec.section + 0;
	for(temp = 0; temp < exec.num_sections; temp++)
	{
		gotoxy(INFO_X, csr_y + temp);
		cprintf("%-8.8s %8lX %8lX %8lX",
			sect->sect_name, sect->adr, sect->off, sect->size);
		sect++;
	}
	gotoxy(MSG_X, MSG_Y);
	if(_windows)
	{
		textattr(0x4E);
		cprintf("Can't boot pmode OS from Windows");
		textattr(0x78);
		err = -1;
	}
	else
	{
		cprintf("Press Enter to boot");
		err = 0;
	}
OUT:
	getch();
	if(err == 0)
		err = run_pmode_kernel(file, &exec, phys);
	erase_screen(INFO_X, INFO_Y, INFO_WD, INFO_HT);
	erase_screen(MSG_X, MSG_Y, MSG_WD, MSG_HT);
	return err;
}
/*****************************************************************************
*****************************************************************************/
static int list_files(file_t *dir)
{
	unsigned num_files_to_skip = 0, num_files_in_dir = -1u, f;
	int key, err, highlighted_file = 0;
	file_t highlighted_ent, ent;

	do
	{
/* seek to beginning of dir */
		err = my_seek(dir, 0, SEEK_SET);
		if(err != 0)
			return err;
/* maybe skip some files */
		for(f = 0; f < num_files_to_skip; f++)
		{
			err = my_readdir(dir, &ent);
			if(err != 0)
				return err;
		}
/* list some files */
		for(; f < num_files_to_skip + LISTBOX_FILES; f++)
		{
			err = my_readdir(dir, &ent);
			if(err == ERR_EOF)
				num_files_in_dir = f;
			else if(err != 0)
				return err;
			if(f >= num_files_in_dir)
				break;
			gotoxy(LISTBOX_X + ((f - num_files_to_skip) %
				LISTBOX_COLS) * LISTBOX_COLWD,
				LISTBOX_Y + ((f - num_files_to_skip) /
				LISTBOX_COLS));
			if(f == highlighted_file)
			{
				textattr(0x1F); /* bright white on blue */
				memcpy(&highlighted_ent, &ent, sizeof(ent));
			}
/* xxx - use function to access is_dir field? my_fstat() ? */
			else if(ent.is_dir)
				textattr(0x01);	/* blue on black */
			else
				textattr(0x07);	/* white on black */
			if(ent.is_dir)
				cprintf("/%-*.*s", LISTBOX_COLWD - 1,
					LISTBOX_COLWD - 1, ent.name);
			else
				cprintf("%-*.*s", LISTBOX_COLWD,
					LISTBOX_COLWD, ent.name);
			textattr(0x07);
		}
/* use keyboard for navigation */
		key = getch();
		if(key == 0)
			key = 0x100 | getch();
		if(key == 0x14D)
			highlighted_file++;
		else if(key == 0x14B)
			highlighted_file--;
		else if(key == 0x148)
			highlighted_file -= LISTBOX_COLS;
		else if(key == 0x150)
			highlighted_file += LISTBOX_COLS;
		else if(key == 13)
		{
			if(highlighted_ent.is_dir)
			{
				dir->inode = highlighted_ent.inode;
				dir->file_size = -1uL;
					/* highlighted_ent.file_size; */
				erase_screen(LISTBOX_X, LISTBOX_Y,
					LISTBOX_WD, LISTBOX_HT);
				num_files_to_skip = 0;
				num_files_in_dir = -1u;
				highlighted_file = 0;
			}
			else
			{
				(void)load_kernel(&highlighted_ent);
			}
		}
		if(highlighted_file < 0)
			highlighted_file = 0;
		else if(highlighted_file < num_files_to_skip)
			num_files_to_skip -= LISTBOX_COLS;
		else if(highlighted_file >= num_files_in_dir)
			highlighted_file = num_files_in_dir - 1;
		else if(highlighted_file >= num_files_to_skip + LISTBOX_FILES)
			num_files_to_skip += LISTBOX_COLS;
	} while(key != 27);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int main(void)
{
	struct REGPACK regs;
	file_t dir;
	dev_t dev;
	int temp;

/* make sure it's really DOS */
	regs.r_ax = 0x1600;
	intr(0x2F, &regs);
	regs.r_ax &= 0xFF;
	if(regs.r_ax != 0 && regs.r_ax != 0x80)
		_windows = 1;
/* */
	textattr(0x70);
	erase_screen(0, 0, WIDTH, HEIGHT);
	gotoxy(0, 0);
	cprintf("BING bootloader version 0.4 - "
		"http://www.execpc.com/~geezer/os");
	gotoxy(0, 1);
	cprintf("Conventional memory: %luK  Extended memory: %luK  "
		"32-bit CPU: %s", _conv_mem_size >> 10, _ext_mem_size >> 10,
		_got_32bit_cpu ? "yes" : "no");
	gotoxy(0, 2);
/* cprintf("%u bytes code/data, %u bytes BSS, %u bytes total",
  bdata - begin, edata - bdata, edata - begin); */
	gotoxy(0, 3);
	cprintf("Use the arrow keys to select a kernel file, "
		"then press Enter");
/* floppy: int13_dev = 0 (partition number is not used)
hard disk: int13_dev = 0x80; specify partition number 0-3 */
	dev.int13_dev = 0x80;
	temp = fat_mount(&_mount, &dev, 1/* =part */);
	if(temp != 0)
	{
		cprintf("fat_mount returned %d\n\r", temp);
		return 0;
	}
	temp = my_open(&dir, "/");
/*	temp = my_open(&dir, "/tc/p/boot/test");
	temp = my_open(&dir, "/tc/p/k4"); */
	if(temp != 0)
	{
		cprintf("my_open returned %d\n\r", temp);
		return 0;
	}
	temp = list_files(&dir);
	if(temp != 0)
	{
		cprintf("list_files returned %d\n\r", temp);
		return 0;
	}
	temp = my_close(&dir);
	if(temp != 0)
	{
		cprintf("my_closedir returned %d\n\r", temp);
		return 0;
	}
	return 0;
}
