/*****************************************************************************
BING BOOTLOADER - command line version
*****************************************************************************/
#include <string.h> /* memset() */
#include <errno.h> /* ERR_FILE_FORMAT */
#include <conio.h> /* outp() */
#include <dos.h> /* struct REGPACK (or union REGPACK), intr() */
#include <io.h> /* open(), lseek(), read(), close() */
#include "execfile.h"
#include "defs.h"

#define	min(X,Y)	(((X) < (Y)) ? (X) : (Y))

/* IMPORTS
from CONIO.C */
int printf(const char *fmt, ...);

/* from LIB.ASM */
extern char _dos, _cpu32, _xms, _v86, _vcpi;
extern unsigned char far *_conv_mem_adr;
extern unsigned long _ext_mem_adr, _ext_mem_size, _conv_mem_size;

void asm_init(void);
void asm_exit(void);
void enter_pmode(unsigned long eip);

/* from EXE.C, ELF.C, COFF.C, PE.C */
int check_exe_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_file_offset);
int check_elf_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_file_offset);
int check_coff_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, unsigned long image_base);
int check_pe_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_file_offset);

/* Check Header Function
typedef int (*chf_t)(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, ...); */
typedef int (*chf_t)();

/* the non-command-line version of BING uses __end defined in STARTUP.ASM */
unsigned _end = 0xFFF0;
/*****************************************************************************
*****************************************************************************/
static void find_extremes(exec_t *exec, unsigned long *lowest,
		unsigned long *highest)
{
	unsigned short temp;
	unsigned long high;

	*lowest = -1uL;
	*highest = 0uL;
	for(temp = 0; temp < exec->num_sections; temp++)
	{
		if(exec->section[temp].adr < *lowest)
			*lowest = exec->section[temp].adr;
		high = exec->section[temp].adr + exec->section[temp].size;
		if(high > *highest)
			*highest = high;
	}
}
/*****************************************************************************
xxx - use XMS copy if available
xxx - support dst < 1 meg
*****************************************************************************/
static int copy_mem(unsigned long dst, unsigned char *src,
		unsigned count)
{
/* global descriptor table, from Table 00499 of Ralf Brown's list */
	unsigned char gdt[] =
	{
/* 0 */		0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS */
/* 8 */		0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS */
/* page-granular 32-bit data segments with limit 4 Gbytes - 1 */
/* 10h */	0xFF, 0xFF, 0, 0, 0, 0x93, 0xCF, 0,/* src seg */
/* 18h */	0xFF, 0xFF, 0, 0, 0, 0x93, 0xCF, 0,/* dst seg */
/* 20h */	0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS for CS */
/* 28h */	0,    0,    0, 0, 0, 0,    0,    0/* used by BIOS for SS */
	};
	unsigned long src_adr;
	regs_t regs;

/* fill in src segment descriptor */
	src_adr = _DS;
	src_adr <<= 4;
	src_adr += (unsigned)src;
	gdt[0x12] = src_adr;
	src_adr >>= 8;
	gdt[0x13] = src_adr;
	src_adr >>= 8;
	gdt[0x14] = src_adr;
	src_adr >>= 8;
	gdt[0x17] = src_adr;
/* fill in dst segment descriptor */
	gdt[0x1A] = dst;
	dst >>= 8;
	gdt[0x1B] = dst;
	dst >>= 8;
	gdt[0x1C] = dst;
	dst >>= 8;
	gdt[0x1F] = dst;
/* call INT 15h AH=87h to copy */
	regs.R_AX = 0x8700;
	count >>= 1;	/* words! */
	regs.R_CX = count;
	regs.R_ES = _SS;/* ES:SI -> GDT */
	regs.R_SI = (unsigned)gdt;
	intr(0x15, &regs);
/* return status byte from AH
Borland compiler bug strikes again! put shifts on their own line of code
http://www.execpc.com/~geezer/embed/bugs.htm
	return regs.R_AX >> 8; */
	regs.R_AX >>= 8;
	if(regs.R_AX != 0)
	{
		printf("error 0x%X in copy_extended_memory\n",
			regs.R_AX);
		return -1;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
#define	BUF_SIZE	4096

static int load_kernel(exec_t *exec, unsigned handle,
		unsigned long virt_to_phys)
{
	static unsigned char buf[BUF_SIZE];
/**/
	unsigned long adr, size;
	off_t err, count = 0;
	unsigned temp;
	sect_t *sect;

/* for each section: */
	sect = exec->section + 0;
	for(temp = 0; temp < exec->num_sections; temp++)
	{
/* BSS */
		if(sect->zero)
		{
unsigned first_page = 1;
			memset(buf, 0, BUF_SIZE);
/* xxx
poke Cosmos-compatible data into BSS */
*(unsigned long *)(buf + 0) = _conv_mem_size;
*(unsigned long *)(buf + 4) = _ext_mem_size;
/* RDSK adr and size */
*(unsigned long *)(buf + 8) = 0;
*(unsigned long *)(buf + 12) = 0;
			adr = sect->adr;
			for(size = sect->size; size != 0; size -= count)
			{
				count = min(size, BUF_SIZE);
				err = copy_mem(adr + virt_to_phys,
					buf, count);
				if(err != 0)
					return (int)err;
if(first_page)
{
	memset(buf, 0, BUF_SIZE);
	first_page = 0;
}
				adr += count;
			}
		}
/* non-BSS section */
		else
		{
			lseek(handle, sect->off, SEEK_SET);
			adr = sect->adr;
			for(size = sect->size; size != 0; size -= count)
			{
				count = min(size, BUF_SIZE);
				err = read(handle, buf, count);
				if(err < 0)
					return (int)err;
				else if(err != count)
					return -1;
				err = copy_mem(adr + virt_to_phys,
					buf, count);
				if(err != 0)
					return (int)err;
				adr += count;
			}
		}
		sect++;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int check_kernel(const char *path)
{
	const chf_t check_header[] =
	{
		check_exe_header, check_elf_header,
		check_coff_header, check_pe_header
	};
/**/
	unsigned long lowest, highest, phys, size, temp, virt_to_phys;
/*	void far (*real_phys)(void); */
	int handle, err, i;
	sect_t *sect;
	exec_t exec;

/* open kernel file */
	handle = open(path, 0);
	if(handle < 0)
	{
		printf("open() returned %d\n", handle);
		return -ENOENT;
	}
/* check if RDSK format
	memset(&exec, 0, sizeof(exec_t));
	err = check_rdsk_header(&exec, handle);
	if(err == -ERR_FILE_FORMAT) */
	{
/* check non-RDSK file formats */
		memset(&exec, 0, sizeof(exec_t));
		i = 0;
		for(; i < sizeof(check_header) / sizeof(check_header[0]); i++)
		{
/* CROCK ALERT: you need the 'L' after the second 0 */
			err = (check_header[i])(&exec, handle, 0, 0L);
			if(err != -ERR_FILE_FORMAT)
				break;
		}
	}
	if(err == -ERR_FILE_FORMAT)
	{
		printf("Unknown file format");
		close(handle);
		return -ERR_FILE_FORMAT;
	}
	else if(err != 0)
	{
		printf("check_header() returned %d", err);
		close(handle);
		return -ERR_FILE_FORMAT;
	}
	printf("\nFile format:  %s\n", exec.format_name);
	printf("Enable pmode: %s\n", exec.pmode ? "yes" : "no");

	find_extremes(&exec, &lowest, &highest);
	printf("Virtual load address:   %8lX\n", lowest);
	printf("Virtual entry point:    %8lX\n", exec.entry_pt);
/* convert highest to extent (size) */
	highest -= lowest;
	printf("Kernel extent (size):   %8lX\n", highest);
/* figure out load address */
	if(exec.pmode)
	{
/* try to load pmode kernel in extended memory */
		if(highest <= _ext_mem_size)
		{
			size = _ext_mem_size;
			phys = _ext_mem_adr;
			temp = phys + highest - 1;
			temp ^= phys;
			temp &= 0xFFC00000L;
			if(temp != 0)
				printf("\x1B[41;33;1m"
				"Warning: kernel straddles 4 meg line\n"
				"\x1B[37;40;0m");
		}
		else
		{
/* convert conventional memory base adr from far ptr to linear address */
			phys = FP_SEG(_conv_mem_adr);
			phys <<= 4;
			phys += FP_OFF(_conv_mem_adr);
/* align to 4K boundary */
			temp = phys + 4095;
			temp &= -4096uL;
/* reduce size due to 4K alignment */
			size = _conv_mem_size - (temp - phys);
			phys = temp;
		}
	}
/* load real-mode kernel in conventional memory */
	else
	{
		phys = FP_SEG(_conv_mem_adr);
		phys <<= 4;
		phys += FP_OFF(_conv_mem_adr);
		size = _conv_mem_size;
	}
	printf("Physical load address:  %8.8lX\n", phys);
/* display section info */
	if(exec.num_sections != 0)
	{
		printf("Section/segment info\n");
		printf("name    address  offset   size     RWX\n");
		printf("------- -------- -------- -------- ---\n");
	}
	sect = exec.section + 0;
	for(i = 0; i < exec.num_sections; i++)
	{
		printf("%-7.7s %8lX %8lX %8lX ",
			sect->name, sect->adr, sect->off, sect->size);
		printf("%c%c%c", sect->read ? 'R' : '-',
			sect->write ? 'W' : '-', sect->exec ? 'X' : '-');
		printf("\n");
		sect++;
	}
	printf("\n");
/* can we boot? */
	if(highest > size)
	{
		printf("Not enough memory");
		close(handle);
		return -ENOMEM;
	}
/* check if we can boot pmode kernel */
	if(exec.pmode)
	{
		if(!_cpu32)
		{
			printf("32-bit CPU required");
			close(handle);
			return -1;
		}
		if(_v86 && !_vcpi)
		{
			printf("Can't boot pmode OS from Windows");
			close(handle);
			return -1;
		}
	}
	else
	{
		printf("Real-mode kernels not yet supported");
		close(handle);
		return -ENOIMP;
	}
	if(exec.pmode)
	{
/* convert lowest virtual address to virt_to_phys */
		virt_to_phys = phys - lowest;
/* load kernel */
		err = load_kernel(&exec, handle, virt_to_phys);
		if(err != 0)
		{
			printf("load_kernel() returned %d\n", err);
			close(handle);
			return err;
		}
/* turn off floppy motor */
		outp(0x3F2, 0);
/* jump to kernel */
		enter_pmode(exec.entry_pt + virt_to_phys);
		printf("enter_pmode returned (A20 trouble?)\n");
	}
/* xxx - load real mode kernel (do .EXE relocations too)
	else
	{

		real_phys();
	} */
	close(handle);
	return -1;
}
/*****************************************************************************
*****************************************************************************/
int main(int arg_c, char *arg_v[])
{
	int err;

	asm_init();
	printf("BING bootloader version 0.6 - "
		"http://www.execpc.com/~geezer/os\n");
	if(arg_c != 2)
	{
		printf("Specify a pmode kernel file on the command line\n");
		return 1;
	}
	printf("%luK conventional memory at %Fp, ",
		_conv_mem_size >> 10, _conv_mem_adr);
	printf("%luK %s memory at %lX\n", _ext_mem_size >> 10,
		_xms ? "locked XMS" : "extended", _ext_mem_adr);
	printf("32-bit CPU:%s", _cpu32 ? "yes" : "no");
	printf("  DOS:%s", _dos ? "yes" : "no");
	printf("   V86 mode:%s", _v86 ? "yes" : "no");
	printf("   VCPI:%s\n", _vcpi ? "yes" : "no");
	err = check_kernel(arg_v[1]);
	asm_exit();
	return err;
}
