/*****************************************************************************
FAT12/16 FILESYSTEM

EXPORTS:
const fs_info_t _fat_ops;

int check_if_fat_bootsector(unsigned char *buf);
int fat_mount(mount_t *mount, drv_t *drv);
*****************************************************************************/
#include <sys/stat.h> /* S_IFDIR */
/* memset(), memchr(), memcpy(), strupr(), strlwr(), strncmp() */
#include <string.h>
#include <dirent.h> /* struct dirent */
#include <errno.h>
#include "defs.h"

/* IMPORTS
from DISKIO.C */
int read_disk(drv_t *drv, unsigned long sector, unsigned offset,
		unsigned count, unsigned char HUGE *buf);
int read_sector(drv_t *drv, unsigned long lba, unsigned char **buf_p);

/* from CONIO.C */
int printf(const char *fmt, ...);

#define	MAGIC_ROOT_CLUSTER	0
#define	FAT_DIRENT_SIZE		32

typedef enum
{
	FAT12, FAT16, FAT32
} fat_type_t;

/* information stored in mount_t for each mounted FAT volume */
typedef struct
{
	fat_type_t fat_type;
	unsigned char sectors_per_cluster;
	unsigned short bytes_per_cluster, root_dir_size, first_fat_sector;
	unsigned short first_root_sector, first_data_sector, max_cluster;
} fat_mount_t;

/* information stored in file_t for each opened file on an FAT volume */
typedef struct
{
	unsigned short first_file_cluster;
} fat_file_t;

/* alas, these do not work with GNU C...
#if sizeof(fat_mount_t)>MOUNT_INFO_SIZE
#error MOUNT_INFO_SIZE is too small
#endif

#if sizeof(fat_file_t)>FILE_INFO_SIZE
#error FILE_INFO_SIZE is too small
#endif */
/*****************************************************************************
*****************************************************************************/
static int fat_walk(mount_t *mount, unsigned long *sector,
		unsigned short *cluster)
{
	fat_mount_t *fat_mount;
	unsigned char buf[2];
	unsigned long temp;
	drv_t *drv;
	int err;

	fat_mount = (fat_mount_t *)mount->fs_mount_info;
	drv = mount->drv;
	temp = *cluster;
	temp -= 2;
	temp *= fat_mount->sectors_per_cluster;
	temp += fat_mount->first_data_sector;
	(*sector) = temp;

	if(fat_mount->fat_type == FAT12)
	{
		temp = (unsigned long)(*cluster) * 3;
		err = read_disk(drv, fat_mount->first_fat_sector,
			temp / 2, 2, buf);
		if(err != 0)
			return err;
		(*cluster) = read_le16(buf);
/* top 12 bits or bottom 12 bits? */
		if(temp & 1)
			(*cluster) >>= 4;
		else
			(*cluster) &= 0x0FFF;
	}
	else if(fat_mount->fat_type == FAT16)
	{
		temp = (unsigned long)(*cluster) << 1;
		err = read_disk(drv,
			fat_mount->first_fat_sector, temp, 2, buf);
		if(err != 0)
			return err;
		(*cluster) = read_le16(buf);
	}
	else
		return -ENOIMP; /* FAT32 */
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int fat_l2p(file_t *file, unsigned short *cluster)
{
	unsigned short logical_cluster, physical_cluster;
	fat_mount_t *fat_mount;
	fat_file_t *fat_file;
	unsigned long dummy;
	int err;

	fat_file = (fat_file_t *)file->fs_file_info;
	fat_mount = (fat_mount_t *)file->mount->fs_mount_info;
	logical_cluster = *cluster;
	physical_cluster = fat_file->first_file_cluster;
	for(; logical_cluster != 0; logical_cluster--)
	{
		if(physical_cluster > fat_mount->max_cluster ||
			physical_cluster < 2)
				return -ERR_FILE_SYSTEM;
		err = fat_walk(file->mount, &dummy, &physical_cluster);
		if(err != 0)
			return err;
	}
	(*cluster) = physical_cluster;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static off_t fat_read(file_t *file, unsigned char HUGE *buf, off_t count)
{
	unsigned short cluster, offset;
	fat_mount_t *fat_mount;
	fat_file_t *fat_file;
	unsigned long sector;
	mount_t *mount;
	off_t total, w;
	drv_t *drv;
	int err;

	mount = file->mount;
	drv = mount->drv;
	fat_mount = (fat_mount_t *)mount->fs_mount_info;
	fat_file = (fat_file_t *)file->fs_file_info;
/* limit count per file size and pos */
	if(file->pos >= file->size)
		return 0;
	if(file->pos + count > file->size)
		count = file->size - file->pos;
/* reading from root dir? */
	if(fat_file->first_file_cluster == MAGIC_ROOT_CLUSTER)
	{
/* limit count per root dir size and pos */
		if(file->pos >= fat_mount->root_dir_size)
			return 0;
		if(file->pos + count > fat_mount->root_dir_size)
			count = fat_mount->root_dir_size = file->pos;
/* read it */
		err = read_disk(drv, fat_mount->first_root_sector,
			file->pos, count, buf);
		if(err != 0)
			return err;
		file->pos += count;
		return count;
	}
/* compute logical cluster number and byte-in-cluster */
	cluster = file->pos / fat_mount->bytes_per_cluster;
	offset  = file->pos % fat_mount->bytes_per_cluster;
/* convert logical cluster to physical cluster */
	err = fat_l2p(file, &cluster);
	if(err != 0)
		return err;
	total = 0;
	while(count != 0)
	{
		if(cluster > fat_mount->max_cluster || cluster < 2)
			return -ERR_FILE_SYSTEM;
/* convert physical cluster to physical sector,
and get next physical cluster */
		err = fat_walk(mount, &sector, &cluster);
		if(err != 0)
			return err;
		w = fat_mount->bytes_per_cluster - offset;
		if(w > count)
			w = count;
		err = read_disk(drv, sector, offset, w, buf);
		if(err != 0)
			return err;
		buf += w;
		count -= w;
		total += w;

		file->pos += w;
		offset = 0;
	}
	return total;
}
/*****************************************************************************
extend name to 8 characters, extend extension to 3 characters
(both padded on the right with spaces), and capitalize them

e.g. "foo.i" -> "FOO     I  "

'dst' must be >=12 bytes
*****************************************************************************/
static int fat_convert_name_to_fat(char *dst, const char *src,
		const unsigned name_len)
{
	const char *dot;
	unsigned len;

/* put dst in FAT format */
	memset(dst, ' ', 11);
	dot = memchr(src, '.', name_len);
/* there is an extension */
	if(dot != NULL)
	{
/* long filenames not supported */
		len = dot - src;
		if(len > 8)
			return -ERR_FILE_NAME;
/* copy filename */
		memcpy(dst, src, len);
/* long extension not supported */
		len = name_len - len - 1;
		if(len > 3)
			return -ERR_FILE_NAME;
/* copy extension */
		memcpy(dst + 8, dot + 1, len);
	}
/* no extension */
	else
	{
/* long filenames not supported */
		len = name_len;
		if(len > 8)
			return -ERR_FILE_NAME;
/* copy filename */
		memcpy(dst, src, len);
	}
/* make it upper case */
	dst[11] = '\0';
	strupr(dst);
	return 0;
}
/*****************************************************************************
e.g. "README  TXT" -> "readme.txt"

'dst' must be >=12 bytes
*****************************************************************************/
static void fat_convert_name_from_fat(char *dst, unsigned char *src)
{
	unsigned char i;
	char *d = dst;

	for(i = 0; i < 8; i++)
	{
		if(src[i] == ' ')
			break;
		*d = src[i];
		d++;
	}
	if(src[8] != ' ')
	{
		*d = '.';
		d++;
		for(i = 8; i < 11; i++)
		{
			if(src[i] == ' ')
				break;
			*d = src[i];
			d++;
		}
	}
	*d = '\0';
/* make it lower case */
	strlwr(dst);
}
/*****************************************************************************
*****************************************************************************/
static int fat_readdir(file_t *file, struct dirent *ent)
{
	unsigned char buf[FAT_DIRENT_SIZE];
	off_t temp;

	while(1)
	{
		temp = fat_read(file, buf, FAT_DIRENT_SIZE);
		if(temp < 0)
			return (int)temp;
		if(temp != FAT_DIRENT_SIZE)
			return -ENOENT;
/* end of dir */
		if(buf[0] == '\0')
			return -ENOENT;
/* deleted entry */
		if(buf[0] == 0xE5)
			continue;
/* volume label or LFN */
		if((buf[11] & 0x08) == 0)
			break;
	}
/* convert name */
	fat_convert_name_from_fat(ent->d_name, buf);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int fat_find(file_t *file, const char *name, const unsigned name_len)
{
	unsigned char buf[FAT_DIRENT_SIZE];
	fat_mount_t *fat_mount;
	fat_file_t *fat_file;
	char fat_name[11];
	off_t temp;
	int err;

	fat_mount = (fat_mount_t *)file->mount->fs_mount_info;
/* opening root dir? */
	if(name[0] == '/' && name_len == 1)
	{
/* get FS-independent info */
		file->pos = 0;
		file->size = fat_mount->root_dir_size;
		file->mode = S_IFDIR | 0700; /* rwx------ */
		fat_file = (fat_file_t *)file->fs_file_info;
		fat_file->first_file_cluster = MAGIC_ROOT_CLUSTER;
		return 0;
	}
	err = fat_convert_name_to_fat(fat_name, name, name_len);
	if(err != 0)
		return err;
	do
	{
		while(1)
		{
			temp = fat_read(file, buf, FAT_DIRENT_SIZE);
			if(temp < 0)
				return (int)temp;
			if(temp != FAT_DIRENT_SIZE)
				return -ENOENT;
/* end of dir */
			if(buf[0] == '\0')
				return -ENOENT;
/* deleted entry */
			if(buf[0] == 0xE5)
				continue;
/* volume label or LFN */
			if((buf[11] & 0x08) == 0)
				break;
		}
	} while(strncmp(fat_name, (char *)buf, 11));
/* convert name */
	fat_convert_name_from_fat(file->name, buf);
/* get FS-independent info */
	file->pos = 0;
	if(buf[11] & 0x10)
	{
		file->size = 0x7FFFFFFFL;
		file->mode = S_IFDIR | 0700;
	}
	else
	{
		file->size = read_le32(buf + 28);
		file->mode = 0700;
	}
/* get FS-dependent info */
	fat_file = (fat_file_t *)file->fs_file_info;
	fat_file->first_file_cluster = read_le16(buf + 26);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
const fs_info_t _fat_ops =
{
	"fat", fat_read, fat_readdir, fat_find
};
/*****************************************************************************
*****************************************************************************/
static int lg2(unsigned arg)
{
	unsigned char log;

	for(log = 0; log < 16; log++)
	{
		if(arg & 1)
		{
			arg >>= 1;
			return (arg != 0) ? -1 : log;
		}
		arg >>= 1;
	}
	return -1;
}
/*****************************************************************************
*****************************************************************************/
int check_if_fat_bootsector(unsigned char *buf)
{
	unsigned short temp;
	int bad = 0;

	DEBUG(printf("check_if_fat_bootsector:\n");)
/* must start with 16-bit JMP or 8-bit JMP plus NOP */
	if(buf[0] == 0xE9)
		/* OK */;
	else if(buf[0] == 0xEB && buf[2] == 0x90)
		/* OK */;
	else
	{
		DEBUG(printf("\tMissing JMP/NOP\n");)
		bad = 1;
	}
	temp = buf[13];
	if(lg2(temp) < 0)
	{
		DEBUG(printf("\tSectors per cluster (%u) "
			"is not a power of 2\n", temp);)
		bad = 1;
	}
	temp = buf[16];
	temp = read_le16(buf + 24);
	if(temp == 0 || temp > 63)
	{
		DEBUG(printf("\tInvalid number of sectors (%u)\n", temp);)
		bad = 1;
	}
	temp = read_le16(buf + 26);
	if(temp == 0 || temp > 255)
	{
		DEBUG(printf("\tInvalid number of heads (%u)\n", temp);)
		bad = 1;
	}
	return bad;
}
/*****************************************************************************
*****************************************************************************/
extern mount_t _mounts[];

int fat_mount(mount_t *mount, drv_t *drv)
{
	unsigned long total_sectors;
	fat_mount_t *fat_mount;
	unsigned short temp;
	unsigned char *buf;
	int err;

	if(drv->part_type != 0 &&	/* unknown */
		drv->part_type != 1 &&	/* FAT12 */
		drv->part_type != 4 &&	/* FAT16, up to 32 meg */
		drv->part_type != 6)	/* FAT16, 32 meg+ */
			return -1;
/* drv->part_type != 0x0B &&	FAT32 CHS
drv->part_type != 0x0C &&	FAT32 LBA
drv->part_type != 0x0E &&	FAT16 LBA */
/* read sector 0 (boot sector) */
	err = read_sector(drv, 0, &buf);
	if(err != 0)
		return err;
/* check if FAT */
	if(check_if_fat_bootsector(buf) != 0)
		return -1;
/* init mount */
	mount->drv = drv;
	mount->fs_info = &_fat_ops;
/* init FS-specific mount data */
	fat_mount = (fat_mount_t *)mount->fs_mount_info;

	fat_mount->sectors_per_cluster = buf[13];
	fat_mount->bytes_per_cluster = fat_mount->sectors_per_cluster *
		drv->bps;
	fat_mount->first_fat_sector = read_le16(buf + 14);
/* number of FATs * sectors per FAT */
	temp = buf[16] * read_le16(buf + 22);
	fat_mount->first_root_sector = fat_mount->first_fat_sector + temp;
/* number of root dir entries * bytes per entry */
	fat_mount->root_dir_size = read_le16(buf + 17) * FAT_DIRENT_SIZE;
/* convert to sectors */
	temp = fat_mount->root_dir_size / drv->bps;
	fat_mount->first_data_sector = fat_mount->first_root_sector + temp;
/* max cluster */
	total_sectors = read_le16(buf + 19);
	if(total_sectors == 0)
		total_sectors = read_le32(buf + 32);
	total_sectors -= fat_mount->first_data_sector;
	fat_mount->max_cluster = total_sectors /
		fat_mount->sectors_per_cluster - 1;
/* FAT12 or FAT16? */
	if(fat_mount->max_cluster < 4084)
		fat_mount->fat_type = FAT12;
	else if(fat_mount->max_cluster < 65524u)
		fat_mount->fat_type = FAT16;
	else
		return -1; /* FAT32 */
	return 0;
}
