/*****************************************************************************
EXT2 FILESYSTEM

EXPORTS:
const fs_info_t _ext2_ops;

int ext2_mount(mount_t *mount, drv_t *drv);

maximum blocks for ext2fs with 1K blocks:
	(1K / 4 = 256)
	12 + 256 + 256*256 + 256*256*256 = 16 843 020 = ~16 Gbytes
maximum blocks for ext2fs with 4K blocks:
	(4K / 4 = 1024)
	12 + 1024 + 1024*1024 + 1024*1024*1024 = 1 074 791 436 = ~256 Gbytes
maximum blocks for ext2fs with 16K blocks:
	(16K / 4 = 4096)
	12 + 4096 + 4096*4096 + 4096*4096*4096 = 68 736 258 060 = ~4097 Gbytes
*****************************************************************************/
#include <sys/stat.h> /* S_IFDIR */
#include <string.h> /* strncpy(), strncmp() */
#include <dirent.h> /* struct dirent */
#include <errno.h>
#include "defs.h"

/* IMPORTS
from DISKIO.C */
int read_disk(drv_t *drv, unsigned long sector, unsigned offset,
		unsigned count, unsigned char HUGE *buf);

#define	EXT2_DIRECT_BLOCKS	12				/* 0-11 */
#define	EXT2_INDIRECT_BLOCK	EXT2_DIRECT_BLOCKS		/* 12 */
#define	EXT2_D_INDIRECT_BLOCK	(EXT2_INDIRECT_BLOCK + 1)	/* 13 */
#define	EXT2_T_INDIRECT_BLOCK	(EXT2_D_INDIRECT_BLOCK + 1)	/* 14 */
#define	EXT2_NUM_BLOCKS		(EXT2_T_INDIRECT_BLOCK + 1)	/* 15 */

#define	EXT2_MAX_DIRENT		256
#define	EXT2_GD_SIZE		32	/* group descriptors */
#define	EXT2_INODE_SIZE		128
#define	EXT2_ROOT_INODE		1

/* information stored in mount_t for each mounted ext2 volume */
typedef struct
{
	unsigned char sectors_per_block;
	unsigned short bytes_per_block; /* usu. 1K */
	unsigned long inodes_per_group, blocks_per_group;
	unsigned long total_blocks;
} ext2_mount_t;

/* information stored in file_t for each opened file on an ext2 volume */
typedef struct
{
	unsigned long blocks[EXT2_NUM_BLOCKS];
} ext2_file_t;

/* alas, these do not work with GNU C...
#if sizeof(ext2_mount_t)>MOUNT_INFO_SIZE
#error MOUNT_INFO_SIZE is too small
#endif

#if sizeof(ext2_file_t)>FILE_INFO_SIZE
#error FILE_INFO_SIZE is too small
#endif */
/*****************************************************************************
convert logical block (block-in-file) to physical block (block-in-disk)
*****************************************************************************/
static int ext2_l2p(file_t *file, unsigned long *physical_block,
		unsigned long logical_block)
{
	unsigned ents_per_block;
	ext2_mount_t *ext2_mount;
	ext2_file_t *ext2_file;
	unsigned long sector;
	unsigned char buf[4];
	unsigned offset;
	mount_t *mount;
	drv_t *drv;
	int err;

	mount = file->mount;
	drv = file->mount->drv;
	ext2_file = (ext2_file_t *)file->fs_file_info;
	ext2_mount = (ext2_mount_t *)mount->fs_mount_info;
	ents_per_block = ext2_mount->bytes_per_block / 4;
/* Assuming EXT2_DIRECT_BLOCKS==12 and 1K block size:
N=0...11 = direct block */
	if(logical_block < EXT2_DIRECT_BLOCKS)
	{
		*physical_block =
			ext2_file->blocks[(unsigned)logical_block];
		return 0;
	}
/* N=12...267 = indirect block (N-12 = 0...255) */
	logical_block -= EXT2_DIRECT_BLOCKS;
	if(logical_block < ents_per_block)
	{
		sector = ext2_file->blocks[EXT2_INDIRECT_BLOCK];
		goto EXT2_L2P_5;
	}
/* N=268...65803 = doubly-indirect block (N-12-256 = 0...?) */
	logical_block -= ents_per_block;
	if(logical_block < (unsigned long)ents_per_block * ents_per_block)
	{
		sector = ext2_file->blocks[EXT2_D_INDIRECT_BLOCK];
		goto EXT2_L2P_4;
	}
/* N=65804...16843019 = triply-indirect block (N-12-256-65536 = 0...?) */
	logical_block -= (unsigned long)ents_per_block * ents_per_block;
	sector = ext2_file->blocks[EXT2_T_INDIRECT_BLOCK];

	sector *= ext2_mount->sectors_per_block;
	offset = (logical_block % ents_per_block) * 4;
	logical_block /= ents_per_block;
	err = read_disk(drv, sector, offset, 4, buf);
	if(err != 0)
		return err;
	sector = read_le32(buf);
EXT2_L2P_4:
	sector *= ext2_mount->sectors_per_block;
	offset = (logical_block % ents_per_block) * 4;
	logical_block /= ents_per_block;
	err = read_disk(drv, sector, offset, 4, buf);
	if(err != 0)
		return err;
	sector = read_le32(buf);
EXT2_L2P_5:
	sector *= ext2_mount->sectors_per_block;
	offset = (logical_block % ents_per_block) * 4;
/*	logical_block /= ents_per_block; */
	err = read_disk(drv, sector, offset, 4, buf);
	if(err != 0)
		return err;
	*physical_block = read_le32(buf);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static off_t ext2_read(file_t *file, unsigned char HUGE *buf, off_t count)
{
	unsigned long logical_block, physical_block;
	ext2_mount_t *ext2_mount;
	unsigned offset;
	mount_t *mount;
	off_t total, w;
	drv_t *drv;
	int err;

	mount = file->mount;
	drv = mount->drv;
	ext2_mount = (ext2_mount_t *)mount->fs_mount_info;
/* limit count per file size and pos */
	if(file->pos >= file->size)
		return 0;
	if(file->pos + count > file->size)
		count = file->size - file->pos;
/* compute logical block and offset into block */
	logical_block = file->pos / ext2_mount->bytes_per_block;
	offset = file->pos % ext2_mount->bytes_per_block;
	total = 0;
	while(count != 0)
	{
/* convert logical block number to physical block number */
		err = ext2_l2p(file, &physical_block, logical_block);
		if(err != 0)
			return err;
/* */
		w = ext2_mount->bytes_per_block - offset;
		if(w > count)
			w = count;
		err = read_disk(drv, ext2_mount->sectors_per_block *
			physical_block, offset, w, buf);
		if(err != 0)
			return err;
		buf += w;
		count -= w;
		total += w;

		file->pos += w;
		offset = 0;

		logical_block++;
	}
	return total;
}
/*****************************************************************************
*****************************************************************************/
static int ext2_read_inode(mount_t *mount, unsigned char *inode,
		unsigned long inode_num)
{
	unsigned long group_descriptor_block, inode_table_block;
	unsigned char group_descriptor[EXT2_GD_SIZE];
	unsigned group, inode_in_group;
	ext2_mount_t *ext2_mount;
	drv_t *drv;
	int err;

	ext2_mount = (ext2_mount_t *)mount->fs_mount_info;
	drv = mount->drv;
/* convert inode_num to group number and inode_in_group */
	group		= inode_num / ext2_mount->inodes_per_group;
	inode_in_group	= inode_num % ext2_mount->inodes_per_group;
/* read group descriptors until you find one that's valid */
	group_descriptor_block = 2;
	while(group_descriptor_block < ext2_mount->total_blocks)
	{
		err = read_disk(drv, ext2_mount->sectors_per_block *
			group_descriptor_block, group * EXT2_GD_SIZE,
			EXT2_GD_SIZE, group_descriptor);
		if(err == 0)
			goto OK;
		group_descriptor_block += ext2_mount->blocks_per_group;
	}
	return -1;
/* get starting block number of inode table */
OK:
	inode_table_block = read_le32(group_descriptor + 8);
	err = read_disk(drv, ext2_mount->sectors_per_block *
		inode_table_block, inode_in_group * EXT2_INODE_SIZE,
		EXT2_INODE_SIZE, inode);
	if(err != 0)
		return err;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int ext2_readdir(file_t *dir, struct dirent *ent)
{
	unsigned dir_rec_len, dir_name_len;
	unsigned char buf[EXT2_MAX_DIRENT];
	off_t temp, count;

/* read initial 8 bytes of ext2 directory entry */
	temp = ext2_read(dir, buf, 8);
	if(temp < 0)
		return (int)temp;
	if(temp != 8)
		return -ENOENT;
/* get total length of this directory entry and length of filename */
	dir_rec_len = read_le16(buf + 4);
//	dir_name_len = read_le16(buf + 6);
	dir_name_len = read_le16(buf + 6) & 0xFF; /* xxx - FIWIX disk */
/* read filename */
	count = dir_name_len;
	if(count == 0)
		return -ENOENT;
	temp = ext2_read(dir, buf, count);
	if(temp < 0)
		return (int)temp;
	if(temp != count)
		return -ENOENT;
/* 0-terminate it, then copy to ent */
	buf[dir_name_len] = '\0';
	strncpy(ent->d_name, (char *)buf, EXT2_MAX_DIRENT);
/* skip pad bytes at end of directory entry */
	if(dir_rec_len > dir_name_len + 8)
/* lseek() needs an integer handle, so set dir->pos directly */
		dir->pos += dir_rec_len - dir_name_len - 8;
	return 0;
}
/*****************************************************************************
find file/dir named 'name' in opened directory specified by 'file',
overwrite 'file' with information about the target file/dir
*****************************************************************************/
static int ext2_find(file_t *file, const char *name, const unsigned name_len)
{
	unsigned dir_rec_len, dir_name_len, i;
	unsigned char buf[EXT2_MAX_DIRENT];
	unsigned long inode_num;
	ext2_file_t *ext2_file;
	mount_t *mount;
	off_t temp;
	int err;

	mount = file->mount;
	ext2_file = (ext2_file_t *)file->fs_file_info;
/* opening root dir? */
	if(name[0] == '/' && name_len == 1)
	{
/* read inode #1 for root dir */
		err = ext2_read_inode(mount, buf, EXT2_ROOT_INODE);
		if(err != 0)
			return err;
/* get FS-independent info */
		file->pos = 0;
		file->size = read_le16(buf + 4);
		file->mode = S_IFDIR | 0700; /* rwx------ */
/* get FS-dependent info, namely... */
		ext2_file = (ext2_file_t *)file->fs_file_info;
/* ...the 15 block pointers */
		for(i = 0; i < EXT2_NUM_BLOCKS; i++)
			ext2_file->blocks[i] = read_le32(buf + 40 + 4 * i);
		return 0;
	}
	while(1)
	{
/* read initial 8 bytes of ext2 directory entry */
		temp = ext2_read(file, buf, 8);
		if(temp < 0)
			return (int)temp;
		if(temp != 8)
			return -ENOENT;
/* save these */
		inode_num = read_le32(buf + 0);
		dir_rec_len = read_le16(buf + 4);
//		dir_name_len = read_le16(buf + 6);
		dir_name_len = read_le16(buf + 6) & 0xFF; /* xxx - FIWIX disk */
/* read filename */
		temp = ext2_read(file, buf, dir_name_len);
		if(temp < 0)
			return temp;
		if(temp != dir_name_len)
			return -ENOENT;
/* 0-terminate it, then compare */
		buf[dir_name_len] = '\0';
		if(!strncmp((char *)buf, name, name_len))
			break;
/* compare failed; skip pad bytes at end of directory entry */
		if(dir_rec_len > dir_name_len + 8)
/* lseek() needs an integer handle, so set dir->pos directly */
			file->pos += dir_rec_len - dir_name_len - 8;
	}
/* found it!
read 128-byte inode */
	err = ext2_read_inode(file->mount, buf, inode_num - 1);
	if(err != 0)
		return err;
/* get FS-independent info */
	file->pos = 0;
	file->size = read_le32(buf + 4);
	file->mode = read_le16(buf + 0);
/* get FS-dependent info (the 15 block pointers) */
	ext2_file = (ext2_file_t *)file->fs_file_info;
	for(i = 0; i < EXT2_NUM_BLOCKS; i++)
		ext2_file->blocks[i] = read_le32(buf + 40 + 4 * i);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
const fs_info_t _ext2_fs =
{
	"ext2", ext2_read, ext2_readdir, ext2_find
};
/*****************************************************************************
*****************************************************************************/
int ext2_mount(mount_t *mount, drv_t *drv)
{
	ext2_mount_t *ext2_mount;
	unsigned char buf[1024];
	unsigned temp;
	int err;

	if(drv->part_type != 0 &&	/* unknown */
		drv->part_type != 0x83)	/* ext2fs */
			return -1;
	ext2_mount = (ext2_mount_t *)mount->fs_mount_info;
/* try reading first superblock, which is always at offset
1024 of disk, regardless of sector size or block size */
	if(drv->bps > 1024)
		err = read_disk(drv, 0, 1024, 1024, buf);
	else
		err = read_disk(drv, 1024 / drv->bps, 0, 1024, buf);
	if(err != 0)
		return err;
/* check magic value */
	temp = read_le16(buf + 56);
	if(temp != 0xEF53)
		return -1;
/* init mount */
	mount->drv = drv;
	mount->fs_info = &_ext2_fs;
/* init FS-specific mount data */
	ext2_mount->total_blocks = read_le32(buf + 4);
	temp = read_le32(buf + 24);	/* lg2(block_size / 1024) */
	temp = 1024 << temp;		/* block_size */
	ext2_mount->bytes_per_block = temp;
	ext2_mount->sectors_per_block =	temp / drv->bps;
	ext2_mount->blocks_per_group = read_le32(buf + 32);
	ext2_mount->inodes_per_group = read_le32(buf + 40);
	return 0;
}
