/*============================================================================
ELF.C - PMODE (32-BIT) ELF KERNELS

EXPORTS:
int check_elf_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset);
============================================================================*/
#include <stdio.h> /* sprintf() */
#include <errno.h>
#include <io.h> /* read(), lseek() */
#include "execfile.h"
#include "defs.h"

#define	ELF_FILE_MAGIC		0	/* "\x7F""ELF" */
#define	ELF_FILE_BITNESS	4	/* 1=32-bit, 2=64-bit */
#define	ELF_FILE_ENDIAN		5	/* 1=little endian, 2=big endian */
#define	ELF_FILE_VER1		6	/* =1 */
/* bytes 7-15 are reserved */
#define	ELF_FILE_TYPE		16	/* 1=.o, 2=executable, 3=DLL */
#define	ELF_FILE_MACHINE	18	/* 2=SPARC, 3=i386, 4=68000... */
#define	ELF_FILE_VER2		20	/* =1 */
#define	ELF_FILE_ENTRY		24	/* initial EIP */
#define	ELF_PHTAB_OFFSET	28	/* file offset of Program Header table */
#define	ELF_SHTAB_OFFSET	32	/* file offset of Section Header table */
#define	ELF_FILE_FLAGS		36	/* (not used for i386?) */
#define	ELF_FILE_HDRSIZE	40	/* size of this header, usu. 52 */
#define	ELF_PHTAB_ENTSIZE	42	/* size of entries in PH table */
#define	ELF_PHTAB_COUNT		44	/* number of entries in PH table */
#define	ELF_SHTAB_ENTSIZE	46	/* size of entries in SH table */
#define	ELF_SHTAB_COUNT		48	/* number of entries in SH table */
#define	ELF_SHSTRTAB_INDEX	50	/* SH table entry for .shstrtab */
/* 52 bytes long */
#define	ELF_FILE_HDRLEN		52

#define	ELF_PH_TYPE		0
#define	ELF_PH_OFFSET		4
#define	ELF_PH_VIRT_ADR		8
#define	ELF_PH_PADDR		12
#define	ELF_PH_SIZE_IN_FILE	16
#define	ELF_PH_SIZE_IN_MEM	20
#define	ELF_PH_FLAGS		24
#define	ELF_PH_ALIGN		28
/* 32 bytes long */
#define	ELF_PH_LEN		32
/*****************************************************************************
*****************************************************************************/
int check_elf_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, unsigned long image_base)
{
	static const char *format_name = "ELF";
/**/
	unsigned short phtab_ent_size, num_phtab_ents, s, i;
	unsigned long temp, phtab_off;
/* 52 = max of ELF_FILE_HDRLEN (=52) and ELF_PH_LEN (=32) */
	char buf[52];
	sect_t *sect;
	int err;

/* seek to start of kernel within RDSK file */
	lseek(handle, rdsk_offset, SEEK_SET);
/* read 52-byte file header */
	if(read(handle, buf, ELF_FILE_HDRLEN) != ELF_FILE_HDRLEN)
/* short file is not ELF */
		return -ERR_FILE_FORMAT;
/* check if valid ELF and get info */
	if(buf[ELF_FILE_BITNESS] != 1 ||		/* 32-bit */
		buf[ELF_FILE_ENDIAN] != 1 ||		/* little endian */
		buf[ELF_FILE_VER1] != 1)		/* ELF ver 1 */
			return -ERR_FILE_FORMAT;
	if(read_le16(buf + ELF_FILE_TYPE) != 2 ||	/* executable */
		read_le16(buf + ELF_FILE_MACHINE) != 3 || /* i386 */
		read_le32(buf + ELF_FILE_VER2) != 1)	/* ELF ver 1 */
			return -ERR_FILE_FORMAT;
/* the entry point */
	exec->entry_pt = read_le32(buf + ELF_FILE_ENTRY);
/* go to program header table and read it */
	phtab_ent_size = read_le16(buf + ELF_PHTAB_ENTSIZE);
	num_phtab_ents = read_le16(buf + ELF_PHTAB_COUNT);
	phtab_off = read_le32(buf + ELF_PHTAB_OFFSET);
	s = 0;
	for(i = 0; i < num_phtab_ents; i++)
	{
		sect = exec->section + s;
		lseek(handle, phtab_off + phtab_ent_size * i + rdsk_offset,
			SEEK_SET);
		err = read(handle, buf, ELF_PH_LEN);
		if(err != ELF_PH_LEN)
			return -EIO;
		temp = read_le32(buf + ELF_PH_TYPE);
/* choke on 2=DYNAMIC and the forbidden 5=SHLIB segments */
		if(temp == 2 || temp == 5)
			return -ERR_FILE_FORMAT;
/* handle 1=LOAD segment */
		/*else*/ if(temp == 1)
		{
			sect->adr = read_le32(buf + ELF_PH_VIRT_ADR);
			sect->size = read_le32(buf + ELF_PH_SIZE_IN_FILE);
			sect->off = read_le32(buf + ELF_PH_OFFSET);
			temp = read_le32(buf + ELF_PH_FLAGS);
			if(temp & 4)
				sect->read = 1;
			if(temp & 2)
				sect->write = 1;
			if(temp & 1)
				sect->exec = 1;
			sect->load = 1;
			sect->zero = 0;
			sprintf(sect->name, "(%u)", s);
			s++;
/* if size-in-mem>size-in-file, this segment contains the BSS */
			temp = read_le32(buf + ELF_PH_SIZE_IN_MEM);
			if(temp > sect->size)
			{
				sect[1].adr = sect->adr + sect->size;
				sect[1].size = temp - sect->size;
				sect[1].read = sect[1].write =
					sect[1].zero = 1;
				sect[1].exec = sect[1].load = 0;
				sprintf(sect[1].name, "(%u)", s);
				s++;
			}
		}
/* ignore 0=NULL, 6=PHDR, 3=INTERP, and 4=NOTE segments
		else
			nothing; */
	}
	exec->num_sections = s;
	exec->format_name = format_name;
	exec->pmode = 1;
	return 0;
}
