/*****************************************************************************
DISK INPUT FUNCTIONS

EXPORTS:
int read_sector(drv_t *drv, unsigned long lba, unsigned char **buf_p);
int open_drive(drv_t *drv, unsigned char dev_num, signed char part_num);
int read_disk(drv_t *drv, unsigned long sector, unsigned offset,
		unsigned count, unsigned char HUGE *buf);

For open_drive():
	dev_num = 0, 1, 2... for floppy drives
	dev_num = 0x80, 0x81, 0x82... for hard drives

	part_num = 0...3 to open one of the hard drive primary partitions
	part_num = -1u (==255) to open the entire hard drive
	part_num ignored if opening floppy drive
*****************************************************************************/
#include <string.h> /* memset() */
#include <errno.h>
#include <bios.h> /* _DISK_... _bios_disk() */
/* FP_SEG(), FP_OFF(), MK_FP(), struct REGPACK, intr() */
#include <dos.h>
#include "defs.h"

/* IMPORTS
from CONIO.C */
int printf(const char *fmt, ...);

/* from FAT.C */
int check_if_fat_bootsector(unsigned char *buf);

#define	CACHE_SIZE	8192
/* maximum and minium bytes-per-sector */
#define	MAX_BPS		4096
#define	MIN_BPS		512
#define	MAX_CACHE	(CACHE_SIZE / MIN_BPS)

#if __TURBOC__<0x0300
#define	_DISK_RESET	0
#define	_DISK_READ	2
#endif
/*****************************************************************************
memory for the disk cache must be aligned on a MAX_BPS boundary
so we don't get error 0x09 from INT 13h (DMA crossed 64K boundary).
Only floppy disk accesses use ISA DMA, so if you want this code
to work only with hard disks, memory alignment does not matter.
*****************************************************************************/
static int alloc_cache(unsigned char **cache)
{
/* pad with MAX_BPS-1 bytes to ensure alignment to MAX_BPS boundary */
	static unsigned char u_cache[CACHE_SIZE + MAX_BPS - 1];
/**/
	unsigned long cache_adr, seg;

/* form linear address of unaligned cache */
	seg = FP_SEG(u_cache);//_DS;
	seg <<= 4;
	cache_adr = seg + FP_OFF(u_cache);//(unsigned)u_cache;
/* align to MAX_BPS-byte boundary */
	cache_adr += MAX_BPS - 1;
	cache_adr &= -(long)MAX_BPS;
/* this is the address of the aligned cache */
	cache_adr -= seg;
	(*cache) = (unsigned char *)cache_adr;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int get_hard_disk_geometry(drv_t *drv)
{
	unsigned cyls;
	regs_t regs;

/* use INT 13h AH=08h to count the number of hard drives and get geometry */
	regs.R_AX = 0x0800;
	regs.R_DX = drv->dev_num;
	intr(0x13, &regs);
	if((regs.R_FLAGS & 0x0001) != 0)
	{
		printf("get_hard_disk_geometry: error getting geometry for "
			"drive 0x%02X\n", drv->dev_num);
		return -1;
	}
	if((drv->dev_num & 0x7F) >= (regs.R_DX & 0xFF))
	{
		printf("get_hard_disk_geometry: no such "
			"hard drive 0x%02X\n", drv->dev_num);
		return -1;
	}
	drv->bps = 512;
	drv->sectors = regs.R_CX & 0x3F;
	regs.R_DX >>= 8;
	drv->heads = regs.R_DX + 1;
	regs.R_CX >>= 6;
	cyls = regs.R_CX;
	if(cyls & 0x01)
		cyls |= 0x400;
	if(cyls & 0x02)
		cyls |= 0x800;
	cyls >>= 2;
	drv->part_size = (unsigned long)cyls * drv->heads * drv->sectors;
	DEBUG(printf("get_hard_disk_geometry: disk 0x%02X: CHS=%u:%u:%u, "
		"size=%lu", drv->dev_num, cyls, drv->heads,
		drv->sectors, drv->part_size);
	if(drv->use_lba)
			printf(", LBA\n");
		else
			printf("\n");
	)
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int read_sector_lba(drv_t *drv, unsigned long lba, unsigned char *buf)
{
	unsigned char lba_pkt[16];
	struct diskinfo_t cmd;
	unsigned tries;
	regs_t regs;

	memset(lba_pkt, 0, 16);
	lba_pkt[0] = 16;
	write_le16(lba_pkt + 2, 1);
	write_le32(lba_pkt + 4, (unsigned long)buf);
	regs.R_DS = FP_SEG(lba_pkt);//_DS;
	regs.R_SI = FP_OFF(lba_pkt);//(unsigned)lba_pkt;
	write_le32(lba_pkt + 8, lba);
	regs.R_AX = 0x4200;
	regs.R_DX = drv->dev_num;
/* make 3 attempts */
	for(tries = 3; tries != 0; tries--)
	{
		intr(0x13, &regs);
		if((regs.R_FLAGS & 0x0001) == 0)
			return 0;
		regs.R_AX >>= 8;
		printf("\nINT 13h AH=42h (LBA disk read) error 0x%02X\n",
			regs.R_AX);
/* reset drive */
		cmd.drive = drv->dev_num;
		_bios_disk(_DISK_RESET, &cmd);
	}
	regs.R_AX >>= 8;
	printf("read_sector_lba: error 0x%02X reading drive 0x%02X\n",
		regs.R_AX, drv->dev_num);
	return -1;
}
/*****************************************************************************
*****************************************************************************/
static int read_sector_chs(drv_t *drv, unsigned long lba, unsigned char *buf)
{
	struct diskinfo_t cmd;
	int tries, err;

	cmd.drive = drv->dev_num;
	cmd.nsectors = 1;
	cmd.buffer = buf;
	if(lba == 0)
	{
		cmd.head = cmd.track = 0;
		cmd.sector = 1;
	}
	else
	{
		if(drv->sectors == 0 || drv->heads == 0)
		{
			printf("read_sector_chs: drive 0x%02X has "
				"unknown CHS geometry\n",
				drv->dev_num);
			return -1;
		}
		cmd.sector = lba % drv->sectors + 1;
		cmd.head = (lba / drv->sectors) % drv->heads;
		cmd.track = (lba / drv->sectors) / drv->heads;
	}
/* make 3 attempts */
	for(tries = 3; tries != 0; tries--)
	{
		err = _bios_disk(_DISK_READ, &cmd);
		err >>= 8; /* grrr...not needed for biosdisk() */
		if(err == 0)
			return 0;
/* reset drive */
		_bios_disk(_DISK_RESET, &cmd);
	}
	printf("read_sector_chs: error 0x%02X reading drive 0x%02X\n",
		err, drv->dev_num);
	return -1;
}
/*****************************************************************************
*****************************************************************************/
int read_sector(drv_t *drv, unsigned long lba, unsigned char **buf_p)
{
	static unsigned long cached_lba[MAX_CACHE];
	static unsigned char cached_dev[MAX_CACHE];
	static unsigned char *cache, evict;
/**/
	unsigned i, max;
	int err;

	if(cache == NULL)
	{
		err = alloc_cache(&cache);
		if(err != 0)
			return err;
		for(i = 0; i < MAX_CACHE; i++)
			cached_lba[i] = -1uL;
	}
	lba += drv->part_start;
	max = CACHE_SIZE / drv->bps;
/* see if this sector is cached
DUMBDRV :) */
	for(i = 0; i < max; i++)
	{
		if(cached_lba[i] == lba &&
			cached_dev[i] == drv->dev_num)
		{
			(*buf_p) = cache + i * drv->bps;
			return 0;
		}
	}
/* not cached, find a free buffer for it */
	for(i = 0; i < max; i++)
	{
		if(cached_lba[i] == -1uL)
			break;
	}
/* no free buffer, kick out someone else */
	if(i >= max)
	{
		i = evict;
		evict++;
		if(evict >= max)
			evict = 0;
	}
/* load it */
	cached_lba[i] = lba;
	cached_dev[i] = drv->dev_num;
	(*buf_p) = cache + i * drv->bps;
/* this is for debug: */
memset(*buf_p, 0x55, drv->bps);
	if(drv->use_lba)
		return read_sector_lba(drv, lba, *buf_p);
	return read_sector_chs(drv, lba, *buf_p);
}
/*****************************************************************************
*****************************************************************************/
static int get_fat_floppy_geometry(drv_t *drv)
{
	unsigned char *buf;
	int err;

	err = read_sector(drv, 0, &buf);
	if(err != 0)
		return -1; /* fatal error, e.g. no disk in drive */
	if(check_if_fat_bootsector(buf) != 0)
		return 1; /* non-fatal error; try something else */
	drv->bps = read_le16(buf + 11);
	drv->sectors = read_le16(buf + 24);
	drv->heads = read_le16(buf + 26);
	drv->part_size = read_le16(buf + 19);
	if(drv->part_size == 0)
		drv->part_size = read_le32(buf + 32);
	DEBUG(printf("get_fat_floppy_geometry: disk 0x%02X: CHS=??:%u:%u, "
		"size=%lu\n", drv->dev_num, drv->heads,
		drv->sectors, drv->part_size);)
	return 0;
}
/*****************************************************************************
now using linear search instead of binary...much faster
*****************************************************************************/
static int probe_floppy_geometry(drv_t *drv)
{
	unsigned char *buf;
	int err;

	drv->bps = 512;
/* check if disk in drive by reading sector 0
this also clears the disk-change state */
	err = read_sector(drv, 0, &buf);
	if(err != 0)
		return err;
/* scan upwards for sector number where read fails */
	for(drv->sectors = 1; drv->sectors < 40; drv->sectors++)
	{
		DEBUG(printf("probe_floppy_geometry: reading sector %u\n",
			drv->sectors - 1);)
		if(read_sector(drv, drv->sectors - 1, &buf) != 0)
			break;
	}
	drv->sectors--;
/* read sector 0 to reset drive */
	(void)read_sector(drv, 0, &buf);
/* xxx - probe number of heads and tracks, too */
	drv->heads = 2;
	drv->part_size = 80L * drv->heads * drv->sectors;
	DEBUG(printf("probe_floppy_geometry: disk 0x%02X: CHS=??:%u:%u, "
		"size=%lu\n", drv->dev_num, drv->heads,
		drv->sectors, drv->part_size);)
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int get_floppy_disk_geometry(drv_t *drv)
{
	int err;

	err = get_fat_floppy_geometry(drv);
	if(err > 0)
	{
/* either probe the number of sectors per track (kinda slow)... */
#if 0
		err = probe_floppy_geometry(drv);
#else
/* ...or just assume 1.44 meg disk */
		drv->bps = 512;
		drv->sectors = 18;
		drv->heads = 2;
		drv->part_size = 80L * drv->heads * drv->sectors;
		err = 0;
#endif
	}
	return err;
}
/*****************************************************************************
*****************************************************************************/
static int get_partition_info(drv_t *drv)
{
	unsigned char *buf, *ptab_rec;
	int err;

/* read sector 0: the partition table, or Master Boot Record (MBR) */
	err = read_sector(drv, 0, &buf);
	if(err != 0)
		return err;
/* make sure the desired partition exists */
	ptab_rec = buf + 446 + 16 * drv->part_num;
	if(ptab_rec[4] == 0)
	{
		printf("get_partition_info: no such partition %u on drive "
			"0x%02X\n", drv->part_num, drv->dev_num);
		return -1;
	}
/* get part_type, part_start, part_size */
	drv->part_type = ptab_rec[4];
	drv->part_start = read_le32(ptab_rec + 8);
	drv->part_size = read_le32(ptab_rec + 12);
	DEBUG(printf("get_partition_info: disk 0x%02X, part %u: start=%lu, "
		"size=%lu, type=0x%02X\n", drv->dev_num, drv->part_num,
		drv->part_start, drv->part_size, drv->part_type);)
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int open_drive(drv_t *drv, unsigned dev_num, int part_num)
{
	int err = -1;
	regs_t regs;

	drv->dev_num = dev_num;
	drv->part_num = part_num;
	drv->part_start = 0;
	drv->part_type = 0;
/* initial values just to make read_sector() work
until disk geometry can be probed */
	drv->bps = 512;
	drv->heads = 1;
	drv->sectors = 127;
/* opening floppy drive */
	if(dev_num < 0x80)
	{
		drv->use_lba = 0;
/* set drv->heads, drv->sectors, drv->part_size */
		err = get_floppy_disk_geometry(drv);
		if(err != 0)
			return err;
	}
/* opening hard drive */
	else
	{
		if(part_num < -1 || part_num > 3)
		{
			printf("open_drive: partition number (%d) must be "
				"-1 ... 3\n", part_num);
			return -1;
		}
/* check if INT 13h LBA extensions supported in BIOS and by drive */
		regs.R_AX = 0x4100;
		regs.R_BX = 0x55AA;
		regs.R_DX = dev_num;
		intr(0x13, &regs);
		if((regs.R_FLAGS & 0x0001) || regs.R_BX != 0xAA55)
		{
			drv->use_lba = 0;
/* set drv->heads, drv->sectors for CHS hard disk,
set drv->part_size, drv->bps for both CHS and LBA hard disks */
			err = get_hard_disk_geometry(drv);
			if(err != 0)
				return err;
		}
/* always use LBA for hard drive if drive and BIOS support it */
		else
			drv->use_lba = 1;
/* set drv->part_size, drv->part_start, drv->part_type */
		if(part_num != -1)
			err = get_partition_info(drv);
	}
	return err;
}
/*****************************************************************************
there is no _fmemcpy() under   Turbo C 2.0
there is no _fmemcpy() under   Turbo C 2.9 (Turbo C++ 1.0)
huge pointers are broken under Turbo C 4.0 (Turbo C++ 3.0)
*****************************************************************************/
#if defined(__WATCOMC__)
static void memcpy2f(unsigned char HUGE **dst_p, unsigned char *src,
		unsigned count)
{
	_fmemcpy(*dst_p, src, count);
}
#elif defined(__TURBOC__)
#if __TURBOC__>=0x0300
static void memcpy2f(unsigned char HUGE **dst_p, unsigned char *src,
		unsigned count)
{
	_fmemcpy(*dst_p, src, count);
}
#else
/* this is slow, but... */
static void memcpy2f(unsigned char HUGE **dst_p, unsigned char *src,
		unsigned count)
{
	unsigned char HUGE *dst;

	dst = (*dst_p);
	for(; count != 0; count--)
	{
		*dst = *src;
		dst++;
		src++;
	}
}
#endif
#endif
/*****************************************************************************
read a "run" of bytes from contiguous disk sectors. Starting offset in
first sector and ending offset in last sector are completely arbitrary
*****************************************************************************/
int read_disk(drv_t *drv, unsigned long sector, unsigned offset,
		unsigned count, unsigned char HUGE *buf)
{
	unsigned char *cache;
	unsigned w;
	int err;

/* skip leading sectors */
	while(offset >= drv->bps)
	{
		sector++;
		offset -= drv->bps;
	}
/* read full or partial sectors */
	while(count != 0)
	{
		err = read_sector(drv, sector, &cache);
		if(err != 0)
			return err;
		w = drv->bps - offset;
		if(w > count)
			w = count;
		memcpy2f(&buf, cache + offset, w);
		/* buf += w; */
		count -= w;

		sector++;
		offset = 0;
	}
	return 0;
}
