/*****************************************************************************
CONSOLE OUTPUT FUNCTIONS

EXPORTS:
int getch(void);
int printf(const char *fmt, ...);
*****************************************************************************/
#include <stdarg.h> /* va_list, va_start(), va_end() */
#include <stdio.h> /* vsprintf() */
#include <ctype.h> /* isdigit() */
#include <dos.h> /* struct REGPACK (or union REGPACK), intr() */
#include "defs.h"

static unsigned char _attrib = 0x07; /* white on black */
/*****************************************************************************
*****************************************************************************/
int getch(void)
{
	static short prev = -1;
/**/
	regs_t regs;
	int ret_val;

	if(prev != -1)
	{
		ret_val = prev;
		prev = -1;
	}
	else
	{
		regs.R_AX = 0; /* AH=subfunction 0 (get key) */
		intr(0x16, &regs);
		ret_val = regs.R_AX & 0xFF;
		if(ret_val == 0)
		{
			regs.R_AX >>= 8;
			prev = regs.R_AX;
		}
	}
	return ret_val;
}
/*****************************************************************************
*****************************************************************************/
static int putch(int c)
{
	unsigned char buf[2];
	regs_t regs;

	buf[0] = c;
	buf[1] = '\0';

	regs.R_BX = 0;		/* BH=video page 0 */
	regs.R_AX = 0x0300; /* AH=subfunction 3: get cursor pos to DH,DL */
	intr(0x10, &regs);

	regs.R_ES = FP_SEG(buf);//_SS;
	regs.R_BP = FP_OFF(buf);//(unsigned)buf;
	regs.R_CX = 1;
	regs.R_BX = _attrib;	/* BH=video page 0, BL=attribute */
	regs.R_AX = 0x1301;	/* AH=subfunction 13h, AL=write mode 1 */
	intr(0x10, &regs);
	return c;
}
/*****************************************************************************
*****************************************************************************/
static void gotoxy(unsigned char x, unsigned char y)
{
	regs_t regs;

	x--;
	y--;
	regs.R_DX = y;
	regs.R_DX <<= 8;
	regs.R_DX |= x;
	regs.R_BX = 0;		/* BH=video page 0 */
	regs.R_AX = 0x0200;	/* AH=subfunction 2 (move csr) */
	intr(0x10, &regs);
}
/*****************************************************************************
*****************************************************************************/
static void clrscr(void)
{
	regs_t regs;

	regs.R_AX = 0x0600;	/* AH=subfunction 6 (scroll), AH=0 (erase) */
	regs.R_BX = _attrib;
	regs.R_BX <<= 8;	/* BH=attribute */
	regs.R_CX = 0;		/* CH=row, CH=col (0,0) */
	regs.R_DX = 0x184F;	/* DH=height, DL=width (25,80) */
	intr(0x10, &regs);
}
/*****************************************************************************
*****************************************************************************/
static void textattr(int a)
{
	_attrib = a;
}
/*****************************************************************************
*****************************************************************************/
static void set_attrib(unsigned att)
{
/* 'static const' doesn't work with Turbo C 2.0 (dunno why)
	static const unsigned char ansi_to_vga[] = */
	static unsigned char ansi_to_vga[] =
	{
		0, 4, 2, 6, 1, 5, 3, 7
	};
/* 0x1F == bright white on blue */
	static unsigned char attrib = 0x1F;
/**/

/* bold on/off */
	if(att == 0)
		attrib &= ~0x08;
	else if(att == 1)
		attrib |= 0x08;
/* set foreground color */
	else if(att >= 30 && att <= 37)
	{
		att = ansi_to_vga[att - 30];
		attrib = (attrib & ~0x07) | att;
	}
/* set background color */
	else if(att >= 40 && att <= 47)
	{
		att = ansi_to_vga[att - 40] << 4;
		attrib = (attrib & ~0x70) | att;
	}
	textattr(attrib);
}
/*****************************************************************************
*****************************************************************************/
static int put_char(int c)
{
	static unsigned char _vc_width = 80, _vc_height = 50;
	static unsigned char esc, esc1, esc2, esc3;
	static signed char csr_x, csr_y;
/**/

/* state machine to handle the escape sequences
ESC */
	if(esc == 1)
	{
		if(c == '[')
		{
			esc++;
			esc1 = 0;
			return c;
		}
		/* else fall-through: zero esc and print c */
	}
/* ESC[ */
	else if(esc == 2)
	{
		if(isdigit(c))
		{
			esc1 = esc1 * 10 + c - '0';
			return c;
		}
		else if(c == ';')
		{
			esc++;
			esc2 = 0;
			return c;
		}
/* ESC[2J -- clear screen */
		else if(c == 'J')
		{
			if(esc1 == 2)
			{
				clrscr();
				csr_x = csr_y = 0;
			}
		}
/* ESC[num1m -- set attribute num1 */
		else if(c == 'm')
			set_attrib(esc1);
		esc = 0;	/* anything else with one numeric arg */
		return c;
	}
/* ESC[num1; */
	else if(esc == 3)
	{
		if(isdigit(c))
		{
			esc2 = esc2 * 10 + c - '0';
			return c;
		}
		else if(c == ';')
		{
			esc++;	/* ESC[num1;num2; */
			esc3 = 0;
			return c;
		}
/* ESC[num1;num2H -- move cursor to num1,num2 */
		else if(c == 'H')
		{
			if(esc2 < _vc_width)
				csr_x = esc2;
			if(esc1 < _vc_height)
				csr_y = esc1;
			gotoxy(csr_x, csr_y);
		}
/* ESC[num1;num2m -- set attributes num1,num2 */
		else if(c == 'm')
		{
			set_attrib(esc1);
			set_attrib(esc2);
		}
		esc = 0;
		return c;
	}
/* ESC[num1;num2;num3 */
	else if(esc == 4)
	{
		if(isdigit(c))
		{
			esc3 = esc3 * 10 + c - '0';
			return c;
		}
/* ESC[num1;num2;num3m -- set attributes num1,num2,num3 */
		else if(c == 'm')
		{
			set_attrib(esc1);
			set_attrib(esc2);
			set_attrib(esc3);
		}
		esc = 0;
		return c;
	}
	esc = 0;

/* escape character */
	if(c == 0x1B)
	{
		esc = 1;
		return c;
	}
/* backspace */
	if(c == 0x08)
	{
		if(csr_x != 0)
			csr_x--;
	}
/* tab */
	else if(c == 0x09)
		csr_x = (csr_x + 8) & ~(8 - 1);
/* carriage return */
	else if(c == '\r')	/* 0x0D */
		csr_x = 0;
/* line feed
	else if(c == '\n')	0x0A
		csr_y++; */
/* CR/LF */
	else if(c == '\n')	/* ### - 0x0A again */
	{
		csr_x = 0;
		csr_y++;
	}
/* printable ASCII */
	else if(c >= ' ')
	{
		gotoxy(csr_x + 1, csr_y + 1);
		putch(c);
		csr_x++;
	}
	if(csr_x >= _vc_width)
	{
		csr_x = 0;
		csr_y++;
	}
	if(csr_y >= _vc_height)
		csr_y = _vc_height - 1;
	gotoxy(csr_x + 1, csr_y + 1);
	return c;
}
/*****************************************************************************
*****************************************************************************/
int printf(const char *fmt, ...)
{
	char buf[256], *s;
	va_list args;
	int ret_val;

	va_start(args, fmt);
	ret_val = vsprintf(buf, fmt, args);
	va_end(args);
	for(s = buf; *s != '\0'; s++)
		put_char(*s);
	return ret_val;
}
