/*============================================================================
COFF.C - PMODE (32-BIT) DJGPP COFF KERNELS

EXPORTS:
int check_coff_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, unsigned long image_base);
============================================================================*/
#include <string.h> /* memcmp(), strcpy() */
#include <stdio.h> /* sprintf() */
#include <errno.h>
#include <io.h> /* read(), lseek() */
#include "execfile.h"
#include "defs.h"

/* COFF file header */
#define	COFF_FILE_MAGIC		0	/* 0x014C */
#define	COFF_FILE_SECT_NUM	2	/* number of sections */
#define	COFF_FILE_TIMEDATE	4	/* time and date stamp */
#define	COFF_FILE_SYMTAB_OFF	8	/* file offset of symbol table */
#define	COFF_FILE_SYMTAB_NUM	12	/* number of symtab entries */
#define	COFF_FILE_OPTHDR_SIZE	16	/* "optional" (aout) header size */
#define	COFF_FILE_FLAGS		18
/* 20 bytes to here */
#define	COFF_AOUT_MAGIC		20	/* 0x010B */
#define	COFF_AOUT_VER		22	/* version stamp */
#define	COFF_AOUT_CODE_SIZE	24
#define	COFF_AOUT_DATA_SIZE	28
#define	COFF_AOUT_BSS_SIZE	32
#define	COFF_AOUT_ENTRY		36	/* initial EIP */
#define	COFF_AOUT_CODE_OFF	40	/* file offset of code */
#define	COFF_AOUT_DATA_OFF	44	/* file offset of data */
/* 48 bytes long */
#define	COFF_FILE_HDRLEN	48

/* COFF section header */
#define	COFF_SECT_NAME		0	/* ".text", ".data", etc. */
#define	COFF_SECT_PHYS_ADR	8	/* physical adr (?) */
#define	COFF_SECT_VIRT_ADR	12	/* load adr (virtual) of section */
#define	COFF_SECT_SIZE		16
#define	COFF_SECT_OFF		20	/* file offset of section */
#define	COFF_SECT_RELOC_OFF	24	/* file offset of relocations */
#define	COFF_SECT_LINENUM_OFF	28	/* file offset of line number info */
#define	COFF_SECT_RELOC_NUM	32	/* number of relocations */
#define	COFF_SECT_LINENUM_NUM	34	/* number of line numbers */
#define	COFF_SECT_FLAGS		36
/* 40 bytes long */
#define	COFF_SECT_HDRLEN	40
/*****************************************************************************
*****************************************************************************/
int check_coff_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, unsigned long image_base)
{
	static const char *format_name = "DJGPP COFF";
/**/
	unsigned short aout_hdr_size;
	unsigned long bss_size;
	unsigned short i;
	char buf[128];
	sect_t *sect;
	off_t err;

/* seek to start of kernel within RDSK file */
	lseek(handle, rdsk_offset, SEEK_SET);
/* read 48-byte file header */
	if(read(handle, buf, COFF_FILE_HDRLEN) != COFF_FILE_HDRLEN)
/* short file is not COFF */
		return -ERR_FILE_FORMAT;
/* check if valid COFF and get info */
	if(read_le16(buf + COFF_FILE_MAGIC) != 0x014C)
		return -ERR_FILE_FORMAT;
	exec->num_sections = read_le16(buf + COFF_FILE_SECT_NUM);
/* only three sections are allowed: .text, .data, .bss */
	if(exec->num_sections != 3)
		return -ERR_FILE_FORMAT;
	aout_hdr_size = read_le16(buf + COFF_FILE_OPTHDR_SIZE);
	if((read_le16(buf + COFF_FILE_FLAGS) & 0x0002) == 0) /* F_EXEC */
		return -ERR_FILE_FORMAT;
	if(read_le16(buf + COFF_AOUT_MAGIC) != 0x010B)
		return -ERR_FILE_FORMAT;
	exec->entry_pt = read_le32(buf + COFF_AOUT_ENTRY);
	exec->entry_pt += image_base;
/* for Win32 PE COFF; must read BSS size from file hdr, not section hdr */
	bss_size = read_le32(buf + COFF_AOUT_BSS_SIZE);
	sect = exec->section;
/* lseek to first section header */
	lseek(handle, rdsk_offset + 20 + aout_hdr_size, SEEK_SET);
	for(i = 0; i < exec->num_sections; i++)
	{
/* read section header */
		err = read(handle, buf, COFF_SECT_HDRLEN);
		if(err != COFF_SECT_HDRLEN)
			return -EIO;
/* code (STYP_TEXT) */
		if(!memcmp(buf + COFF_SECT_NAME, ".text", 5) &&
			(buf[COFF_SECT_FLAGS] & 0xE0) == 0x20)
		{
			sect->load = sect->read = sect->exec = 1;
			sect->zero = sect->write = 0;
			goto FOO;
		}
/* data (STYP_DATA) */
		else if(!memcmp(buf + COFF_SECT_NAME, ".data", 5) &&
			(buf[COFF_SECT_FLAGS] & 0xE0) == 0x40)
		{
			sect->load = sect->read = sect->write = 1;
			sect->zero = sect->exec = 0;
FOO:
			sect->off = read_le32(buf + COFF_SECT_OFF);
			sect->size = read_le32(buf + COFF_SECT_SIZE);
			goto BAR;
		}
/* BSS (STYP_BSS) */
		else if(!memcmp(buf + COFF_SECT_NAME, ".bss", 5) &&
			(buf[COFF_SECT_FLAGS] & 0xE0) == 0x80)
		{
			sect->zero = sect->read = sect->write = 1;
			sect->load = sect->exec = 0;
			sect->off = 0;
			sect->size = bss_size;
//  printf("\x1B[%u;%uH", 23, 0);
//  printf("sect->size=%lX, bss_size=%lX, sect->phys_adr=%lX\n",
//	read_le32(buf + COFF_SECT_SIZE), bss_size,
//	read_le32(buf + COFF_SECT_PHYS_ADR));
BAR:
			sect->adr = read_le32(buf + COFF_SECT_VIRT_ADR);
			sect->adr += image_base;
			strcpy(sect->name, buf + COFF_SECT_NAME);
		}
/* anything else, e.g.	.idata	.rsrc	.reloc */
		else
			return -ERR_FILE_FORMAT;
		sect++;
	}
	exec->format_name = format_name;
	exec->pmode = 1;
	return 0;
}
