/*****************************************************************************
BING BOOTLOADER
*****************************************************************************/
#include <string.h> /* memset() */
#include <errno.h>
#include <conio.h> /* outp() */
#include <dos.h> /* struct REGPACK (or union REGPACK), intr() */
#include <io.h> /* open(), lseek(), read(), close() */
#include "execfile.h"
#include "defs.h"

#define	min(X,Y)	(((X) < (Y)) ? (X) : (Y))

/* IMPORTS
from BROWSE.C */
int browse_files(char *cwd_path, char *selection_path);

/* from CONIO.C */
int printf(const char *fmt, ...);
int getch(void);

/* from DISKIO.C */
int open_drive(drv_t *drv, unsigned char dev_num, signed char part_num);

/* from MOUNTS.C */
int mount(drv_t *drv, const char *mount_pt);

/* from LIB.ASM */
extern char _dos, _cpu32, _xms, _v86, _vcpi;
extern unsigned char far *_conv_mem_adr;
extern unsigned long _ext_mem_adr, _ext_mem_size, _conv_mem_size;

void enter_pmode(unsigned long eip);
#if defined(__WATCOMC__)
#pragma aux enter_pmode "_*"	/* leading underscore; not trailing */
#endif

/* from STARTUP.ASM */
extern unsigned char _end[];

/* from EXE.C, ELF.C, COFF.C, PE.C */
int check_exe_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_file_offset, unsigned long image_base);
int check_elf_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_file_offset, unsigned long image_base);
int check_coff_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, unsigned long image_base);
int check_pe_header(exec_t *exec, unsigned handle,
		unsigned long rdsk_file_offset, unsigned long image_base);

/* Check Header Function
typedef int (*chf_t)(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, ...);

typedef int (*chf_t)(); */

typedef int (*chf_t)(exec_t *exec, unsigned handle,
		unsigned long rdsk_offset, unsigned long image_base);
/*****************************************************************************
*****************************************************************************/
static void find_extremes(exec_t *exec, unsigned long *lowest,
		unsigned long *highest)
{
	unsigned short temp;
	unsigned long high;

	*lowest = -1uL;
	*highest = 0uL;
	for(temp = 0; temp < exec->num_sections; temp++)
	{
		if(exec->section[temp].adr < *lowest)
			*lowest = exec->section[temp].adr;
		high = exec->section[temp].adr + exec->section[temp].size;
		if(high > *highest)
			*highest = high;
	}
}
/*****************************************************************************
xxx - use XMS copy if available
xxx - support dst < 1 meg
*****************************************************************************/
static int copy_mem(unsigned long dst, unsigned char *src,
		unsigned count)
{
/* global descriptor table, from Table 00499 of Ralf Brown's list
make it static so it's in DS instead of SS */
	static unsigned char gdt[] =
	{
/* 0 */		0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS */
/* 8 */		0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS */
/* page-granular 32-bit data segments with limit 4 Gbytes - 1 */
/* 10h */	0xFF, 0xFF, 0, 0, 0, 0x93, 0xCF, 0,/* src seg */
/* 18h */	0xFF, 0xFF, 0, 0, 0, 0x93, 0xCF, 0,/* dst seg */
/* 20h */	0,    0,    0, 0, 0, 0,    0,    0,/* used by BIOS for CS */
/* 28h */	0,    0,    0, 0, 0, 0,    0,    0/* used by BIOS for SS */
	};
	unsigned long src_adr;
	regs_t regs;

/* fill in src segment descriptor */
	src_adr = FP_SEG(gdt);//_DS;
	src_adr <<= 4;
	src_adr += (unsigned)src;
	gdt[0x12] = src_adr;
	src_adr >>= 8;
	gdt[0x13] = src_adr;
	src_adr >>= 8;
	gdt[0x14] = src_adr;
	src_adr >>= 8;
	gdt[0x17] = src_adr;
/* fill in dst segment descriptor */
	gdt[0x1A] = dst;
	dst >>= 8;
	gdt[0x1B] = dst;
	dst >>= 8;
	gdt[0x1C] = dst;
	dst >>= 8;
	gdt[0x1F] = dst;
/* call INT 15h AH=87h to copy */
	regs.R_AX = 0x8700;
	count >>= 1;	/* words! */
	regs.R_CX = count;
	regs.R_ES = FP_SEG(src_adr);//_SS;/* ES:SI -> GDT */
	regs.R_SI = (unsigned)gdt;
	intr(0x15, &regs);
/* return status byte from AH
Borland compiler bug strikes again! put shifts on their own line of code
http://www.execpc.com/~geezer/embed/bugs.htm
	return regs.R_AX >> 8; */
	regs.R_AX >>= 8;
	if(regs.R_AX != 0)
	{
		printf("error 0x%X in copy_extended_memory\n",
			regs.R_AX);
		return -1;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
#define	BUF_SIZE	4096

static int load_kernel(exec_t *exec, unsigned handle,
		unsigned long virt_to_phys)
{
	static unsigned char buf[BUF_SIZE];
/**/
	unsigned long adr, size;
	off_t err, count = 0;
	unsigned temp;
	sect_t *sect;

/* for each section: */
	sect = exec->section + 0;
	for(temp = 0; temp < exec->num_sections; temp++)
	{
/* BSS */
		if(sect->zero)
		{
unsigned first_page = 1;
			memset(buf, 0, BUF_SIZE);
/* xxx
poke Cosmos-compatible data into BSS */
*(unsigned long *)(buf + 0) = _conv_mem_size;
*(unsigned long *)(buf + 4) = _ext_mem_size;
/* RDSK adr and size */
*(unsigned long *)(buf + 8) = 0;
*(unsigned long *)(buf + 12) = 0;
			adr = sect->adr;
			for(size = sect->size; size != 0; size -= count)
			{
				count = min(size, BUF_SIZE);
				err = copy_mem(adr + virt_to_phys,
					buf, count);
				if(err != 0)
					return (int)err;
if(first_page)
{
	memset(buf, 0, BUF_SIZE);
	first_page = 0;
}
				adr += count;
			}
		}
/* non-BSS section */
		else
		{
			lseek(handle, sect->off, SEEK_SET);
			adr = sect->adr;
			for(size = sect->size; size != 0; size -= count)
			{
				count = min(size, BUF_SIZE);
				err = read(handle, buf, count);
				if(err < 0)
					return (int)err;
				else if(err != count)
					return -1;
				err = copy_mem(adr + virt_to_phys,
					buf, count);
				if(err != 0)
					return (int)err;
				adr += count;
			}
		}
		sect++;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int check_kernel(const char *path)
{
	const chf_t check_header[] =
	{
		check_exe_header, check_elf_header,
		check_coff_header, check_pe_header
	};
/**/
	unsigned long lowest, highest, phys, size, temp, virt_to_phys;
	void far (*real_phys)(void);
	unsigned i, csr_y;
	int handle, err;
	sect_t *sect;
	exec_t exec;

/* open kernel file */
	handle = open(path, 0);
	if(handle < 0)
	{
		printf("\x1B[%u;%uH""\x1B[41;33;1m"
			"open() returned %d\n", MSG_Y, MSG_X, handle);
		getch();
		return -ENOENT;
	}
/* check if RDSK format
	memset(&exec, 0, sizeof(exec_t));
	err = check_rdsk_header(&exec, handle);
	if(err == -ERR_FILE_FORMAT) */
	{
/* check non-RDSK file formats */
		memset(&exec, 0, sizeof(exec_t));
		i = 0;
		for(; i < sizeof(check_header) / sizeof(check_header[0]); i++)
		{
/* CROCK ALERT: you need the 'L' after the second 0 */
			err = (check_header[i])(&exec, handle, 0, 0L);
			if(err != -ERR_FILE_FORMAT)
				break;
		}
	}
	if(err == -ERR_FILE_FORMAT)
	{
		printf("\x1B[%u;%uH""\x1B[41;33;1m"
			"Unknown file format", MSG_Y, MSG_X);
		close(handle);
		getch();
		return -ERR_FILE_FORMAT;
	}
	else if(err != 0)
	{
		printf("\x1B[%u;%uH""\x1B[41;33;1m"
			"check_header() returned %d", MSG_Y, MSG_X, err);
		close(handle);
		getch();
		return -ERR_FILE_FORMAT;
	}
/* print some info */
	csr_y = INFO_Y;
	printf("\x1B[%u;%uH""\x1B[30;47;1m", csr_y++, INFO_X);
	printf("File format:  %s", exec.format_name);

	printf("\x1B[%u;%uH", csr_y++, INFO_X);
	printf("Enable pmode: %s", exec.pmode ? "yes" : "no");

	printf("\x1B[%u;%uH", csr_y++, INFO_X);
/* display this in bright white on black */
	printf("System data:  ""\x1B[37;40;1m");
	printf("%-3s""\x1B[30;47;1m",
//		exec.rdsk ? "in BSS (Cosmos)" : "Multiboot");
		"in BSS (Cosmos)");
/* usually, the .text section is at the lowest address,
so it's the virtual load (start) address */
	find_extremes(&exec, &lowest, &highest);
	printf("\x1B[%u;%uH", csr_y++, INFO_X);
	printf("Virtual load address:   %8lX", lowest);

	printf("\x1B[%u;%uH", csr_y++, INFO_X);
	printf("Virtual entry point:    %8lX", exec.entry_pt);

/* maybe the sections are not contiguous (i.e. there are gaps
between them), so the extent of the kernel is not neccessarily
the sum of the section sizes */
	printf("\x1B[%u;%uH", csr_y++, INFO_X);
/* convert highest to extent (size) */
	highest -= lowest;
	printf("Kernel extent (size):   %8lX", highest);

/* figure out load address */
	if(exec.pmode)
	{
/* try to load pmode kernel in extended memory */
		if(highest <= _ext_mem_size)
		{
			size = _ext_mem_size;
			phys = _ext_mem_adr;
			temp = phys + highest - 1;
			temp ^= phys;
			temp &= 0xFFC00000L;
			if(temp != 0)
				printf("\x1B[%u;%uH""\x1B[41;33;1m"
				"Warning: kernel straddles 4 meg line",
				MSG_Y + 1, MSG_X);
		}
		else
		{
/* convert conventional memory base adr from far ptr to linear address */
			phys = FP_SEG(_conv_mem_adr);
			phys <<= 4;
			phys += FP_OFF(_conv_mem_adr);
/* align to 4K boundary */
			temp = phys + 4095;
			temp &= -4096uL;
/* reduce size due to 4K alignment */
			size = _conv_mem_size - (temp - phys);
			phys = temp;
		}
	}
/* load real-mode kernel in conventional memory */
	else
	{
		phys = FP_SEG(_conv_mem_adr);
		phys <<= 4;
		phys += FP_OFF(_conv_mem_adr);
		size = _conv_mem_size;
	}
	printf("\x1B[%u;%uH", csr_y++, INFO_X);
	printf("Physical load address:  ");
/* display this in bright white on black */
	printf("\x1B[37;40;1m""%8.8lX""\x1B[30;47;1m", phys);
/* display section info */
	csr_y++;
	if(exec.num_sections != 0)
	{
		printf("\x1B[%u;%uH", csr_y++, INFO_X);
		printf("Section/segment info");
		printf("\x1B[%u;%uH", csr_y++, INFO_X);
		printf("name    address  offset   size     RWX");
		printf("\x1B[%u;%uH", csr_y++, INFO_X);
		printf("------- -------- -------- -------- ---");
	}
	sect = exec.section + 0;
	for(i = 0; i < exec.num_sections; i++)
	{
		printf("\x1B[%u;%uH", csr_y + i, INFO_X);
		printf("%-7.7s %8lX %8lX %8lX ",
			sect->name, sect->adr, sect->off, sect->size);
		printf("%c%c%c", sect->read ? 'R' : '-',
			sect->write ? 'W' : '-', sect->exec ? 'X' : '-');
		sect++;
	}
/* can we boot? */
	printf("\x1B[%u;%uH""\x1B[41;33;1m", MSG_Y, MSG_X);
	if(highest > size)
	{
		printf("Not enough memory");
		close(handle);
		getch();
		return -ENOMEM;
	}
/* check if we can boot pmode kernel */
	if(exec.pmode)
	{
		if(!_cpu32)
		{
			printf("32-bit CPU required");
			close(handle);
			getch();
			return -1;
		}
		if(_v86 && !_vcpi)
		{
			printf("Can't boot pmode OS from Windows");
			close(handle);
			getch();
			return -1;
		}
	}
	else
	{
		printf("Real-mode kernels not yet supported");
		close(handle);
		getch();
		return -ENOIMP;
	}
	printf("\x1B[47;30;0m""\x1B[%u;%uH"
		"Press Enter to boot", MSG_Y, MSG_X);
	if(getch() != 13)
	{
		close(handle);
		return -1;
	}
	if(exec.pmode)
	{
/* convert lowest virtual address to virt_to_phys */
		virt_to_phys = phys - lowest;
/* load kernel */
		err = load_kernel(&exec, handle, virt_to_phys);
		if(err != 0)
		{
			printf("\x1B[%u;%uH""\x1B[41;33;1m"
			"load_kernel() returned %d\n", MSG_Y, MSG_X, err);
			close(handle);
			getch();
			return err;
		}
/* turn off floppy motor */
		outp(0x3F2, 0);
/* jump to kernel */
		enter_pmode(exec.entry_pt + virt_to_phys);
		printf("\x1B[%u;%uH""\x1B[41;33;1m"
			"enter_pmode returned (A20 trouble?)\n",
			MSG_Y, MSG_X);
	}
	else
	{
		/* xxx - load real mode kernel (do .EXE relocations too) */
		real_phys();	/* run it */
	}
	close(handle);
	getch();
	return -1;
}
/*****************************************************************************
*****************************************************************************/
int main(void)
{
	char cwd_path[MAX_NAME_LEN] = "/";
	char selection_path[MAX_NAME_LEN];
	drv_t drv;
	int err;

// xxx - let user select drive
/* 2nd partition (1) on 1st hard drive (0x80) == /dev/hda2 */
//	err = open_drive(&drv, 0x80, 1);
	err = open_drive(&drv, 0, 0);
	if(err != 0)
		return 1;
	err = mount(&drv, "/");
	if(err != 0)
		return 1;
	do
	{
/* set black on white color scheme and clear screen */
		printf("\x1B[47;30;0m" "\x1B[2J");
		printf("BING bootloader version 0.5 - "
			"http://www.execpc.com/~geezer/os\n");
		printf("%luK conventional memory at %Fp, ",
			_conv_mem_size >> 10, _conv_mem_adr);
		printf("%luK %s memory at %lX\n", _ext_mem_size >> 10,
			_xms ? "locked XMS" : "extended", _ext_mem_adr);
		printf("32-bit CPU:%s", _cpu32 ? "yes" : "no");
		printf("  DOS:%s", _dos ? "yes" : "no");
		printf("   V86 mode:%s", _v86 ? "yes" : "no");
		printf("   VCPI:%s", _vcpi ? "yes" : "no");
		printf("   LBA:%s", drv.use_lba ? "yes" : "no");
/* */
		printf("\nUse the arrow keys to select a kernel file, "
			"then press Enter");
		err = browse_files(cwd_path, selection_path);
		if(err < 0)
			break;
		err = check_kernel(selection_path);
	} while(err != 0);
	return 0;
}
