/*****************************************************************************
- dd_write() should do the Right Thing when write
	ends in the middle of a block

>>>- testing
	Turbo C 2.0, Turbo C 3.0, DJGPP
	486 system, Pentium system
	plain DOS, DOS w/ SMARTDRV, Windows DOS box
	1.44 meg disks, 1.68 meg disks
	random things: initial offset, whole middle blocks, terminal offset
>>>- faster?

later:
- need zero-fill if reading beyond end of disk?
- support hard drive (entire drive and partitions)
*****************************************************************************/
#include <stdlib.h> /* atol(), malloc(), free() */
#include <string.h> /* NULL, movedata(), memset(), strlen(), memicmp() */
#include <ctype.h> /* isalpha(), toupper(), tolower() */
#include <stdio.h> /* printf() */
#include <fcntl.h> /* O_nnn */
#include <bios.h> /* _DISK_..., biosdisk() */
#include <dos.h> /* union REGS, int86(), peek(), poke(), pokeb() */
#include <io.h> /* open(), lseek(), tell(), read(), write(), close() */

#if defined(__DJGPP__)
#include <unistd.h> /* all of the io.h functions except tell() */
#include <sys/farptr.h> /* _far...() */
#include <dpmi.h> /* __dpmi... */
#include <go32.h> /* _dos_ds */

/* the nice thing about standards is... */
#define	to_linear(S,O)	(((unsigned long)(S) << 4) + O)
#define	peek(S,O)	_farpeekw(_dos_ds, to_linear(S,O))
#define	poke(S,O,V)	_farpokew(_dos_ds, to_linear(S,O), V)
#define	pokeb(S,O,V)	_farpokeb(_dos_ds, to_linear(S,O), V)
#define	my_movedata(SS,SO,DS,DO,C) 				\
			movedata(_dos_ds, to_linear(SS,SO),	\
				_dos_ds, to_linear(DS,DO), C)
#define	dos_int(N,R)	__dpmi_int(N,R)

typedef __dpmi_regs regs_t;

#elif defined(__TURBOC__)
#define	my_movedata(SS,SO,DS,DO,C)	\
			movedata(SS,SO,DS,DO,C)
#define	dos_int(N,R)	int86(N,R,R)

typedef union REGS regs_t;

#else
#error Not DJGPP, not Turbo C. Sorry.
#endif

#if 0
#define	DEBUG(X)	X
#else
#define	DEBUG(X)
#endif

#define	BUFLEN	8192

#define	false	0
#define	true	1

typedef unsigned char	bool;

typedef struct
{
	unsigned is_device : 1;
	unsigned needs_fill : 1;
	unsigned needs_flush : 1;
	unsigned long size;
/* read/write position */
	unsigned short curr_blk, curr_off;
/* block buffer */
	unsigned short buf_size;/* max number of bytes in buffer */
	unsigned char *buf;	/* base address of buffer */
/* used if input/output is a regular file */
	int handle;
/* use if input/output is a block device */
	unsigned char int13_dev;
	unsigned short cyls, sects, heads, bps;
} file_t;

static const char *_usage =
"\n"
"dd for DOS, version 0.3     http://www.execpc.com/~geezer/os\n"
"Performs low-level byte-wise copying between files and floppies.\n"
"Usage:\n"
"          dd count=... skip=... seek=... if=... of=... probe\n"
"count=number of bytes to copy           (default: size of input)\n"
"skip =number of input bytes to skip     (default: 0)\n"
"seek =number of output bytes to skip    (default: 0)\n"
"if   =input file or drive\n"
"of   =output file or drive\n"
"  Examples:\n"
"    dd count=512 if=boot.bin of=a:      (install bootloader)\n"
"    dd  skip=512 if=boot.bin of=a:\\loader.bin\n"
"    dd skip=18K count=18K if=a: of=d:t1 (copy 2nd track of 1.44M disk)\n"
"    dd count=5k if=a: of=c:save         (backup boot sector and 1st FAT)\n"
"    dd skip=100 count=100 if=foo of=bar (pick out second 100 bytes of file)\n"
"    dd if=image of=b:                   (RAWRITE)\n"
"    dd if=a: of=c:\\tmp\\image.bin        (RAWREAD, DSKIMG)\n"
"    dd if=a: of=b:                      (DISKCOPY)\n"
"Use 'probe' for unusual floppy sizes such as 1.68 meg\n";

static bool _probe_floppy, _changed_fpt;
static unsigned short _old_fpt_seg, _old_fpt_off;
/*////////////////////////////////////////////////////////////////////////////
	PROBING DISK GEOMETRY
////////////////////////////////////////////////////////////////////////////*/
/*****************************************************************************
*****************************************************************************/
static void new_fpt(void)
{
	unsigned short new_fpt_seg;
	regs_t regs;

	if(_changed_fpt)
		return;
/* save old FPT */
	_old_fpt_off = peek(0, 0x1E * 4 + 0);
	_old_fpt_seg = peek(0, 0x1E * 4 + 2);
/* alloc 1 paragraph (16 bytes) DOS memory */
	regs.h.ah = 0x48;
	regs.x.bx = 1;
	dos_int(0x21, &regs);
	new_fpt_seg = regs.x.ax;
/* copy old FPT to new */
	my_movedata(_old_fpt_seg, _old_fpt_off,
		new_fpt_seg, 0,
		11);
/* set sectors-per-track to inflated value */
	pokeb(new_fpt_seg, 4, 96);
/* install new FPT */
	poke(0, 0x1E * 4 + 0, 0);
	poke(0, 0x1E * 4 + 2, new_fpt_seg);
	_changed_fpt = true;
}
/*****************************************************************************
*****************************************************************************/
static void old_fpt(void)
{
	if(!_changed_fpt)
		return;
	poke(0, 0x1E * 4 + 0, _old_fpt_off);
	poke(0, 0x1E * 4 + 2, _old_fpt_seg);
}
/*****************************************************************************
*****************************************************************************/
static int dd_get_geom(file_t *file)
{
	unsigned short temp;
	union REGS regs;

/* first do INT 13h AH=08h to get number of drives */
	regs.h.ah = 0x08;
	if(file->int13_dev >= 0x80)
		regs.h.dl = 0x80;
	else
		regs.h.dl = 0;
	int86(0x13, &regs, &regs);
	if(regs.x.cflag)
	{
		printf("error getting drive count in dd_get_geom\n");
		return -1;
	}
	if((file->int13_dev & 0x7F) >= regs.h.dl)
	{
		printf("error: no such drive in dd_get_geom\n");
		return -1;
	}
	temp = regs.h.cl;
	temp <<= 2;
	temp |= regs.h.ch;

	file->bps = 512;
	file->cyls = temp + 1;
	file->heads = regs.h.dh + 1;
/* hard disk: use the sectors-per-track returned by INT 13h AH=08h */
	if(file->int13_dev >= 0x80)
		file->sects = regs.h.cl & 0x3F;
/* floppy disk, no probe: use sectors-per-track returned by INT 13h AH=08h */
	else if(!_probe_floppy)
		file->sects = regs.h.cl & 0x3F;
/* floppy disk, probed geometry. Use this for non-standard floppy sizes
like 1.68 meg (21 sectors per track), since INT 13h AH=08h is unreliable
in this case. */
	else
	{
		unsigned short sects, delta, old_err;
/* NOTE: 'try' is a C++ reserved word */
		unsigned char try, buffer[512];

/* install new floppy parameter table */
		new_fpt();
		printf("probing floppy geometry; please wait...\n");
AGAIN:
		delta = sects = 16;
		do
		{
			delta >>= 1;
			DEBUG(printf("sects=%d, delta=%d\n", sects, delta);)
			old_err = 0;
/* The floppy drive is easily confused by this probing.
Try each read up to 3 times before giving up. */
			for(try = 3; try != 0; try--)
			{
/* *** WARNING, WARNING -- biosdisk() doesn't return what
Turbo C online help says it returns. It returns the INT 13h
error value only, not the number of sectors moved. */
				temp = biosdisk(_DISK_READ, file->int13_dev,
					0/*head*/, 0/*track*/,
					sects, 1/*nsects*/, buffer);
				if(temp == 6)
				{
					DEBUG(printf("disk change, "
						"restarting...\n");)
					goto AGAIN;
				}
				else if(temp == 0)
				{
					sects += delta;
					break;
				}
				DEBUG(printf("\terror 0x%X\n", temp);)
				(void)biosdisk(_DISK_RESET, file->int13_dev,
					0, 0, 0, 0, 0);
/* don't decrement sects more than once */
				if(old_err == 0)
					sects -= delta;
				old_err = temp;
			}
		} while(delta != 0);
/* no, do this when the program exits
		old_fpt(); */
		(void)biosdisk(_DISK_RESET, file->int13_dev, 0, 0, 0, 0, 0);
		if(temp != 0)
			sects--;

		file->sects = sects;
	}
	printf("CHS=%u:%u:%u\n", file->cyls, file->heads, file->sects);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int dd_rwdisk(file_t *file, int cmd, unsigned long lba,
		unsigned short nsects, unsigned char *buf)
{
	unsigned short cyl, head, sect;
	unsigned char try;
	int err;

	if(nsects == 0)
		return 0;
	if(lba * file->bps >= file->size)
	{
		printf("error: access beyond end of disk in dd_rwdisk\n");
		return -1;
	}
/* convert to CHS */
	sect = lba % file->sects + 1;
	lba /= file->sects;
	head = lba % file->heads;
	cyl = lba / file->heads;
	for(try = 3; try != 0; try--)
	{
		DEBUG(printf("dd_rwdisk: CHS=%u:%u:%u\n", cyl, head, sect);)
		err = biosdisk(cmd, file->int13_dev, head, cyl, sect,
			nsects, buf);
		if(err == 0 || err == 0x11)
			return 0;
		DEBUG(printf("disk error 0x%X in dd_rwdisk\n", err);)
	}
	printf("disk error 0x%X in dd_rwdisk\n", err);
	return -1;
}
/*////////////////////////////////////////////////////////////////////////////
	INPUT
////////////////////////////////////////////////////////////////////////////*/
/*****************************************************************************
*****************************************************************************/
static int dd_fill(file_t *file)
{
	int temp;

	if(!file->needs_fill)
		return 0;
	DEBUG(printf("dd_fill: block %u\n", file->curr_blk);)
/* read block from regular file */
	if(!file->is_device)
	{
		lseek(file->handle, (unsigned long)file->curr_blk *
			file->buf_size, SEEK_SET);
		temp = read(file->handle, file->buf, file->buf_size);
		if(temp < 0)
		{
			printf("error reading file in dd_fill\n");
			return -1;
		}
/* zero-fill if short read */
		else if(temp < file->buf_size)
			memset(file->buf + temp, 0, file->buf_size - temp);
	}
/* read block from disk
xxx - do zero-fill if full or partial read beyond end of disk */
	else
	{
		temp = dd_rwdisk(file, _DISK_READ, file->curr_blk *
			file->sects, file->buf_size / file->bps, file->buf);
		if(temp != 0)
			return -1;
	}
	DEBUG(printf("dd_fill: read %lu bytes so far\n",
		(unsigned long)file->curr_blk * file->buf_size);)
	file->needs_fill = false;
	file->curr_blk++;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int dd_read(file_t *file, unsigned char *byte)
{
	int err;

/* fill the buffer if it's empty */
	if(file->needs_fill)
	{
		err = dd_fill(file);
		if(err != 0)
			return -1;
		file->curr_off = 0;
	}
/* read char from buffer */
	*byte = file->buf[file->curr_off];
	file->curr_off++;
	if(file->curr_off >= file->buf_size)
		file->needs_fill = true;
	return 0;
}
/*////////////////////////////////////////////////////////////////////////////
	OUTPUT
////////////////////////////////////////////////////////////////////////////*/
/*****************************************************************************
*****************************************************************************/
static int dd_flush(file_t *file)
{
	int temp;

	if(!file->needs_flush)
		return 0;
	DEBUG(printf("dd_flush: block %u\n", file->curr_blk);)
/* write block to regular file */
	if(!file->is_device)
	{
		lseek(file->handle, (unsigned long)file->curr_blk *
			file->buf_size, SEEK_SET);
		temp = write(file->handle, file->buf, file->curr_off);
		if(temp < 0)
		{
			printf("error writing file in dd_flush\n");
			return -1;
		}
	}
/* write block to disk */
	else
	{
		temp = dd_rwdisk(file, _DISK_WRITE, file->curr_blk *
			file->sects, file->buf_size / file->bps, file->buf);
		if(temp != 0)
			return -1;
	}
	DEBUG(printf("dd_flush: wrote %lu bytes so far\n",
		(unsigned long)file->curr_blk * file->buf_size);)
	file->needs_flush = 0;
	file->curr_blk++;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static int dd_write(file_t *file, unsigned char byte)
{
	int err;

	file->buf[file->curr_off] = byte;
	file->curr_off++;
	file->needs_flush = true;
	if(file->curr_off < file->buf_size)
		return 0;
	err = dd_flush(file);
	file->curr_off = 0;
	return err;
}
/*////////////////////////////////////////////////////////////////////////////
	MAIN
////////////////////////////////////////////////////////////////////////////*/
/*****************************************************************************
*****************************************************************************/
static int dd_open(file_t *file, char *name, bool output)
{
	unsigned long pos;
	int temp;

	memset(file, 0, sizeof(file_t));
/* opening a disk drive? */
	if(isalpha(name[0]) && (name[1] == ':') && (name[2] == '\0'))
	{
/* compute INT 13h-style drive number */
		temp = toupper(name[0]);
		if(temp >= 'C')
			file->int13_dev = 0x80 + temp - 'C';
		else
			file->int13_dev = temp - 'A';
/* check if drive exists, and get disk geometry */
		if(dd_get_geom(file) != 0)
			return -1;
/* set buf_size equal to one track, for faster floppy access */
		file->buf_size = file->bps * file->sects;
		file->is_device = true;
	}
/* else it's a regular file */
	else
	{
		if(output)
			temp = open(name, O_BINARY | O_RDWR | O_CREAT);
		else
			temp = open(name, O_BINARY | O_RDONLY);
		if(temp < 0)
		{
			printf("error: could not open file '%s' "
				"in dd_open\n", name);
			return -1;
		}
		file->handle = temp;
		file->buf_size = BUFLEN;
	}
/* allocate block buffer
xxx - do this only for devices? */
	file->buf = malloc(file->buf_size);
	if(file->buf == NULL)
	{
		printf("error: could not allocate %u "
			"bytes memory in dd_open\n", file->buf_size);
		return -1;
	}
/* compute size of file/device */
	if(file->is_device)
		file->size = (unsigned long)file->cyls * file->sects *
			file->heads * file->bps;
	else
	{
		pos = tell(file->handle);
		lseek(file->handle, 0, SEEK_END);
		file->size = tell(file->handle);
		lseek(file->handle, pos, SEEK_SET);
	}
	if(!output)
		file->needs_fill = true;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static void dd_close(file_t *file)
{
	if(!file->is_device)
		close(file->handle);
	if(file->buf != NULL)
		free(file->buf);
	memset(file, 0, sizeof(file_t));
}
/*****************************************************************************
*****************************************************************************/
int main(int arg_c, char *arg_v[])
{
	unsigned long skip = 0, seek = 0, count = 0;
	char *iname = NULL, *oname = NULL;
	unsigned char one_byte;
	file_t in, out;
	int temp;
	bool k;

/* usage */
	if(arg_c < 2)
	{
		printf(_usage);
		return 1;
	}
/* get command line options */
	for(temp = 1; temp < arg_c; temp++)
	{
		if(tolower(arg_v[temp][strlen(arg_v[temp]) - 1]) == 'k')
			k = true;
		else
			k = false;
		if(!memicmp(arg_v[temp], "if=", 3))
			iname = arg_v[temp] + 3;
		else if(!memicmp(arg_v[temp], "of=", 3))
			oname = arg_v[temp] + 3;
		else if(!memicmp(arg_v[temp], "skip=", 5))
		{
			skip = atol(arg_v[temp] + 5);
			if(k)
				skip *= 1024;
		}
		else if(!memicmp(arg_v[temp], "seek=", 5))
		{
			seek = atol(arg_v[temp] + 5);
			if(k)
				seek *= 1024;
		}
		else if(!memicmp(arg_v[temp], "count=", 6))
		{
			count = atol(arg_v[temp] + 6);
			if(k)
				count *= 1024;
		}
		else if(!memicmp(arg_v[temp], "probe", 5))
		{
#if defined(__DJGPP__)
/* see file SRC/LIBC/BIOS/BIOSDISK.C of DJLSR203.ZIP */
			printf("WARNING: DJGPP biosdisk() can't handle "
				">18 sectors/track,\nso 'probe' "
				"may not work\n");
#endif
			_probe_floppy = true;
		}
	}
/* open files */
	if(iname == NULL)
	{
		printf("error: no input file/device specified\n");
		return 1;
	}
	if(oname == NULL)
	{
		printf("error: no output file/device specified\n");
		return 1;
	}
	temp = dd_open(&in, iname, false);
	if(temp < 0)
		return 2;
	temp = dd_open(&out, oname, true);
	if(temp < 0)
		return 2;
	in.curr_off = skip % in.buf_size;
	in.curr_blk = skip / in.buf_size;

	out.curr_off = seek % out.buf_size;
/* write starts in the middle of a block? */
	if(out.curr_off != 0)
	{
		out.needs_fill = true;
		if(dd_fill(&out) != 0)
			goto ERR;
	}
	out.curr_blk = seek / out.buf_size;
	out.curr_off = seek % out.buf_size;
/* if count not given, use size of input */
	if(count == 0)
		count = in.size;
/* copy! */
	for(; count != 0; count--)
	{
		if(dd_read(&in, &one_byte) != 0)
			break;
		if(dd_write(&out, one_byte) != 0)
			break;
	}
	dd_flush(&out);
ERR:
	dd_close(&out);
	dd_close(&in);
	old_fpt();
	return 0;
}
