;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; load.asm - load and run kernel inside Mbius-format initial RAM disk
; Copyright (C) 2000 - Christopher Giese <geezer@execpc.com>
; http://www.execpc.com/~geezer/os/
;
; Tim Robinson's Mbius OS: http://www.gaat.freeserve.co.uk/
;
; Assemble this file with NASM:
;	nasm -f bin -o load.com load.asm
; put your kernel into a Mobius initial RAM disk:
;	rdsk -o krnl.dsk krnl.x <other_files_here>
; then boot it:
;	load krnl.dsk
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	BITS 16
	ORG 100h		; .COM file

	mov ax,cs
	mov ds,ax
	mov ss,ax
	mov es,ax
	mov [real_mode_cs],ax
	mov sp,stack		; move the stack somewhere safe

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; before using 32-bit instructions, check for 32-bit CPU (386SX or better)
; I (Chris Giese) have not tested this code with 8088/'286 systems
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	pushf
		pushf
		pop bx		; old FLAGS -> BX
		mov ax,bx
		xor ah,70h	; try changing b14 (NT)...
		push ax		; ... or b13:b12 (IOPL)
		popf
		pushf
		pop ax		; new FLAGS -> AX
	popf
	xor ah,bh		; 32-bit CPU if we changed NT...
	mov si,cpu_msg
	and ah,70h		; ...or IOPL
	je near die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; OK for pmode code to use DOS stack, but zero the top 16 bits of ESP
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	xor eax,eax
	mov ax,sp
	mov esp,eax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; we'll use this value a lot:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	xor ebp,ebp
	mov bp,cs
	shl ebp,4

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; patch things that depend on the load adr
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov eax,ebp
	mov [gdt3 + 2],ax	; CODE_SEL32
	mov [gdt4 + 2],ax	; DATA_SEL32
	mov [gdt5 + 2],ax	; CODE_SEL16
	mov [gdt6 + 2],ax	; DATA_SEL16
	shr eax,16
	mov [gdt3 + 4],al
	mov [gdt4 + 4],al
	mov [gdt5 + 4],al
	mov [gdt6 + 4],al
	mov [gdt3 + 7],ah
	mov [gdt4 + 7],ah
	; mov [gdt5 + 7],ah	; no, these are 16-bit
	; mov [gdt6 + 7],ah

	mov eax,tss
	add eax,ebp
	mov [gdt7 + 2],ax	; TSS_SEL
	shr eax,16
	mov [gdt7 + 4],al
	mov [gdt7 + 7],ah

	add [gdt_ptr + 2],ebp
	add [idt_ptr + 2],ebp

	add [vcpi_gdtr],ebp
	add [vcpi_idtr],ebp

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; get memory sizes from CMOS
; xxx - use BIOS calls
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	xor eax,eax
	mov al,18h
	out 70h,al
	in al,71h
	mov ah,al
	mov al,17h
	out 70h,al
	in al,71h
	shl eax,10		; Kbytes -> bytes
	mov [ext_mem],eax

	xor eax,eax
	mov al,16h
	out 70h,al
	in al,71h
	mov ah,al
	mov al,15h
	out 70h,al
	in al,71h
	shl eax,10		; Kbytes -> bytes
	mov [conv_mem],eax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; grind the command line
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov si,80h
	lodsb
	xor ch,ch
	mov cl,al
	or cl,cl		; zero length means no command-line arg
	je usage_err
skip_spaces:
	lodsb
	cmp al,' '
	jne found_arg_start
	loop skip_spaces
usage_err:			; all spaces? how'd that happen?
	mov si,usage_msg
	jmp die

found_arg_start:
	lea dx,[si - 1]		; save ptr to arg in DX for use by OPEN
find_end:
	lodsb
	cmp al,' '
	je found_arg_end
	cmp al,0Dh
	loopne find_end
found_arg_end:
	xor al,al
	mov [si - 1],al		; convert 1st arg to ASCIZ (0-terminated)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; open the file named on the command line
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ax,3D00h		; DOS "OPEN"; for read
	int 21h
	mov si,open_msg		; CY=1 if error
	jc near die
	mov bx,ax		; BX=file handle

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; validate file format
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ah,3Fh		; DOS "READ"
	mov cx,32		; 32 bytes
	mov dx,end		; read to buffer that's just beyond this file
	int 21h
	jc read_error
	cmp ax,32
	je validate
read_error:
	mov si,read_msg
close_and_error:
	mov ah,3Eh		; DOS "CLOSE"
	int 21h
	jmp die

validate:			; Mobius-format initial RAM disk
	cmp dword [end],"RDSK"
	mov si,invalid_msg
	jne close_and_error
	cmp dword [end + 4],1	; number of files in RAM disk; must be >= 1
	jb close_and_error

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; get file size
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ax,4202h		; DOS "LSEEK"; from end of file
	xor cx,cx
	xor dx,dx
	int 21h
	mov si,seek_msg
	jc close_and_error
	mov [file_len + 0],ax
	mov [file_len + 2],dx
	mov ax,4200h		; DOS "LSEEK"; from start of file
	xor cx,cx
	xor dx,dx
	int 21h
	jc close_and_error

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; see if enough conventional memory to load initial RAM disk
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov eax,ebp
	add eax,end		; linear adr of end of this .COM file
	add eax,[file_len]	; linear adr of end of loaded RAM disk
	cmp eax,[conv_mem]	; is it still within conventional mem?
	mov si,mem_msg
	jae close_and_error

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; load it
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov dx,end
%if 0
	mov cx,(32768 - end)	; ** sigh **
%else
	mov cx,end
	neg cx
	add cx,32768
%endif
	push ds
		jmp load2
load1:
		xor dx,dx
		mov ax,ds
		add ax,(32768 / 16)
		mov ds,ax
		mov cx,32768
load2:
		mov ah,3Fh	; DOS "READ"
		int 21h
		jnc load3
	pop ds
	jmp read_error
load3:
		or ax,ax
		jne load1	; ...until EOF
	pop ds

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; close input file
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ah,3Eh		; DOS "CLOSE"
	int 21h

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; point to 1st file in initial RAM disk; see if it's DJGPP COFF
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ebx,[end + 8]
	add ebx,end		; offset of end of this .COM file

COFF_MAGIC		EQU 0	; 0x014C
COFF_NUM_SECTS		EQU 2	; number of sections
COFF_OPTHDR_SIZE	EQU 16	; "optional" (aout) header size
COFF_FLAGS		EQU 18

AOUT_MAGIC		EQU 20 + 0 ; 0x010B
AOUT_ENTRY_POINT	EQU 20 + 16

COFF_TEXT_VIRT		EQU 20 + 28 + 40 * 0 + 12
COFF_TEXT_SIZE		EQU 20 + 28 + 40 * 0 + 16
COFF_TEXT_OFFSET	EQU 20 + 28 + 40 * 0 + 20

COFF_DATA_VIRT		EQU 20 + 28 + 40 * 1 + 12
COFF_DATA_SIZE		EQU 20 + 28 + 40 * 1 + 16
COFF_DATA_OFFSET	EQU 20 + 28 + 40 * 1 + 20

COFF_BSS_VIRT		EQU 20 + 28 + 40 * 2 + 12
COFF_BSS_SIZE		EQU 20 + 28 + 40 * 2 + 16
COFF_BSS_OFFSET		EQU 20 + 28 + 40 * 2 + 20

	cmp word [bx + COFF_MAGIC],014Ch
	jne not_coff
	cmp word [bx + COFF_NUM_SECTS],3
	jne not_coff
	cmp word [bx + COFF_OPTHDR_SIZE],28
	jne not_coff
	test word [bx + COFF_FLAGS],2
	je not_coff
	cmp word [bx + AOUT_MAGIC],010Bh
	je coff_ok

; xxx - make sure all sections in same 4 meg address space
; xxx - make sure .text section doesn't overlap others

not_coff:
	mov si,coff_msg
	jmp die
coff_ok:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; get sizes of .text, .data, and .bss sections; sum is kernel size
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov edx,[bx + COFF_TEXT_SIZE]
	add edx,[bx + COFF_DATA_SIZE]
	add edx,[bx + COFF_BSS_SIZE]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; see if enough extended memory to copy kernel to 1 meg
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov edi,100000h		; normal load to 1 meg if possible
 ;	mov edi,3FF000h		; straddle 4 meg mark
 ;	mov edi,7FF000h		; straddle 8 meg mark

	cmp edx,[ext_mem]
	jb mem_ok		; comment this out to load < 1 meg

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; see if enough conventional memory to copy kernel beyond loaded file
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov edi,ebp
	add edi,end		; linear adr of end of this .COM file
	add edi,[file_len]	; linear adr of end of initial RAM disk
	add edi,4095
	and di,0F000h		; round up to page (4K) boundary
	mov eax,edi
	add eax,edx
	cmp eax,[conv_mem]
	mov si,mem_msg
	jae near close_and_error
mem_ok:
	mov [load_adr],edi

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; enable A20 line using 'AT' ('Linux') method, then test if it worked
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	cli

	call kbd
	mov al,0D1h
	out 64h,al
	call kbd
	mov al,0DFh
	out 60h,al
	call kbd

	call test_a20
	jne a20_ok

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oop, try the 'Vectra' method
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	call kbd
	mov al,0DFh
	out 64h,al
	call kbd

	mov si,a20_msg
	call test_a20
	je near die
a20_ok:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; no more errors are possible, and we're done with DOS,
; so move to unreal mode. But first, check for virtual 8086 mode:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	smsw ax
	test al,1		; look at PE bit of MSW (CR0)
	je near real

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; in V86 mode; check for VCPI (e.g. DOS with EMM386 loaded)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ax,0DE00h
	int 67h
	cmp ah,0
	mov si,v86_msg
	jne near die

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; use VCPI to switch from V86 mode to paged pmode
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov edi,ebp		; find 4K-aligned mem for page table
	add edi,(page_info + 4095)
	and di,0F000h		; EDI=linear adr of page table

	mov eax,edi
	add eax,4096		; linear adr of page dir, 4K above table
	mov [vcpi_cr3],eax

	mov eax,edi
	sub edi,ebp		; DI=offset of page table
	add di,4096		; point to page dir
	or al,7			; ring 3, writable, present
	mov [di + 0],eax	; page dir[0] -> linear adr of page table
	sub di,4096		; back to page table; VCPI will fill it

	mov si,gdt8		; get 3 VCPI descriptors to here
	mov ax,0DE01h
	int 67h
	cmp ah,0
	mov si,vcpi_err_msg
	jne near die
	push dword 0		; disable interrupts (set IF=0)...
	popfd			; ...set IOPL=0, and clear the NT bit
	mov esi,ebp
	add esi,vcpi_control_block
	mov ax,0DE0Ch		; switch from V86 mode to paged pmode
	int 67h			; jump to vcpi_to_pmode32

	BITS 32

vcpi_to_pmode32:		; now in ring 0 paged pmode
	mov eax,cr0
	and eax,7FFFFFFFh	; turn off paging
	mov cr0,eax
	xor eax,eax
	mov cr3,eax		; flush TLB
	jmp CODE_SEL16:pmode32_to_pmode16

	BITS 16

pmode32_to_pmode16:		; finish switching to 16-bit pmode
	mov ax,DATA_SEL16
	mov ss,ax
	mov ds,ax
	mov es,ax
	mov fs,ax
	mov gs,ax
	mov eax,cr0
	and al,0FEh		; pmode off
	mov cr0,eax
	jmp far [real_mode_ip]

pmode16_to_real:		; finish switching to real mode
	lidt [real_idt_ptr]
	mov ax,cs
	mov ss,ax
	mov ds,ax
	mov es,ax
	mov fs,ax
	mov gs,ax
real:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; enable unreal mode
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	cli
	push ds
		lgdt [gdt_ptr]
		mov eax,cr0
		or al,1
		mov cr0,eax	; partial switch to pmode
		mov ax,LINEAR_DATA_SEL
		mov ds,ax	; selector to segment w/ 4G limit
		mov es,ax	; set seg limits in descriptor caches
		mov fs,ax
		mov gs,ax	; (do NOT change CS nor SS)
		mov eax,cr0
		and al,0FEh
		mov cr0,eax	; back to (un)real mode
		nop
	pop ds			; segment regs back to old values,
	xor ax,ax		; but now 32-bit addresses are OK
	mov es,ax
	mov fs,ax
	mov gs,ax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; get virt-to-phys conversion value to EDX
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ebx,end		; offset of end of this .COM file
	add ebx,[end + 8]	; offset of DJGPP kernel in initial RAM disk

	mov edi,[load_adr]	; physical (load) adr of .text
	mov edx,edi
	sub edx,[bx + COFF_TEXT_VIRT] ; virtual adr of .text

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; copy kernel .text and .data
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ecx,[bx + COFF_TEXT_SIZE]
	mov esi,[bx + COFF_TEXT_OFFSET]
	add esi,ebx
	mov edi,[bx + COFF_TEXT_VIRT]
	add edi,edx
	a32
	rep movsb

	mov ecx,[bx + COFF_DATA_SIZE]
	mov esi,[bx + COFF_DATA_OFFSET]
	add esi,ebx
	mov edi,[bx + COFF_DATA_VIRT]
	add edi,edx
	a32
	rep movsb

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; zero kernel .bss
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov ecx,[bx + COFF_BSS_SIZE]
	mov edi,[bx + COFF_BSS_VIRT]
	add edi,edx
	xor eax,eax
	push edi
		a32
		rep stosb
	pop edi

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; store machine state in kernel BSS and registers:
; BSS + 0	conventional memory size (bytes)
; BSS + 4	extended memory size (bytes)
; BSS + 8	linear address of initial RAM disk (RDSK file)
; BSS + 12	size of initial RAM disk file
; BSS + 16	kernel command line
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	mov eax,[conv_mem]	; conventional memory size
	mov [es:edi + 0],eax

	mov eax,[ext_mem]	; extended memory size
	mov [es:edi + 4],eax

	mov eax,end
	add eax,ebp
	mov [es:edi + 8],eax	; initial RAM disk address

	mov eax,[file_len]	; RDSK file length
	mov [es:edi + 12],eax

	xor eax,eax
	mov [es:edi + 16],eax	; kernel command line (xxx - implement)

	mov eax,[bx + AOUT_ENTRY_POINT]	; virtual entry point
	add eax,edx			; physical entry point

; stash the entry point. This is self-modifying code,
; but it's the only way I can see of jumping to the kernel
; with all of the data segment registers already loaded
	mov [entry + 2],eax

; interrupts off
	push dword 0
	popfd

; convert SS:SP to ESP
	xor eax,eax
	mov ax,ss
	shl eax,4
	xor ebx,ebx
	mov bx,sp
	add eax,ebx
	mov esp,eax

; set PE bit
	mov ebx,cr0
	inc bx
	mov cr0,ebx

; load data segment registers
	mov eax,LINEAR_DATA_SEL
	mov ds,eax
	mov ss,eax
	mov es,eax
	mov fs,eax
	mov gs,eax
entry:
; 2-byte JMP at [entry+0]:	66 EA
; 4-byte offset at [entry+2]:	00 00 00 00
; 2-byte selector at [entry+6]:	08 00
	jmp LINEAR_CODE_SEL:dword 0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; exit to DOS with error code and message
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

die:
	mov ah,0Eh		; INT 10h: teletype output
	xor bx,bx		; video page 0
	jmp die3
die2:
	int 10h
die3:
	lodsb
	or al,al
	jne die2
	mov ax,4C00h		; DOS terminate
	int 21h

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; await keyboard controller ready
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

kbd0:
	jmp short $+2
	in al,60h
kbd:
	jmp short $+2
	in al,64h
	test al,1
	jnz kbd0
	test al,2
	jnz kbd
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; verify A20 is on
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

test_a20:
	push ax
	push ds
	push es
		xor ax,ax
		mov ds,ax		; DS=0
		dec ax
		mov es,ax		; ES=0FFFFh

		mov ax,[es:10h]		; read word at FFFF:0010 (1 meg)
		not ax			; 1's complement
		push word [0]		; save word at 0000:0000
			mov [0],ax	; word at 0 = ~(word at 1 meg)
			mov ax,[0]	; read it back
		pop word [0]
		cmp ax,[es:10h]		; fail if word at 0 == word at 1 meg
	pop es
	pop ds
	pop ax
	ret		; if ZF=1, the A20 gate is NOT enabled

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; handler for VCPI exceptions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	BITS 32

unhand:
	mov ax,LINEAR_DATA_SEL
	mov ds,ax
	mov byte [dword 0B8000h],'!'
	jmp $

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; data
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

page_info:
	times 1024 dd 0         ; padding to 4K boundary
	times 1024 dd 0         ; page table somewhere in here
	dd 0			; a page dir with one entry

tss:
	dw 0, 0			; back link

	dd 0			; ESP0
	dw DATA_SEL32, 0	; SS0, reserved

	dd 0			; ESP1
	dw 0, 0			; SS1, reserved

	dd 0			; ESP2
	dw 0, 0			; SS2, reserved

	dd 0			; CR3
	dd 0, 0			; EIP, EFLAGS
	dd 0, 0, 0, 0		; EAX, ECX, EDX, EBX
	dd 0, 0, 0, 0		; ESP, EBP, ESI, EDI
	dw 0, 0			; ES, reserved
	dw 0, 0			; CS, reserved
	dw 0, 0			; SS, reserved
	dw 0, 0			; DS, reserved
	dw 0, 0			; FS, reserved
	dw 0, 0			; GS, reserved
	dw 0, 0			; LDT, reserved
	dw 0, 0			; debug, IO perm. bitmap

gdt:
	dw 0			; limit 15:0
	dw 0			; base 15:0
	db 0			; base 23:16
	db 0			; type
	db 0			; limit 19:16, flags
	db 0			; base 31:24
LINEAR_CODE_SEL	equ	$-gdt
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah			; present, ring 0, code, non-conforming, readable
	db 0CFh			; page-granular, 32-bit
	db 0
LINEAR_DATA_SEL	equ	$-gdt
	dw 0FFFFh
	dw 0
	db 0
	db 92h			; present, ring 0, data, expand-up, writable
	db 0CFh			; page-granular, 32-bit
	db 0
CODE_SEL32	equ	$-gdt
gdt3:
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah			; present, ring 0, code, non-conforming, readable
	db 0CFh			; page-granular, 32-bit
	db 0
DATA_SEL32	equ	$-gdt
gdt4:
	dw 0FFFFh
	dw 0
	db 0
	db 92h			; present, ring 0, data, expand-up, writable
	db 0CFh			; page-granular, 32-bit
	db 0
CODE_SEL16	equ	$-gdt
gdt5:
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah			; present, ring 0, code, non-conforming, readable
	db 0			; byte-granular, 16-bit
	db 0
DATA_SEL16	equ	$-gdt
gdt6:
	dw 0FFFFh
	dw 0
	db 0
	db 92h			; present, ring 0, data, expand-up, writable
	db 0			; byte-granular, 16-bit
	db 0
TSS_SEL		equ	$-gdt
gdt7:
	dw 103
	dw 0
	db 0
	db 089h			; Ring 0 available 32-bit TSS
	db 0
	db 0
VCPI_CODE_SEL	equ	$-gdt
gdt8:				; these are set by INT 67h AX=DE01h
	dd 0, 0
VCPI_DATA_SEL	equ	$-gdt
	dd 0, 0
VCPI_LINEAR_SEL	equ	$-gdt
	dd 0, 0
gdt_end:

idt:
	%rep 32
		dw unhand	; low 16 bits of ISR offset
		dw CODE_SEL32	; selector
		db 0
		db 8Eh		; present, ring 0, 32-bit intr gate
		dw 0		; high 16 bits of ISR (unhand >> 16)
	%endrep
idt_end:

gdt_ptr:
	dw gdt_end - gdt - 1	; GDT limit
	dd gdt			; linear, physical adr of GDT

idt_ptr:
	dw idt_end - idt - 1	; IDT limit
	dd idt			; linear, physical address of IDT

real_idt_ptr:
	dw 3FFh			; limit 1023
	dd 0			; IDT (IVT, actually) at address 0

real_mode_ip:
	dw pmode16_to_real
real_mode_cs:
	dw 0

vcpi_control_block:
vcpi_cr3:
	dd 0
vcpi_gdtr:
	dd gdt_ptr
vcpi_idtr:
	dd idt_ptr
;vcpi_ldtr:
	dw 0
;vcpi_tr:
	dw TSS_SEL
;vcpi_eip:
	dd vcpi_to_pmode32
;vcpi_cs:
	dw CODE_SEL32

ext_mem:
	dd 0

conv_mem:
	dd 0

file_len:
	dd 0

load_adr:
	dd 0

cpu_msg:
	db "32-bit CPU (386SX or better) required", 13, 10, 0

usage_msg:
	db "Loads and executes Mobius-format initial RAM disk"
	db 13, 10, 0

open_msg:
	db "Couldn't open file", 13, 10, 0

read_msg:
	db "Error reading file", 13, 10, 0

invalid_msg:
	db "This file is not a Mobius-format initial RAM disk", 13, 10, 0

seek_msg:
	db "Seek failed (file may be corrupt)", 13, 10, 0

mem_msg:
	db "Not enough contiguous memory to load kernel", 13, 10, 0

coff_msg:
	db "First file in this RDSK file is not a DJGPP COFF file", 13, 10, 0

a20_msg:
	db "Could not enable A20 gate", 13, 10, 0

v86_msg:
	db "CPU is in virtual 8086 mode, but no VCPI", 13, 10
	db "Are you running this code from Windows? (don't)", 13, 10, 0

vcpi_err_msg:
	db "VCPI call (INT 67h AX=DE01h) failed (?)", 13, 10, 0

success_msg:
	db "Success!", 13, 10, 0

	times 512 dw 0
stack:

end:
