/*****************************************************************************
TEXT VIDEO

EXPORTS:
extern console_t _vc[MAX_VC], *_curr_vc;
extern unsigned _num_vc;

void blink(void);
void putch(console_t *con, unsigned c);
void select_vc(unsigned which_con);
void init_video(void);
void init_vc(void);
*****************************************************************************/
#include <string.h> /* memcpy(), memsetw() */
#include <stdio.h> /* sprintf() */
#include <ctype.h> /* isdigit() */
#include <_krnl.h> /* console_t */
#include <_x86.h> /* inportb(), outportb(), disable(), crite() */

/* IMPORTS
from MAIN.C */
void kprintf(const char *fmt, ...);

/* from MM.C */
void *kmalloc(size_t size);

#define	VGA_MISC_READ	0x3CC
#define	VGA_GC_INDEX	0x3CE
#define	VGA_GC_DATA	0x3CF

console_t _vc[MAX_VC];
unsigned short _num_vc;
console_t *_curr_vc;

static unsigned short *_vga_fb_adr;
static int _crtc_io_adr, _vc_width, _vc_height;
/*****************************************************************************
*****************************************************************************/
void blink(void)
{
	(*(unsigned char *)_vga_fb_adr)++;
}
/*****************************************************************************
*****************************************************************************/
static void scroll(console_t *con)
{
	unsigned short *fb_adr;
	unsigned blank, temp;

	blank = 0x20 | ((unsigned)con->attrib << 8);
	if(con == _curr_vc)
		fb_adr = _vga_fb_adr;
	else
		fb_adr = con->fb_adr;
/* scroll up */
	if(con->csr_y >= _vc_height)
	{
		temp = con->csr_y - _vc_height + 1;
		memcpy(fb_adr, fb_adr + temp * _vc_width,
			(_vc_height - temp) * _vc_width * 2);
/* blank the bottom line of the screen */
		memsetw(fb_adr + (_vc_height - temp) * _vc_width,
			blank, _vc_width);
		con->csr_y = _vc_height - 1;
	}
}
/*****************************************************************************
*****************************************************************************/
static void set_attrib(unsigned att, console_t *con)
{
	static const unsigned ansi_to_vga[] =
	{
		0, 4, 2, 6, 1, 5, 3, 7
	};
/**/
	unsigned new_att;

	new_att = con->attrib;
	if(att == 0)
		new_att &= ~0x08;		/* bold off */
	else if(att == 1)
		new_att |= 0x08;		/* bold on */
	else if(att >= 30 && att <= 37)
	{
		att = ansi_to_vga[att - 30];
		new_att = (new_att & ~0x07) | att;/* fg color */
	}
	else if(att >= 40 && att <= 47)
	{
		att = ansi_to_vga[att - 40] << 4;
		new_att = (new_att & ~0x70) | att;/* bg color */
	}
	con->attrib = new_att;
}
/*****************************************************************************
*****************************************************************************/
static void move_csr(void)
{
	unsigned off, flags;

	off = (_curr_vc->csr_y * _vc_width + _curr_vc->csr_x);
	flags = disable();
	outportb(_crtc_io_adr + 0, 14);
	outportb(_crtc_io_adr + 1, off >> 8);
	outportb(_crtc_io_adr + 0, 15);
	outportb(_crtc_io_adr + 1, off);
	crite(flags);
}
/*****************************************************************************
*****************************************************************************/
void select_vc(unsigned which_con)
{
	if(which_con >= _num_vc)
		return;
	memcpy(_curr_vc->fb_adr, _vga_fb_adr,
		_vc_width * _vc_height * 2);
	memcpy(_vga_fb_adr, _vc[which_con].fb_adr,
		_vc_width * _vc_height * 2);
	_curr_vc = _vc + which_con;
	move_csr();
}
/*****************************************************************************
*****************************************************************************/
void putch(console_t *con, unsigned c)
{
	unsigned short *fb_adr;
	unsigned att;

	att = (unsigned)con->attrib << 8;
	if(con == _curr_vc)
		fb_adr = _vga_fb_adr;
	else
		fb_adr = con->fb_adr;
/* state machine to handle the escape sequences
ESC */
	if(con->esc == 1)
	{
		if(c == '[')
		{
			con->esc++;
			con->esc1 = 0;
			return;
		}
		/* else fall-through: zero esc and print c */
	}
/* ESC[ */
	else if(con->esc == 2)
	{
		if(isdigit(c))
		{
			con->esc1 = con->esc1 * 10 + c - '0';
			return;
		}
		else if(c == ';')
		{
			con->esc++;
			con->esc2 = 0;
			return;
		}
/* ESC[2J -- clear screen */
		else if(c == 'J')
		{
			if(con->esc1 == 2)
			{
				memsetw(fb_adr, ' ' | att,
					_vc_height * _vc_width);
				con->csr_x = con->csr_y = 0;
			}
		}
/* ESC[num1m -- set attribute num1 */
		else if(c == 'm')
			set_attrib(con->esc1, con);
		con->esc = 0;	/* anything else with one numeric arg */
		return;
	}
/* ESC[num1; */
	else if(con->esc == 3)
	{
		if(isdigit(c))
		{
			con->esc2 = con->esc2 * 10 + c - '0';
			return;
		}
		else if(c == ';')
		{
			con->esc++;	/* ESC[num1;num2; */
			con->esc3 = 0;
			return;
		}
/* ESC[num1;num2H -- move cursor to num1,num2 */
		else if(c == 'H')
		{
			if(con->esc2 < _vc_width)
				con->csr_x = con->esc2;
			if(con->esc1 < _vc_height)
				con->csr_y = con->esc1;
		}
/* ESC[num1;num2m -- set attributes num1,num2 */
		else if(c == 'm')
		{
			set_attrib(con->esc1, con);
			set_attrib(con->esc2, con);
		}
		con->esc = 0;
		return;
	}
/* ESC[num1;num2;num3 */
	else if(con->esc == 4)
	{
		if(isdigit(c))
		{
			con->esc3 = con->esc3 * 10 + c - '0';
			return;
		}
/* ESC[num1;num2;num3m -- set attributes num1,num2,num3 */
		else if(c == 'm')
		{
			set_attrib(con->esc1, con);
			set_attrib(con->esc2, con);
			set_attrib(con->esc3, con);
		}
		con->esc = 0;
		return;
	}
	con->esc = 0;

/* escape character */
	if(c == 0x1B)
	{
		con->esc = 1;
		return;
	}
/* backspace */
	if(c == 0x08)
	{
		if(con->csr_x != 0)
			con->csr_x--;
	}
/* tab */
	else if(c == 0x09)
		con->csr_x = (con->csr_x + 8) & ~(8 - 1);
/* carriage return */
	else if(c == '\r')	/* 0x0D */
		con->csr_x = 0;
/* line feed */
//	else if(c == '\n')	/* 0x0A */
//		con->csr_y++;
/* CR/LF */
	else if(c == '\n')	/* ### - 0x0A again */
	{
		con->csr_x = 0;
		con->csr_y++;
	}
/* printable ASCII */
	else if(c >= ' ')
	{
		unsigned short *where;

		where = fb_adr + (con->csr_y * _vc_width + con->csr_x);
		*where = (c | att);
		con->csr_x++;
	}
	if(con->csr_x >= _vc_width)
	{
		con->csr_x = 0;
		con->csr_y++;
	}
	scroll(con);
/* move cursor only if the VC we're writing is the current VC */
	if(_curr_vc == con)
		move_csr();
}
/*****************************************************************************
*****************************************************************************/
static void kputs(console_t *con, char *str)
{
	while(*str != '\0')
	{
		putch(con, *str);
		str++;
	}
}
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(void init_video(void))
{
	unsigned v_disp, char_ht;

/* code to detect mono/color emulation
cobbled from info in Finn Thoegersen's VGADOC4
Mono is UNTESTED. */
	if((inportb(VGA_MISC_READ) & 0x01) != 0)
	{
		_crtc_io_adr = 0x3D4;	/* color */
		_vga_fb_adr = (unsigned short *)0xB8000L;
	}
	else
	{
		_crtc_io_adr = 0x3B4;	/* mono */
		_vga_fb_adr = (unsigned short *)0xB0000L;
	}
#if 1
/* check KSTART.ASM to make sure page 0 of physical memory is
identity-mapped before accessing the BIOS data segment like this */
	_vc_width = *(unsigned short *)0x44A;
	_vc_height = *(unsigned char *)0x484 + 1;
#else
/* figure out width and height of current display:
vertical scan lines displayed */
	outportb(_crtc_io_adr, 0x12);
	v_disp = inportb(_crtc_io_adr + 1);
/* pull in the 9th and 10th bits from the dread overflow register */
	outportb(_crtc_io_adr, 0x07);
	temp = inportb(_crtc_io_adr + 1);
	if((temp & 0x02) != 0)
		v_disp |= 0x100;
	if((temp & 0x40) != 0)
		v_disp |= 0x200;
/* the value in the register is actually vertical displayed - 1, so fix it */
	v_disp++;
/* scan lines/char */
	outportb(_crtc_io_adr, 0x09);
	char_ht = (inportb(_crtc_io_adr + 1) & 0x1F) + 1;
/* vertical resolution in characters is the quotient.
If the monochrome/color detection is faulty, char_ht may be 0,
and the divide-by-zero will cause a triple fault. */
	_vc_height = v_disp / char_ht;
/* horizontal resolution in characters */
	outportb(_crtc_io_adr, 0x01);
	_vc_width = inportb(_crtc_io_adr + 1) + 1;
#endif
/* kernel heap doesn't exist yet, and we can't have additional VCs
without kmalloc(), so initialize only VC #0 for now */
	_num_vc = 1;
	_curr_vc = _vc + 0;
	_curr_vc->attrib = 0x01;		/* bright blue on black */
	_curr_vc->fb_adr = _vga_fb_adr;		/* use real framebuffer */
	kprintf("\x1B[2J");			/* clear screen */
/* leave a space or two to avoid the upper left corner */
	kprintf("  init_video: %s emulation, %u x %u, framebuffer at "
		"0x%lX\n", (_crtc_io_adr == 0x3D4) ? "color" : "mono",
		_vc_width, _vc_height, _vga_fb_adr);
}
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(void init_vc(void))
{
	unsigned i, j;

	j = _vc_width * _vc_height * 2;
	for(i = 0; i < MAX_VC; i++)
	{
		_vc[i].attrib = i + 1;		/* different colors */
		_vc[i].fb_adr = kmalloc(j);	/* virtual framebuffer */
		if(_vc[i].fb_adr == NULL)
			break;
		if(i != 0)			/* clear screen */
			kputs(_vc + i, "\x1B[2J");
	}
/* even if all kmalloc()s fail, we will have one VC
backed with video memory, instead of memory from kernel heap */
	_num_vc = i;
	if(i == 0)
		i = 1;
	kprintf("init_vc: %u virtual consoles, %u bytes each\n", _num_vc, j);
}

