/*****************************************************************************
TASKS AND KERNEL THREADS

EXPORTS:
extern task_t _tasks[], *_curr_task;

int init_tasks(unsigned char *image);
void wake_up(wait_queue_t *queue);
int sleep_on(wait_queue_t *queue, unsigned *timeout);
void timer_irq(unsigned eip);
int new_thread(task_t *task, unsigned eip);
*****************************************************************************/
#include <string.h> /* NULL, memcmp() */
#include <_krnl.h> /* MAX_VC, TS_..., task_t, read_le32() */
#include <_x86.h> /* outportb() */

/* IMPORTS
from MAIN.C */
void kprintf(const char *fmt, ...);

/* from VIDEO.C */
extern console_t _vc[];
extern unsigned _num_vc;

/* from LOADER.C */
int run(task_t *task, unsigned char *image);

/* from PAGING.C */
unsigned alloc_page(void);

task_t _tasks[MAX_TASK], *_curr_task;
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(int init_tasks(unsigned char *image))
{
	unsigned char *rdsk_dir, *exec_file;
	unsigned short num_objs, t, o;
	task_t *task;

/* make sure it's a valid initial RAM disk */
	if(memcmp(image, "RDSK", 4))
	{
		kprintf("init_tasks: did not find RDSK signature\n");
		return -1;
	}
/* the number of objects in the image is right after the signature */
	num_objs = read_le32(image + 4);
	t = 1;
	for(o = 1; o < num_objs; o++)
	{
/* don't run more tasks than we have virtual consoles */
		if(t >= _num_vc)
			break;
		kprintf("obj %02u -> task %02u,  ", o, t);
/* point to the 24-byte directory entry in the image for this task */
		rdsk_dir = image + 8 + 24 * o;
/* get offset of object from directory of image; point to executable file */
		exec_file = read_le32(rdsk_dir + 0) + image;
/* try loading as DJGPP COFF */
		task = _tasks + t;
		if(run(task, exec_file) != 0)
			continue;
/* set other task stuff */
		task->status = TS_RUNNABLE;
		task->vc = _vc + t;
		t++;
	}
	kprintf("\n");
/* return nonzero if no tasks were loaded */
	return (t == 1) ? -1 : 0;
}
/*****************************************************************************
*****************************************************************************/
void wake_up(wait_queue_t *queue)
{
	task_t *task, *next;

/* make sure queue is not empty */
	task = queue->head;
	if(task == NULL)
		return;
/* mark head task in queue runnable */
	task->status = TS_RUNNABLE;
/* remove head task from queue */
	next = task->next;
	queue->head = next;
	if(next != NULL)
		next->prev = NULL;
	else
		queue->tail = NULL;
}
/*****************************************************************************
*****************************************************************************/
int sleep_on(wait_queue_t *queue, unsigned *timeout)
{
	int ret_val = 0;
	task_t *prev;

/* mark task blocked */
	_curr_task->status = TS_BLOCKED;
/* splice into wait queue at queue->tail */
	prev = queue->tail;
	queue->tail = _curr_task;
	if(prev == NULL)
	{
		queue->head = _curr_task;
		_curr_task->prev = NULL;
	}
	else
	{
		_curr_task->prev = prev;
		prev->next = _curr_task;
	}
	_curr_task->next = NULL;
/* set the timeout, if there is one */
	if(timeout != NULL)
		_curr_task->timeout = *timeout;
	else
		_curr_task->timeout = 0;
/* "fake" a timer interrupt. timer_irq() must detect this */
	__asm__ __volatile__("int $0x20");
/* now: why did we return? */
	if(timeout != NULL)
	{
		*timeout = _curr_task->timeout;
/* there was a timeout, so timer_irq() awoke us. Return -1 */
		if(*timeout == 0)
			ret_val = -1;
	}
/* someone called wake_up(), making us TS_RUNNABLE again. Return 0 */
	return ret_val;
}
/*****************************************************************************
xxx - should be static
*****************************************************************************/
#if 0
/*static*/ void schedule(void)
{
	static unsigned current;
/**/

	do
	{
		current++;
		if(current >= MAX_TASK)
			current = 0;
		_curr_task = _tasks + current;
	} while(_curr_task->status != TS_RUNNABLE);
}
#else
/*static*/ void schedule(void)
{
	static unsigned current;
/**/
	unsigned i;
	int p = -127;

/* find TS_RUNNABLE task with highest priority */
	for(i = 0; i < MAX_TASK; i++)
	{
		if(_tasks[i].status == TS_RUNNABLE)
		{
			if(_tasks[i].priority > p)
				p = _tasks[i].priority;
		}
	}
/* find next task beyond current with this priority and run it */
	do
	{
		current++;
		if(current >= MAX_TASK)
			current = 0;
		_curr_task = _tasks + current;
	} while(_curr_task->status != TS_RUNNABLE ||
		_curr_task->priority < p);
}
#endif
/*****************************************************************************
xxx - calculation of timeouts is fubar
*****************************************************************************/
void timer_irq(unsigned eip)
{
	unsigned new_time;
//	unsigned char *instr;
	unsigned short i;

/* was it a real timer interrupt? */
// xxx - this causes the OS to freeze
//	if(inportb(0x20) & 0x01)

// xxx - this causes page fault in kernel mode; CR2=0xBFFFF000
//	instr = (unsigned char *)eip - 2;
//	if(*instr != 0xCD)
	{
/* decrement timeouts for tasks that have them */
		for(i = 0; i < MAX_TASK; i++)
		{
			new_time = _tasks[i].timeout;
			if(new_time == 0)
				continue;
/* number of microseconds per timer IRQ */
			new_time -= (1000000L / HZ);
			if(new_time > _tasks[i].timeout)
				new_time = 0;	/* underflow */
			_tasks[i].timeout = new_time;
/* timeout? wake up task */
			if(new_time == 0 && _tasks[i].status == TS_BLOCKED)
				_tasks[i].status = TS_RUNNABLE;
		}
//		outportb(0x20, 0x20); /* reset 8259 chip */
	}
outportb(0x20, 0x20);
/* run the scheduler, whether the interrupt was hardware (IRQ 0)
or software (INT 20h) */
//	schedule();
}
/*****************************************************************************
*****************************************************************************/
int new_thread(task_t *task, unsigned eip)
{
	unsigned long foo;
	regs_t *regs;

	memset(task, 0, sizeof(task_t));
/* use page tables of current task */
	task->page_dir = _curr_task->page_dir;
/* create a kernel stack for the thread */
	foo = alloc_page();
	if(foo == NULL)
	{
		kprintf("out of memory in new_thread()\n");
		return -1;
	}
/* NOTE: the kernel stack is a physical address */
	task->init_krnl_esp = foo + PAGE_SIZE;
/* put initial register values on kernel stack, to be popped by task-switch */
	task->krnl_esp = task->init_krnl_esp - sizeof(regs_t);
	regs = (regs_t *)task->krnl_esp;
	regs->ds = regs->es = regs->fs =
		regs->gs = regs->user_ss = SYS_DATA_SEL;
	task->entry_pt = regs->eip = eip;
	regs->cs = SYS_CODE_SEL;
/* run thread with interrupts enabled */
	regs->eflags = 0x200;
	return 0;
}
