/*****************************************************************************
SYSCALLS

EXPORTS:
void sys_exit(int exit_code);
bool syscall(regs_t *regs);
*****************************************************************************/
#include <string.h> /* NULL */
#include <_syscall.h> /* SYS_... */
#include <_krnl.h>

/* IMPORTS
from MAIN.C */
extern task_t *_curr_task, _tasks[];
extern wait_queue_t _task_died;

/* from TASKS.C */
int sleep_on(wait_queue_t *queue, unsigned *timeout);
void wake_up(wait_queue_t *queue);

/* from MAIN.C */
void kprintf(const char *fmt, ...);

/* from VIDEO.C */
void putch(console_t *con, unsigned char c);

/* from TIME.C */
unsigned long sys_time(void);

/* from KBD.C */
int deq(queue_t *q, unsigned char *data);
/*****************************************************************************
*****************************************************************************/
static unsigned long sys_sbrk(long delta)
{
	sect_t *sect;

	sect = _curr_task->sect + HEAP_SECT;
/* shrink heap */
	if(delta < 0)
	{
/* don't lower break value if delta is too large */
		if((unsigned long)(-delta) < sect->size)
			sect->size += delta;
	}
/* grow heap */
	else if(delta > 0)
	{
/* don't increase break value if delta is too large */
		if(delta + sect->size < MAX_HEAP_SIZE)
			sect->size += delta;
	}
/* return new or current break value */
	return sect->adr + sect->size;
}
/*****************************************************************************
*****************************************************************************/
void sys_exit(int exit_code)
{

	_curr_task->exit_code = exit_code;
	_curr_task->status = TS_ZOMBIE;
	wake_up(&_task_died);
}
/*****************************************************************************
*****************************************************************************/
static bool sys_sleep(unsigned timeout)
{
/* 'dummy' because wake_up() is not used; only the timeout */
	static wait_queue_t dummy;
/**/

	if(timeout != 0)
	{
		sleep_on(&dummy, &timeout);
		return false;
	}
	return true;
}
/*****************************************************************************
*****************************************************************************/
static int sys_write(unsigned handle, void *bufp,
		unsigned len)
{
	char *buf = (char *)bufp;
	unsigned pos;

	for(pos = 0; pos < len; pos++)
	{
		putch(_curr_task->vc, *buf);
		buf++;
	}
	return len;
}
/*****************************************************************************
*****************************************************************************/
static int empty(queue_t *q)
{
	return q->out_ptr == q->in_base;
}
/*****************************************************************************
*****************************************************************************/
static int sys_read(unsigned handle, unsigned char *buf,
		unsigned want)
{
	console_t *con;
	unsigned got;
	queue_t *q;

	con = _curr_task->vc;
	q = &con->keystrokes;
if(con->unbuffered)
 want = 1;
	for(got = 0; got < want; got++)
	{
		do
		{
			while(empty(q))
				sleep_on(/*_curr_task,*/ &con->wait_queue, NULL);
		} while(deq(q, buf) != 0);
		if(*buf == '\0')
			break;
		buf++;
	}
	return got;
}
/*****************************************************************************
*****************************************************************************/
static int sys_ioctl(unsigned handle, unsigned char opcode, unsigned arg)
{

	switch(opcode)
	{
/* turn keyboard buffering on/off */
		case 0:
			switch(arg)
			{
				case 0:
				case 1:
					_curr_task->vc->unbuffered = arg;
					break;
				default:
					return -1;
			}
			break;
		default:
			return -1;
	}
	return 0;
}
/*****************************************************************************
returns nonzero if timeout while waiting for input
*****************************************************************************/
static int sys_select(unsigned handle, unsigned timeout)
{
	console_t *con;
	queue_t *q;

	con = _curr_task->vc;
	q = &con->keystrokes;
	if(!empty(q))
		return 0;
	sleep_on(&con->wait_queue, &timeout);
	return timeout == 0;
}
/*****************************************************************************
*****************************************************************************/
bool syscall(regs_t *regs)
{
	bool must_call_schedule = false;

	switch(regs->eax)
	{
		case SYS_WRITE:
			regs->eax = sys_write(regs->edx,
				(unsigned char *)regs->ebx,
				regs->ecx);
			break;
		case SYS_READ:
			regs->eax = sys_read(regs->edx,
				(unsigned char *)regs->ebx,
				regs->ecx);
			break;
		case SYS_IOCTL:
			regs->eax = sys_ioctl(regs->edx,
				regs->ebx, regs->ecx);
			break;
		case SYS_SELECT:
			regs->eax = sys_select(regs->edx,
				regs->ebx);
			break;
		case SYS_SBRK:
			regs->eax = sys_sbrk(regs->ebx);
			break;
		case SYS_TIME:
			regs->eax = sys_time();
			break;
		case SYS_EXIT:
			sys_exit(regs->ebx);
			must_call_schedule = true;
			break;
		case SYS_SLEEP:
			must_call_schedule = sys_sleep(regs->ebx);
			break;
		default:
			kprintf("Illegal syscall 0x%X; task %u killed\n",
				regs->eax, _curr_task - _tasks);
			sys_exit(-1);
			must_call_schedule = true;
			break;
	}
	return must_call_schedule;
}
