/*****************************************************************************
after you figure this out yourself, write some paging demo code:
- everything allocated statically
- task memory allocated when task created, alloc_page(), free_page()
- same as above, for variable-sized and discontiguous RAM
- same as above, plus demand-loading
- same as above, plus shared memory
- same as above, plus COW memory
- same as above, plus virtual memory (swapping)
- same as above, plus virtual page fault handler methods


kstart32.asm should map memory as follows:
	what			bottom	top		priv
	-----------------------	-------	---------------	----
	conventional memory	0x1000	_conv_mem_size	R/W
	video memory		0xA0000	0xBFFFF		R/W
	upper memory/BIOSes/
		adapter memory	0xC0000	0xFFFFF		RO
	kernel code, rodata	...	...		RO
	kernel data, BSS, heap,
		stack		...	...		RW

when you create a new address space:
- allocate 1 page for the page directory, set PAGEDIR_SELF
- set page dir entries for identity-mapped RAM
- set page dir entries for kernel memory



xxx - need verify_area() on pointers passed to kernel, or try to
	figure out that exception-based magic that Linux uses

xxx - support discontiguous chunks of extended memory

invalidate when freeing pages, sharing or unsharing pages,
or making pages read-only or read-write
*****************************************************************************/
/*****************************************************************************
EXPORTS:
int init_paging(void);
int map_page(unsigned virt, unsigned phys, unsigned priv);
void discard_mem(void);
unsigned alloc_page(void);
int free_page(unsigned phys_adr);
void *kbrk(int incr);
void free_task_pages(unsigned *page_dir);
int page_fault(unsigned err_code);
*****************************************************************************/
#include <string.h>
#include <_krnl.h>

/* IMPORTS
from kernel linker script file */
extern unsigned char _code[], _d_code[], _data[];
extern unsigned char _d_data[], _bss[], _d_bss[], _end[];

/* from TASKS.C */
extern task_t *_curr_task;

/* from MAIN.C */
void kprintf(const char *fmt, ...);

/* from KSTART.ASM */
extern unsigned _conv_mem_size, _ext_mem_size, _kvirt_to_phys;
extern unsigned _init_ramdisk_adr, _init_ramdisk_size;

unsigned get_page_fault_adr(void);
unsigned get_page_dir(void);
void set_page_dir(unsigned cr3);

/* from DEBUG.C */
void dump_stack_frame(void);

#if 0
#define	DEBUG(X)	X
#else
#define DEBUG(X)	/* nothing */
#endif

/* the 12 bits at the bottom of a page directory/table entry: */
#define	PRIV_PRESENT	0x001
#define	PRIV_WRITABLE	0x002
#define	PRIV_USER	0x004
/* b3, b4 used for cache control on 486+ */
//efine	PRIV_ACCESSED	0x020
//efine	PRIV_DIRTY	0x040
/* b7, b8 reserved */
//efine	PRIV_USER2	0x200		/* user-defined */
//efine	PRIV_USER4	0x400
//efine	PRIV_USER8	0x800
//efine	PRIV_COW	PRIV_USER2	/* copy-on-write */
//efine	PRIV_ALL	0xFFF

/* xxx - 16 meg RAM exactly */
#define	MAX	4096

static unsigned char _page_use_count[MAX];
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(int init_paging(void))
{
	unsigned start, end, i;

/* mark page 0 in use */
	_page_use_count[0] = (unsigned char)-1;
/* mark conventional memory used by RDSK file in use */
	start = _init_ramdisk_adr / PAGE_SIZE;
	end = (_init_ramdisk_adr + _init_ramdisk_size) / PAGE_SIZE;
	for(i = start; i < end; i++)
		_page_use_count[i] = (unsigned char)-1;
/* mark non-existent conventional memory and adapter memory in use */
	start = _conv_mem_size / PAGE_SIZE;
	end = 0x100000L / PAGE_SIZE;
	for(i = start; i < end; i++)
		_page_use_count[i] = (unsigned char)-1;
/* mark kernel physical memory in use */
	start = ((unsigned)_code + _kvirt_to_phys) / PAGE_SIZE;
	end = ((unsigned)_end + _kvirt_to_phys) / PAGE_SIZE;
	for(i = start; i < end; i++)
		_page_use_count[i] = (unsigned char)-1;
/* mark non-existent extended memory in use */
	start = (0x100000L + _ext_mem_size) / PAGE_SIZE;
	end = MAX;
	for(i = start; i < end; i++)
		_page_use_count[i] = (unsigned char)-1;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int map_page(unsigned virt, unsigned phys, unsigned priv)
{
	unsigned *page_dir, d, dir_ent, dt, *page_tables;

	page_dir = (unsigned *)PAGE_DIR_VA;
	d = virt >> 22;
	dir_ent = page_dir[d];
	if((dir_ent & PRIV_PRESENT) == 0)
{kprintf("map_page(): missing page table while mapping 0x%X -> 0x%X\n", virt, phys);
		return -1;
}
	page_tables = (unsigned *)PAGE_TABLES_VA;
	dt = virt >> 12;
/* store mapping */
	phys &= -PAGE_SIZE;
	priv &= (PAGE_SIZE - 1);
	page_tables[dt] = phys | priv;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
//static int map_mem(unsigned *page_dir, unsigned virt, unsigned phys,
//		unsigned priv, unsigned len)
//{
//	unsigned tvirt;
//
//	for(tvirt = virt; tvirt < virt + len; tvirt += PAGE_SIZE)
//	{
//		if(map_page(tvirt, phys, priv) != 0)
//			return -1;
//		phys += PAGE_SIZE;
//	}
//	return 0;
//}
/*****************************************************************************
*****************************************************************************/
static int unmap_mem(unsigned virt, unsigned len)
{
	unsigned tvirt;

	for(tvirt = virt; tvirt < virt + len; tvirt += PAGE_SIZE)
	{
		if(map_page(tvirt, 0, /* priv== */0) != 0)
			return -1;
	}
	return 0;
}
/*****************************************************************************
*****************************************************************************/
void discard_mem(void)
{
	unsigned discard, base, size;

	discard = (_data - _d_code) + (_bss - _d_data) + (_end - _d_bss);
	kprintf("Discarding %u bytes kernel memory...\n", discard);
	(void)unmap_mem((unsigned)_d_code, _data - _d_code);
	(void)unmap_mem((unsigned)_d_data, _bss - _d_data);
	(void)unmap_mem((unsigned)_d_bss, _end - _d_bss);
//kprintf("Invalidating...\n");
//set_page_dir(get_page_dir());
//kprintf("...done...\n");

/* delete identity-mapped kernel memory */
	base = (unsigned)_code + _kvirt_to_phys;
	size = _end - _code;
/* don't unmap the bottom 4 meg, even if the kernel is there */
	if(base + size >= 0x400000L)
	{
		if(base < 0x400000L)
		{
			size -= (0x400000L - base);
			base = 0x400000L;
		}
//kprintf("Removing kernel identity-mapping: 0x%X bytes at 0x%X\n", size, base);
		unmap_mem(base, size);
	}
//kprintf("Invalidating...\n");
//set_page_dir(get_page_dir());
//kprintf("...done...\n");

//kprintf("Unmapping page 0...\n");
	unmap_mem(0, PAGE_SIZE);
//kprintf("Invalidating...\n");
//set_page_dir(get_page_dir());
//kprintf("...done...\n");
}
/*****************************************************************************
allocates one page (4K) of memory
returns PHYSICAL address or NULL if out of memory
*****************************************************************************/
unsigned alloc_page(void)
{
	unsigned i;

/* skip page 0; start with 1 */
	for(i = 1; i < MAX; i++)
	{
		if(_page_use_count[i] == 0)
		{
			_page_use_count[i] = 1;
			return i * PAGE_SIZE;
		}
	}
	return NULL;
}
/*****************************************************************************
*****************************************************************************/
int free_page(unsigned phys_adr)
{
	unsigned i;

	i = phys_adr >> 12;
	if(i >= MAX)
	{
		kprintf("free_page: bad page address 0x%X\n", phys_adr);
		return -1;
	}
	if(_page_use_count[i] == 0)
	{
		kprintf("free_page: trying to free already-free page 0x%X\n",
			phys_adr);
		return -1;
	}
	(_page_use_count[i])--;
	return 0;
}
/*****************************************************************************
xxx - returning old brk value doesn't signal failure or success
*****************************************************************************/
void *kbrk(int incr)
{
	static unsigned char *brk;
/**/
	void *ret_val;
	unsigned phys;
	int virt, err;

	if(brk == NULL)
	{
		brk = _end;
		kprintf("initialized kernel heap\n");
	}
	ret_val = brk;
/* round allocation to next higher multiple of one page */
	if(incr > 0)
		incr += (PAGE_SIZE - 1);
	incr &= -PAGE_SIZE;
/* xxx - can't yet downsize kernel heap */
	if(incr < 0)
		return brk;
/* allocate and map memory */
	for(virt = (int)brk; virt < (int)brk + incr; virt += PAGE_SIZE)
	{
		phys = alloc_page();
		if(phys == NULL)
		{
			// xxx - clean up
kprintf("kbrk() error\n");
			return ret_val;
		}
		err = map_page(virt, phys, PRIV_PRESENT | PRIV_WRITABLE);
		if(err)
		{
			// xxx - clean up
kprintf("kbrk() error\n");
			return ret_val;
		}
	}
	brk = brk + incr;
	return ret_val;
}
/*****************************************************************************
xxx - using physical addresses here
*****************************************************************************/
void free_task_pages(unsigned *page_dir)
{
	unsigned dir_ent, *page_tab, tab_ent, d, t;

	if(page_dir == NULL)
		return;
/* for each page directory entry (i.e. for each page table)... */
	for(d = 256; d < 768; d++)
//	for(d = PDI(USER_BOT); d < PDI(USER_TOP); d++)
	{
		dir_ent = page_dir[d];
/* page table not Present */
		if((dir_ent & PRIV_PRESENT) == 0)
			continue;
/* convert page dir entry to physical address of page table */
		dir_ent &= -PAGE_SIZE;
		page_tab = (unsigned *)dir_ent;
/* for each entry in page table (i.e. for each page)... */
		for(t = 0; t < 1024; t++)
		{
			tab_ent = page_tab[t];
/* page not present */
			if((tab_ent & PRIV_PRESENT) == 0)
				continue;
/* convert page table entry to physical address of page */
			tab_ent &= -PAGE_SIZE;
/* free the page */
			free_page(tab_ent);
		}
/* free the page table */
		free_page(dir_ent);
	}
/* free the page dir */
	free_page((unsigned)page_dir);
}
/*****************************************************************************
*****************************************************************************/
void free_pages(unsigned start_phys_adr, unsigned size)
{
	for(; size != 0; size -= PAGE_SIZE)
	{
		if(free_page(start_phys_adr) != 0)
			break;
		start_phys_adr += PAGE_SIZE;
	}
}
/*****************************************************************************
converts fault_adr to index-into-page-directory,
checks if page table installed there,
installs one if necessary
returns VIRTUAL address of page table
*****************************************************************************/
static unsigned *get_page_table(unsigned *dir_virt, unsigned fault_adr)
{
	unsigned tab_phys, *tab_virt, de;

/* get page directory entry (PDE)
The PDE contains the physical address of the page table */
	de = (fault_adr >> 22) & 0x3FF;
	tab_phys = dir_virt[de];
/* the directory entry number can also be converted into the
VIRTUAL address of the page table */
	tab_virt = (unsigned *)(PAGE_TABLES_VA | (de << 12));
/* if no page table here, create one */
	if(tab_phys == 0) /* ### - maybe ((tab_phys & PRIV_PRESENT) == 0) */
	{
		tab_phys = alloc_page();
		if(tab_phys == NULL)
		{
			kprintf("get_page_table: out of memory\n");
			return NULL;
		}
/* graft new page table into page directory
For the underlying pages to be PRIV_USER, the page table must be PRIV_USER
### - set PRIV_ACCESSED and PRIV_DIRTY as well? to prevent TLB miss? */
		dir_virt[de] = tab_phys | PRIV_PRESENT |
			PRIV_WRITABLE | PRIV_USER;
		memset(tab_virt, 0, PAGE_SIZE);
	}
	return tab_virt;
}
/*****************************************************************************
We need no invalidates here. As the source code to Linux says,
"no need to invalidate: a not-present page shouldn't be cached"
*****************************************************************************/
static int no_page(unsigned fault_adr)
{
	unsigned new_page_phys = 0, *dir_virt, *tab_virt, i, priv, te;
	sect_t *sect;

	DEBUG(kprintf("task %u: pg fault @ 0x%lX: ",
		_curr_task - _tasks, fault_adr);)
	fault_adr &= -PAGE_SIZE;
/* demand-load, demand-zero, or demand-allocate user task memory */
	for(i = 0; i < _curr_task->num_sects; i++)
	{
		sect = _curr_task->sect + i;
		if(fault_adr >= sect->adr &&
			fault_adr < sect->adr + sect->size)
				goto FOUND_IT;
	}
	kprintf("oops, invalid page fault; address 0x%lX\n", fault_adr);
dump_stack_frame();
	return -1;
FOUND_IT:
	DEBUG(kprintf("sect %u, 0x%lX-0x%lX, ",
		i, sect->adr, sect->adr + sect->size);)
/* can't do anything with sect->exec for x86 pages */
	priv = PRIV_PRESENT;
	if(sect->write)
		priv |= PRIV_WRITABLE;
	priv |= PRIV_USER;
/* alloc page and check for out-of-memory */
	new_page_phys = alloc_page();
	if(new_page_phys == 0)
	{
		kprintf("no_page: out of memory\n");
		return -1;
	}
	dir_virt = (unsigned *)PAGE_DIR_VA;
/* get page directory entry (PDE),
which contains the physical address of the page table */
	tab_virt = get_page_table(dir_virt, fault_adr);
	if(tab_virt == NULL)
	{
		kprintf("oops, missing page table\n");
dump_stack_frame();
		return -1;
	}
/* set page table entry (PTE) to
PHYSICAL address of new page and correct privilege */
	te = (fault_adr >> 12) & 0x3FF;
/* even if the page is read-only, make it writable so we can load it */
	tab_virt[te] = new_page_phys |
		(PRIV_USER | PRIV_WRITABLE | PRIV_PRESENT);
/* zero the page if necessary */
	if(sect->zero)
	{
		DEBUG(kprintf("zeroed\n");)
		memset((void *)fault_adr, 0, PAGE_SIZE);
	}
	else if(sect->load)
	{
		unsigned char *src;

		DEBUG(kprintf("loaded\n");)
/* file offsets (sect->off) were converted to conventional memory
physical addresses by the task load code */
		src = (unsigned char *)(fault_adr - sect->adr + sect->off);
/* no block devices yet, just copy from pre-loaded ELF or COFF image
The RDSK file was loaded in conventional by the bootloader,
and conventional memory is identity-mapped,
so we can indeed use a physical address (src) here */
		memcpy((void *)fault_adr, (void *)src, PAGE_SIZE);
	}
	DEBUG(
	else
		kprintf("allocated\n");
	)
/* NOW make the page read-only, if necessary */
	tab_virt[te] = new_page_phys | priv;
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int page_fault(unsigned err_code)
{
	unsigned fault_adr;

	fault_adr = get_page_fault_adr();
/* PAGE NOT PRESENT (write or read) */
	if((err_code & 1) == 0)
		return no_page(fault_adr);
/* PRIVILEGE VIOLATION
let fault() kill the task */
	if(err_code & 4)
	{
		kprintf("page_fault: attempt to access privileged "
			"memory at 0x%X\n", fault_adr);
		return -1;
	}
/* WRITE TO READ-ONLY PAGE
let fault() kill the task or panic
	else if(err_code & 2) */
	return -1;
}
