/*****************************************************************************
KERNEL MEMORY MANAGEMENT

EXPORTS:
void *kmalloc(size_t size);
void kfree(void *blk);
void *krealloc(void *blk, size_t size);
*****************************************************************************/
#include <string.h> /* memcpy() */
#include <_malloc.h> /* MALLOC_MAGIC, malloc_t */
#include <_krnl.h>

/* IMPORTS
from PAGING.C */
void *kbrk(int incr);

static malloc_t *_kheap_bot, *_kheap_top;
/*****************************************************************************
*****************************************************************************/
void *kmalloc(size_t size)
{
	size_t total_size;
	malloc_t *blk;

/* this is easy */
	if(size == 0)
		return NULL;
	total_size = size + sizeof(malloc_t);
/* heap doesn't exist yet */
	if(_kheap_bot == NULL)
	{
/* expand data segment with kbrk() */
		blk = kbrk(total_size);
//dump_page_tables();
		if((int)blk == -1)
			return NULL;
		_kheap_bot = _kheap_top = blk;
/* if kbrk() gave us more memory than we wanted,
split off the excess to form an unused block
xxx - punt
		blk->size = (size_t)kbrk(0) - (size_t)blk; */
		blk->size = size;

		blk->next = NULL;
		blk->magic = MALLOC_MAGIC;
		blk->used = 1;
		return (char *)blk + sizeof(malloc_t);
	}
/* search heap for free block (FIRST FIT) */
	for(blk = _kheap_bot; blk != NULL; blk = blk->next)
	{
/* xxx - size == blk->size would be a perfect fit; should I allow it? */
		if(!blk->used && total_size <= blk->size)
			break;
	}
/* found one */
	if(blk != NULL)
	{
		malloc_t *new_blk;

		new_blk = (malloc_t *)((char *)blk + total_size);
/* point to new empty block and downsize it */
		new_blk->used = 0;
		new_blk->size = blk->size - total_size;
		new_blk->next = blk->next;
		new_blk->magic = MALLOC_MAGIC;
/* set _kheap_top if necessary */
		if(new_blk->next == NULL)
			_kheap_top = new_blk;
/* point to new used block and set it */
		blk->used = 1;
		blk->size = size;
		blk->next = new_blk;
		blk->magic = MALLOC_MAGIC;
		return (char *)blk + sizeof(malloc_t);
	}
/* enlarge heap */
	blk = kbrk(total_size);
	if((int)blk == -1)
		return NULL;
	blk->used = 1;
	blk->size = size;
	blk->next = NULL;
	blk->magic = MALLOC_MAGIC;
/* this block is the new end-of-heap */
	_kheap_top->next = blk;
	_kheap_top = blk;
	return (char *)blk + sizeof(malloc_t);
}
/*****************************************************************************
*****************************************************************************/
void kfree(void *blk)
{
	malloc_t *t_blk;

/* get the actual address of the block */
	blk = (char *)blk - sizeof(malloc_t);
	t_blk = (malloc_t *)blk;
	if(t_blk->magic != MALLOC_MAGIC)
	{
/*		kprintf("*** warning: kfree() called with bad pointer %p "
			"(bad magic value) ***\n", blk); */
		return;
	}
/* find it */
	for(t_blk = _kheap_bot; t_blk != NULL; t_blk = t_blk->next)
	{
		if(t_blk == blk)
			break;
	}
/* not found? bad pointer or no heap or something else? */
	if(t_blk == NULL)
	{
/*		kprintf("*** warning: kfree() called with bad pointer %p "
			"(pointer not found) ***\n", blk); */
		return;
	}
/* free the block */
	t_blk->used = 0;
/* coalesce adjacent kfree blocks
Hard to spell, hard to do */
	for(t_blk = _kheap_bot; t_blk != NULL; t_blk = t_blk->next)
	{
		while(!t_blk->used && (t_blk->next != NULL) &&
			!t_blk->next->used)
		{
/* resize this block */
			t_blk->size += sizeof(malloc_t) + t_blk->next->size;
/* merge with next block */
			t_blk->next = t_blk->next->next;
		}
	}
}
/*****************************************************************************
*****************************************************************************/
void *krealloc(void *blk, size_t size)
{
	malloc_t *t_blk;
	void *new_blk;

/* this is easy */
	if(blk == NULL)
		return kmalloc(size);
/* so is this */
	if(size == 0)
	{
		kfree(blk);
		return NULL;
	}
/* point to data in old block */
	blk = (char *)blk - sizeof(malloc_t);
	t_blk = (malloc_t *)blk;
/* xxx - is this needed? won't free() do this for us?
should we make the other test, like free() does? */
	if(t_blk->magic != MALLOC_MAGIC)
	{
/*		kprintf("*** warning: krealloc() called with bad pointer %p "
			"(bad magic value) ***\n", blk); */
		return blk; /* xxx- NULL */
	}
/* alloc new block (larger or smaller) */
	new_blk = kmalloc(size);
	if(new_blk == NULL)
		return NULL;
/* copy data */
	memcpy(new_blk, blk, t_blk->size);
/* release old block */
	kfree(blk);
/* return new block */
	return new_blk;
}
