/*****************************************************************************
MAIN KERNEL

EXPORTS:
void kprintf(const char *fmt, ...);
void fault(regs_t *regs);
int main(void);
*****************************************************************************/
#include <string.h> /* NULL, memset() */
#include <_syscall.h> /* SYS_... */
#include <_printf.h> /* do_printf() */
#include <_krnl.h> /* MAX_VC, TS_..., task_t, read_le32() */
#include <_x86.h> /* outportb(), disable(), enable() */

/* IMPORTS
from STARTUP.ASM */
void halt(void);
unsigned *get_page_dir(void);

extern unsigned _conv_mem_size, _ext_mem_size;
extern unsigned _kvirt_to_phys, _init_ramdisk_adr;

/* from DEBUG.C */
void dump_regs(regs_t *regs);

/* from PAGING.C */
int init_paging(void);
void discard_mem(void);
void free_task_pages(unsigned *page_dir);
int page_fault(unsigned err_code);

/* from VIDEO.C */
extern console_t *_curr_vc;

void blink(void);
void putch(console_t *con, unsigned c);
void init_video(void);
void init_vc(void);

/* from SYSCALLS.C */
void sys_exit(int exit_code);
bool syscall(regs_t *regs);

/* from KBD.C */
void kbd_bh(void);
void kbd_irq(void);
void init_kbd(void);

/* from TASKS.C */
extern task_t *_curr_task, _tasks[];

int init_tasks(unsigned char *image);
int sleep_on(wait_queue_t *queue, unsigned *timeout);
void timer_irq(unsigned eip);
int new_thread(task_t *task, unsigned eip);

/* from kernel linker script file */
extern unsigned char _code[], _d_code[], _data[];
extern unsigned char _d_data[], _bss[], _d_bss[], _end[];
/*****************************************************************************
*****************************************************************************/
static int kprintf_help(char c, void **ptr)
{
	putch(_curr_vc, c);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
void kprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	do_printf(fmt, args, kprintf_help, NULL);
	va_end(args);
}
/*****************************************************************************
*****************************************************************************/
static void panic(const char *fmt, ...)
{
	va_list args;

	disable();
	va_start(args, fmt);
	do_printf(fmt, args, kprintf_help, NULL);
	halt();
}
/*****************************************************************************
*****************************************************************************/
void fault(regs_t *regs)
{
	static const char *msg[] =
	{
		"divide error", "debug exception", "NMI", "INT3",
		"INTO", "BOUND exception", "invalid opcode", "no coprocessor",
		"double fault", "coprocessor segment overrun",
			"bad TSS", "segment not present",
		"stack fault", "GPF", "page fault", "??",
		"coprocessor error", "alignment check", "??", "??",
		"??", "??", "??", "??",
		"??", "??", "??", "??",
		"??", "??", "??", "??"
		"IRQ0", "IRQ1", "IRQ2", "IRQ3",
		"IRQ4", "IRQ5", "IRQ6", "IRQ7",
		"IRQ8", "IRQ9", "IRQ10", "IRQ11",
		"IRQ12", "IRQ13", "IRQ14", "IRQ15",
		"syscall"
	};
/**/

	switch(regs->which_int)
	{
/* page fault */
		case 0x0E:
			if(page_fault(regs->err_code) != 0)
				goto ERR;
			break;
/* timer (IRQ 0) */
		case 0x20:
			blink();
			timer_irq(regs->eip);
			break;
/* keyboard (IRQ 1) */
		case 0x21:
			kbd_irq();
			outportb(0x20, 0x20);
			break;
/* int 30h (syscall) */
		case SYSCALL_INT:
			if(!syscall(regs))
				return; /* no reschedule */
			break;
/* anything else */
		default:
ERR:			kprintf("Exception #%u ", regs->which_int);
			if(regs->which_int <= sizeof(msg) / sizeof(char *))
				kprintf("(%s) ", msg[regs->which_int]);
/* oops, it happened in kernel mode */
			if((regs->cs & 3) == 0)
			{
				kprintf("in kernel mode.\n");
				dump_regs(regs);
				panic("System halted\n");
			}
/* else user mode fault: kill the task */
			kprintf("in user mode; task %u killed\n",
				_curr_task - _tasks);
			sys_exit(-1);
			break;
	}
	schedule();
}
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(static void init_8259s(void))
{
	outportb(0x20, 0x11); /* ICW1 */
	outportb(0xA0, 0x11);

	outportb(0x21, 0x20); /* ICW2: route IRQs 0...7 to INTs 20h...27h */
	outportb(0xA1, 0x28); /* ...IRQs 8...15 to INTs 28h...2Fh */

	outportb(0x21, 0x04); /* ICW3 */
	outportb(0xA1, 0x02);

	outportb(0x21, 0x01); /* ICW4 */
	outportb(0xA1, 0x01);
/* enable IRQ0 (timer) and IRQ1 (keyboard) */
	outportb(0x21, ~0x03);
	outportb(0xA1, ~0x00);
}
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(static void init_8253(void))
{
/* I can remember the NTSC TV color burst frequency, but not the PC
peripheral clock. Fortunately, they are related: */
	static const unsigned short foo = (3579545L / 3) / HZ;
/**/

/* reprogram the 8253 timer chip to run at 'HZ', instead of 18 Hz */
	outportb(0x43, 0x36);	/* channel 0, LSB/MSB, mode 3, binary */
	outportb(0x40, foo & 0xFF);	/* LSB */
	outportb(0x40, foo >> 8);	/* MSB */
}
/*****************************************************************************
for MinGW32
*****************************************************************************/
#ifdef __WIN32__
int __main(void) { return 0; }
#endif
/*****************************************************************************
*****************************************************************************/
wait_queue_t _task_died;

static void buffy(void)
{
	unsigned i;

	while(1)
	{
		sleep_on(&_task_died, NULL);
		for(i = 0; i < MAX_TASK; i++)
		{
			if(_tasks[i].status == TS_ZOMBIE)
			{
				kprintf("buffy the dead thread slayer: "
					"found zombie task %u\n", i);
				_tasks[i].status = 0;
				free_task_pages(_tasks[i].page_dir);
			}
		}
	}
}
/*****************************************************************************
*****************************************************************************/
int main(void)
{
	unsigned keep, discard, temp;

	init_video();
	kprintf("\x1B[31m""C""\x1B[32m""o""\x1B[33m""s""\x1B[34m""m"
		"\x1B[35m""o""\x1B[36m""s""\x1B[37m"" OS release 10 "
		"- Copyright (C) 2002 Chris Giese <geezer@execpc.com>\n");
	kprintf("%uK conventional memory, %uK extended memory. "
		"Virt-to-phys=0x%X,\n", _conv_mem_size >> 10,
		_ext_mem_size >> 10, _kvirt_to_phys);
	kprintf("kernel virtual address=0x%X, physical address=0x%X\n",
		_code, _code + _kvirt_to_phys);
	keep = (_d_code - _code) + (_d_data - _data) + (_d_bss - _bss);
	discard = (_data - _d_code) + (_bss - _d_data) + (_end - _d_bss);
	kprintf("Kernel memory:\tcode\tdata\tbss\tTOTAL\n"
		"\tTOTAL\t%u\t%u\t%u\t%u\n"
		"\tDISCARD\t%u\t%u\t%u\t%u\n"
		"\tKEEP\t%u\t%u\t%u\t%u\t(all values in bytes)\n",
		_data - _code,	_bss - _data,	_end - _bss, keep + discard,
		_data - _d_code,_bss - _d_data,	_end - _d_bss,	discard,
		_d_code - _code,_d_data - _data,_d_bss - _bss,	keep);
	init_kbd();
	init_8259s();
	init_8253();
	temp = init_paging();
	if(temp != 0)
	{
		kprintf("init_paging returned %d\n", temp);
		halt();
	}
/* need these to make memory management work */
	_curr_task = _tasks + 0;
	_curr_task->page_dir = get_page_dir();
	_curr_task->status = TS_RUNNABLE;
//dump_page_tables();
//while(1);

/* OK, NOW set up VCs */
	init_vc();
/* init tasks */
	if(_init_ramdisk_adr == 0)
	{
		kprintf("no initial RAM disk found so no tasks to run\n");
		halt();
	}
	temp = init_tasks((unsigned char *)_init_ramdisk_adr);
	if(temp != 0)
	{
		kprintf("sorry, did not find any tasks to run\n");
		halt();
	}
/* done with init, can now discard init-only code/data */
	discard_mem();
/* kernel threads */
	kprintf("Kernel threads:\n\tidle (0)\n");

	kprintf("\tbuffy (%u)\n", MAX_TASK - 1);
	(void)new_thread(_tasks + MAX_TASK - 1, (unsigned)buffy);
	_tasks[MAX_TASK - 1].status = TS_RUNNABLE;

	kprintf("\tkeyboard bottom-half interrupt handler (%u)\n",
		MAX_TASK - 2);
	(void)new_thread(_tasks + MAX_TASK - 2, (unsigned)kbd_bh);
	_tasks[MAX_TASK - 2].status = TS_RUNNABLE;
	_tasks[MAX_TASK - 2].priority = +1;

	kprintf("Press F1, F2, etc. to select virtual console, "
		"Ctrl+Alt+Del to reboot\n");
/* eat stray keypresses to prevent keyboard freezing up */
	(void)inportb(0x60);
/* give the idle thread reduced priority */
	_curr_task->priority = -1;
/* enabling timer interrupt starts the scheduler */
	enable();
/* freeze */
	while(1)
		halt();
	return 0;
}
