;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; EXPORTS:
;
; extern unsigned _conv_mem_size, _ext_mem_size;
; extern unsigned _init_ramdisk_adr, _init_ramdisk_size;
;
; void halt(void);
; unsigned get_page_fault_adr(void);
; unsigned get_page_dir(void);
; void set_page_dir(unsigned cr3);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

%include "asm32.inc"

; IMPORTS
; from linker script:
IMP _code, _data, _bss, _end

DS_MAGIC	equ	3544DA2Ah

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; pmode entry point
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

[SECTION .dtext]		; discardable kernel code
[BITS 32]

EXP entry
; where did the loader put us?
	call where_am_i
where_am_i:
	pop esi			; ESI=physical adr (where we are)
	sub esi,where_am_i	; subtract virtual adr (where we want to be)
				; now ESI=virt-to-phys

; all data segment accesses must be relative to ESI until we activate paging
; virt_adr + virt_to_phys = virt_adr + (phys - virt) = phys_adr
; phys_adr - virt_to_phys = phys_adr - (phys - virt) = virt_adr

; check if data segment linked/located/loaded properly
	mov eax,[esi + ds_magic]
	cmp eax,DS_MAGIC
	je ds_ok

; display a blinking white-on-blue 'D' if bad data segment
	mov word [0B8000h],9F44h
	jmp short $
ds_ok:

; save kernel virt_to_phys conversion value
	mov [esi + _kvirt_to_phys],esi

; set up kernel page tables
	call init_paging
	jnc paging_ok

; display a blinking white-on-blute 'P' if error setting up page tables
	mov word [0B8000h],9F50h
	jmp short $

paging_ok:
	lea eax,[esi + task0_page_dir]
	mov cr3,eax

	mov eax,cr0
;	or eax,080000000h	; xxx - make sure you have a 486+
;	or eax,080010000h	; set the 486 WP (page write protect) bit
	or eax,0C0010000h	; set the 486 CE (cache enable) and WP bits
	mov cr0,eax
	jmp short paging_enabled ; near, EIP-relative JMP; to physical address

paging_enabled:
	mov ebx,virtual_addresses_enabled
	jmp ebx			; near, absolute JMP; to virtual address

virtual_addresses_enabled:
	lgdt [gdt_ptr]		; stop using bootloader GDT; load new GDT
	jmp SYS_CODE_SEL:new_gdt_loaded ; far, absolute JMP

new_gdt_loaded:
	mov ax,SYS_DATA_SEL
	mov ds,eax
	mov ss,eax
	mov es,eax
	mov fs,eax
	mov gs,eax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; from this point on, we can use absolute addresses
; for items in the data and BSS segments
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; set up pmode stack
	mov esp,stack

; set up TSS descriptor, then load task register
	mov eax,tss
	mov [gdt1 + 2],ax
	shr eax,16
	mov [gdt1 + 4],al
	mov [gdt1 + 7],ah
	mov ax,TSS_SEL
	ltr ax

; set up interrupt handlers, then load IDT register
	mov ecx,(idt_end - idt) >> 3 ; number of exception handlers
	mov ebx,idt
	mov edx,isr0
do_idt:
	mov eax,edx		; EAX=offset of entry point
	mov [ebx],ax		; set low 16 bits of gate offset
	shr eax,16
	mov [ebx + 6],ax	; set high 16 bits of gate offset
	add ebx,8		; 8 bytes per interrupt gate
	add edx,(isr1 - isr0)	; N bytes per stub
	loop do_idt
	lidt [idt_ptr]

IMP main
	call main		; call C code
	jmp $			; freeze

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			map_page
; action:		maps one page into the page tables
; in:			EAX=page virtual address
;			EBX=page directory physical address
;			DX=page privilege
;			EDI=page physical address
; out (error):		CY=1 if a necessary page table is not installed
; out (success):	CY=0
; modifies:		(nothing)
; notes:		all address translation must be turned off when
;			this function is called: no paging,
;			and base address of data segment == 0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

map_page:
	push ebx
	push eax

; get index-into-page-directory from page virtual address
		shr eax,22

; add index to page directory physical address
		shl eax,2
		add ebx,eax
	pop eax
	push eax

; get page directory entry (PDE)
		mov ebx,[ebx]

; convert PDE to page table physical address
; error if no page table installed
		and ebx,0FFFFF000h
		stc
		je map_page_err

; get index-into-page-table from page virtual address
		shr eax,12
		and eax,000003FFh
		shl eax,2

; add index to page table physical address, forming physical
; address of page table entry (PTE)
		add ebx,eax

; create PTE from page physical address and privilege
		mov eax,edi
		and eax,0FFFFF000h
		or ax,dx
; store PTE
		mov [ebx],eax
map_page_err:
	pop eax
	pop ebx
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			map_pages
; action:		maps a range of pages with contiguous virtual
;			and physical addresses into the page tables
; in:			EAX=first page virtual address
;			EBX=page directory physical address
;			ECX=range size, in bytes
;			DX=page privilege
;			EDI=first page physical address
; out (error):		CY=1 if a necessary page table is not installed
; out (success):	CY=0
; modifies:		(nothing)
; notes:		all address translation must be turned off when
;			this function is called: no paging,
;			and base address of data segment == 0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

map_pages:
	pusha
		and eax,0FFFFF000h
		and ebx,0FFFFF000h
		and ecx,0FFFFF000h
		je map_pages_2
map_pages_1:
		call map_page
		jc map_pages_2
		add edi,4096
		add eax,4096
		sub ecx,4096
		jne map_pages_1
map_pages_2:
	popa
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			init_paging
; action:
; in:			ESI=kernel virt-to-phys
; out (error):		CY=1 if a necessary page table is not installed
; out (success):	CY=0
; modifies:		(nothing)
; notes:		all address translation must be turned off when
;			this function is called: no paging,
;			and base address of data segment == 0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

init_paging:
	pusha

; get physical address of page dir to EBX
		lea ebx,[esi + task0_page_dir]

; SET UP PAGE DIRECTORY:
; make an entry for conv_mem_page_table
		lea eax,[esi + conv_mem_page_table]
		or al,7		; ring 3, writable, present
		mov [ebx + 0],eax

; get physical address of kernel to EDI, convert to page dir index
		mov edi,_code
		add edi,esi
		shr edi,22
		shl edi,2

; make entries for im_kernel_page_tables. If im_kernel_page_table_1
; maps the same memory as conv_mem_page_table, then use conv_mem_page_table
		xor eax,eax
		or eax,[ebx + edi]
		jne init_paging_1

		lea eax,[esi + im_kernel_page_table_1]
		or al,7		; ring 3, writable, present
		mov [ebx + edi],eax
init_paging_1:
		lea eax,[esi + im_kernel_page_table_2]
		or al,7		; ring 3, writable, present
		mov [ebx + edi + 4],eax

; get virtual address of kernel to EDI, convert to page dir index
		mov edi,_code
		shr edi,22
		shl edi,2

; make entries for kernel_page_tables
		xor eax,eax
		or eax,[ebx + edi]
		lea eax,[esi + conv_mem_page_table]
		jne init_paging_2
		lea eax,[esi + kernel_page_table_1]
init_paging_2:
		or al,7		; ring 3, writable, present
		mov [ebx + edi],eax

		lea eax,[esi + kernel_page_table_2]
		or al,7		; ring 3, writable, present
		mov [ebx + edi + 4],eax

; make an entry for page dir itself
; xxx - maybe the 4092 should be a named constant, not a magic value
		lea eax,[esi + task0_page_dir]
		or al,7		; ring 3, writable, present
		mov [ebx + 4092],eax

; SET UP PAGE TABLES
; identity-map conventional RAM read-write
; this includes the BIOS data segment in page 0 and the RDSK file
		mov dx,3	; privilege = ring 0, writable, present
		xor eax,eax	; virtual address = 0
		mov edi,eax	; physical address is the same
		mov ecx,[esi + _conv_mem_size]
		call map_pages
		jc near init_paging_err

; identity-map video memory (framebuffer) read-write
		mov eax,0A0000h	; virtual address
		mov edi,eax	; physical address is the same
		mov ecx,20000h	; 0xA0000 to 0xBFFFF
		call map_pages
		jc near init_paging_err

; identity-map BIOS ROMs read-only
; xxx - what I'd really like to do here is:
; 1. find location and size of video BIOS ROM and map it read-only
; 2. find location and size of motherboard BIOS ROM and map it read-only
; 3. leave other memory unmapped, mapping it later if necessary
;	e.g. for Ethernet card with shared memory
		mov dx,1	; privilege = ring 0, read-only, present
		mov eax,0C0000h	; virtual address
		mov edi,eax	; physical address is the same
		mov ecx,40000h	; 0xC0000 to 0xFFFFF
		call map_pages
		jc near init_paging_err
mov byte [0B8002h],'d'

; identity-map kernel code (and possibly rodata) read-only
		mov dx,1	; privilege = ring 0, read-only, present
		mov edi,_code
		add edi,esi	; physical address
		mov eax,edi	; virtual address is the same
		mov ecx,_data
		sub ecx,_code
		call map_pages
		jc init_paging_err
; xxx - fails here
mov byte [0B8002h],'e'

; identity-map kernel data and BSS read-write
		mov dx,3	; privilege = ring 0, writable, present
		mov edi,_data
		add edi,esi	; physical address
		mov eax,edi	; virtual address is the same
		mov ecx,_end
		sub ecx,_data
		call map_pages
		jc init_paging_err
mov byte [0B8002h],'f'

; map kernel code (and possibly rodata) read-only
		mov dx,1	; privilege = ring 0, read-only, present
		mov eax,_code	; virtual address
		mov edi,eax
		add edi,esi	; physical address
		mov ecx,_data
		sub ecx,_code
		call map_pages
		jc init_paging_err
mov byte [0B8002h],'g'

; map kernel data and BSS read-write
		mov dx,3	; privilege = ring 0, writable, present
		mov eax,_data	; virtual address
		mov edi,eax
		add edi,esi	; physical address
		mov ecx,_end
		sub ecx,_data
		call map_pages
jc init_paging_err
mov byte [0B8002h],'h'
init_paging_err:
	popa
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; regular (non-discardable) kernel code
[SECTION .text]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			halt
; action:		halts CPU until an interrupt occurs
; in:			(nothing)
; out:			(nothing)
; modifies:		(nothing)
; notes:		system will freeze if interrupts are disabled
;			C prototype: void halt(void);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

EXP halt
	hlt
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			get_page_fault_adr
; action:		gets virtual address of latest page fault
;			from register CR2
; in:			(nothing)
; out:			EAX=page fault address
; modifies:		EAX
; notes:		C prototype: unsigned get_page_fault_adr(void);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

EXP get_page_fault_adr
	mov eax,cr2
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			get_page_dir
; action:		reads physical address of current page directory
;			from CPU register CR3
; in:			(nothing)
; out:			EAX=page directory address
; modifies:		EAX
; notes:		C prototype: unsigned get_page_dir(void);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

EXP get_page_dir
	mov eax,cr3
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			set_page_dir
; action:		stores physical address of new page directory
;			in CPU register CR3
; in:			[esp + 8] (left-most argument) is new CR3 value
; out:			(nothing)
; modifies:		(nothing)
; notes:		this flushes the entire TLB (the page table cache)
;			C prototype: void set_page_dir(unsigned cr3);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

EXP set_page_dir
	push eax		; we declared it void; must save EAX
		mov eax,[esp + 8]
		mov cr3,eax
	pop eax
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; interrupt/exception handlers
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

IMP _curr_task
IMP fault

all_ints:
	push gs				; push segment registers
	push fs
	push es
	push ds
	pusha				; push 8 general-purpose registers
		mov ax,SYS_DATA_SEL
		mov ds,eax		; put known-good values in seg regs
		mov es,eax
		mov fs,eax
		mov gs,eax

		mov eax,esp		; push old _curr_task
		push dword [_curr_task]
		push eax		; push pointer to stacked regs_t
			call fault	; call C language handler
		pop ecx			; drop pointer to stacked regs_t
		pop ebx			; get old _curr_task
		mov esi,[_curr_task]	; get new _curr_task
		mov [ebx + 0],esp	; save old ESP
		mov esp,[esi + 0]	; load new ESP
		mov eax,[esi + 4]	; get new CR3
		cmp eax,[ebx + 4]	; changing CR3?
		je no_switch
		mov cr3,eax
no_switch:

; I guess the task-switch doesn't automatically store ESP
; in TSS.ESP0, so we do it ourselves
;
; if we IRET to ring 0 code, we don't pop SS and ESP, so this should be
;	lea eax,[esp + 68]
; Right? Actually, if we IRET to ring 0 code this value is not used,
; so it doesn't matter if it's wrong
		lea eax,[esp + 76]
		mov [tss_esp0],eax
	popa				; pop 8 general-purpose registers
	pop ds				; pop segment registers
	pop es
	pop fs
	pop gs
	add esp,8			; drop exception number and error code
	iret

; the stack is always 32 bits wide in 32-bit pmode
; the push will zero-extend (sign-extend?) the byte to 32 bits
isr0:
	push byte 0
	push byte 0
	jmp all_ints		; zero divide (fault)
isr1:
	push byte 0
	push byte 1
	jmp all_ints		; debug/single step
isr2:
	push byte 0
	push byte 2
	jmp all_ints		; non-maskable interrupt (trap)
isr3:
	push byte 0
	push byte 3
	jmp all_ints		; INT3 (trap)
isr4:
	push byte 0
	push byte 4
	jmp all_ints		; INTO (trap)
isr5:
	push byte 0
	push byte 5
	jmp all_ints		; BOUND (fault)
isr6:
	push byte 0
	push byte 6
	jmp all_ints		; invalid opcode (fault)
isr7:
	push byte 0
	push byte 7
	jmp all_ints		; coprocessor not available (fault)
isr8:
	nop
	nop
	push byte 8
	jmp all_ints		; double fault (abort w/ error code)
isr9:
	push byte 0
	push byte 9
	jmp all_ints		; coproc segment overrun (abort; 386/486SX only)
isr0A:
	nop
	nop
	push byte 0Ah
	jmp all_ints		; bad TSS (fault w/ error code)
isr0B:
	nop
	nop
	push byte 0Bh
	jmp all_ints		; segment not present (fault w/ error code)
isr0C:
	nop
	nop
	push byte 0Ch
	jmp all_ints		; stack fault (fault w/ error code)
isr0D:
	nop
	nop
	push byte 0Dh
	jmp all_ints		; GPF (fault w/ error code)
isr0E:
	nop
	nop
	push byte 0Eh
	jmp all_ints		; page fault
isr0F:
	push byte 0
	push byte 0Fh
	jmp all_ints		; reserved
isr10:
	push byte 0
	push byte 10h
	jmp all_ints		; FP exception/coprocessor error (trap)
isr11:
	push byte 0
	push byte 11h
	jmp all_ints		; alignment check (trap; 486+ only)
isr12:
	push byte 0
	push byte 12h
	jmp all_ints		; machine check (Pentium+ only)
isr13:
	push byte 0
	push byte 13h
	jmp all_ints
isr14:
	push byte 0
	push byte 14h
	jmp all_ints
isr15:
	push byte 0
	push byte 15h
	jmp all_ints
isr16:
	push byte 0
	push byte 16h
	jmp all_ints
isr17:
	push byte 0
	push byte 17h
	jmp all_ints
isr18:
	push byte 0
	push byte 18h
	jmp all_ints
isr19:
	push byte 0
	push byte 19h
	jmp all_ints
isr1A:
	push byte 0
	push byte 1Ah
	jmp all_ints
isr1B:
	push byte 0
	push byte 1Bh
	jmp all_ints
isr1C:
	push byte 0
	push byte 1Ch
	jmp all_ints
isr1D:
	push byte 0
	push byte 1Dh
	jmp all_ints
isr1E:
	push byte 0
	push byte 1Eh
	jmp all_ints
isr1F:
	push byte 0
	push byte 1Fh
	jmp all_ints

; isr20 through isr2F are hardware interrupts. The 8259 programmable
; interrupt controller (PIC) chips must be reprogrammed to make these work.
isr20:
	push byte 0
	push byte 20h
	jmp all_ints		; IRQ 0/timer interrupt
isr21:
	push byte 0
	push byte 21h
	jmp all_ints		; IRQ 1/keyboard interrupt
isr22:
	push byte 0
	push byte 22h
	jmp all_ints
isr23:
	push byte 0
	push byte 23h
	jmp all_ints
isr24:
	push byte 0
	push byte 24h
	jmp all_ints
isr25:
	push byte 0
	push byte 25h
	jmp all_ints
isr26:
	push byte 0
	push byte 26h
	jmp all_ints		; IRQ 6/floppy interrupt
isr27:
	push byte 0
	push byte 27h
	jmp all_ints
isr28:
	push byte 0
	push byte 28h
	jmp all_ints		; IRQ 8/real-time clock interrupt
isr29:
	push byte 0
	push byte 29h
	jmp all_ints
isr2A:
	push byte 0
	push byte 2Ah
	jmp all_ints
isr2B:
	push byte 0
	push byte 2Bh
	jmp all_ints
isr2C:
	push byte 0
	push byte 2Ch
	jmp all_ints
isr2D:
	push byte 0
	push byte 2Dh
	jmp all_ints		; IRQ 13/math coprocessor interrupt
isr2E:
	push byte 0
	push byte 2Eh
	jmp all_ints		; IRQ 14/primary ATA ("IDE") drive interrupt
isr2F:
	push byte 0
	push byte 2Fh
	jmp all_ints		; IRQ 15/secondary ATA drive interrupt

; syscall software interrupt
isr30:
	push byte 0
	push byte 30h
	jmp all_ints

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

[SECTION .data]

ds_magic:
	dd DS_MAGIC

; 32 ring 0 interrupt gates
idt:
	times 48	dw 0, SYS_CODE_SEL, 8E00h, 0

; one ring 3 interrupt gate for syscalls (INT 30h)
	dw 0			; offset 15:0
	dw SYS_CODE_SEL		; selector
	db 0			; (always 0 for interrupt gates)
	db 0EEh			; present,ring 3,'386 interrupt gate
	dw 0			; offset 31:16
idt_end:

idt_ptr:
	dw idt_end - idt - 1	; IDT limit
	dd idt			; linear adr of IDT (set above)

; we don't use the TSS for task-switching, but we still need it to store
; the kernel (ring 0) stack pointer while user (ring 3) code is running
; (also need it for the I/O permission bitmap)
tss:
	dw 0, 0			; back link
tss_esp0:
	dd 0			; ESP0
	dw SYS_DATA_SEL, 0	; SS0, reserved

	dd 0			; ESP1
	dw 0, 0			; SS1, reserved

	dd 0			; ESP2
	dw 0, 0			; SS2, reserved

	dd 0			; CR3
	dd 0, 0			; EIP, EFLAGS
	dd 0, 0, 0, 0		; EAX, ECX, EDX, EBX
	dd 0, 0, 0, 0		; ESP, EBP, ESI, EDI
	dw 0, 0			; ES, reserved
	dw 0, 0			; CS, reserved
	dw 0, 0			; SS, reserved
	dw 0, 0			; DS, reserved
	dw 0, 0			; FS, reserved
	dw 0, 0			; GS, reserved
	dw 0, 0			; LDT, reserved
	dw 0, 103		; debug, IO permission bitmap base

; null descriptor. gdt_ptr could be put here to save a few
; bytes, but that can be confusing.
gdt:
	dw 0			; limit 15:0
	dw 0			; base 15:0
	db 0			; base 23:16
	db 0			; type
	db 0			; limit 19:16, flags
	db 0			; base 31:24

; descriptor for task-state segment (TSS)
TSS_SEL		equ	$-gdt
gdt1:
	dw 103
	dw 0
	db 0
	db 89h			; ring 0 available 32-bit TSS
	db 0
	db 0

; ring 0 kernel code segment descriptor. Segmentation is deprecated
; by setting the segment limits to 4Gig - 1 and the segment bases to 0.
SYS_CODE_SEL	equ	$-gdt
gdt2:
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah			; present,ring 0,code,non-conforming,readable
	db 0CFh
	db 0

; ring 0 kernel data segment descriptor
SYS_DATA_SEL	equ	$-gdt
gdt3:
	dw 0FFFFh
	dw 0
	db 0
	db 92h			; present,ring 0,data,expand-up,writable
	db 0CFh
	db 0

; ring 3 user code segment descriptor
USER_CODE_SEL	equ	($-gdt) | 3
	dw 0FFFFh
	dw 0
	db 0
	db 0FAh			; present,ring 3,code,non-conforming,readable
	db 0CFh
	db 0

; ring 3 user data segment descriptor
USER_DATA_SEL	equ	($-gdt) | 3
	dw 0FFFFh
	dw 0
	db 0
	db 0F2h			; present,ring 3,data,expand-up,writable
	db 0CFh
	db 0

; linear data segment descriptor
LINEAR_DATA_SEL	equ	$-gdt
	dw 0FFFFh
	dw 0
	db 0
	db 92h			; present,ring 0,data,expand-up,writable
	db 0CFh
	db 0

; linear code segment descriptor
LINEAR_CODE_SEL	equ	$-gdt
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah			; present,ring 0,code,non-conforming,readable
	db 0CFh
	db 0
gdt_end:

gdt_ptr:
	dw gdt_end - gdt - 1	; GDT limit
	dd gdt			; linear adr of GDT (set above)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

[SECTION .bss]

; /------- START OF 4K BOOT DATA
EXP _conv_mem_size		; conventional memory size (bytes)
	resd 1

EXP _ext_mem_size		; extended memory size (bytes)
	resd 1

EXP _init_ramdisk_adr		; where the initial RAM disk is loaded
	resd 1

EXP _init_ramdisk_size		; size of initial RAM disk
	resd 1

	resd (1024 - 4)		; padding to 4K
; \------- END OF 4K BOOT DATA

; *** WARNING ***: page tables must be aligned to page (4K) boundaries:
; 1. make sure the linker script aligns the BSS to a page boundary
; 2. make sure the bootloader loads the kernel to a page boundary
; 3. make sure that anything placed before the page tables in the BSS
;    (i.e. the boot data above) is a multiple of 4K in size.

task0_page_dir:			; initial page directory
	times 1024 resd 1
conv_mem_page_table:		; page table to identity-map <= 4 meg RAM
	times 1024 resd 1

;im_kernel_page_table_1:		; page tables to identity-map kernel
;	times 1024 resd 1
;im_kernel_page_table_2:
;	times 1024 resd 1

kernel_page_table_1:		; page tables to map kernel
	times 1024 resd 1
kernel_page_table_2:
	times 1024 resd 1

; task #0 kernel stack
	resd 1024
stack:

EXP _kvirt_to_phys
	resd 1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

[SECTION .dbss bss]

im_kernel_page_table_1:		; page tables to identity-map kernel
	times 1024 resd 1
im_kernel_page_table_2:
	times 1024 resd 1
