/*****************************************************************************
PC KEYBOARD

EXPORTS:
int deq(queue_t *q, unsigned char *data);
void kbd_bh(void);
void kbd_irq(void);
void init_kbd(void);
*****************************************************************************/
#include <conio.h> /* KEY_... */
#include <_krnl.h>
#include <_x86.h> /* inportb(), outportb() */

/* IMPORTS
from STARTUP.ASM */
void halt(void);

/* from VIDEO.C */
extern console_t _vc[];
extern console_t *_curr_vc;

void select_vc(unsigned which_con);
void putch(console_t *con, unsigned c);

/* from MAIN.C */
void kprintf(const char *fmt, ...);
void wake_up(wait_queue_t *queue);
int sleep_on(wait_queue_t *queue, unsigned *timeout);

/* "raw" set 1 scancodes from PC keyboard. Keyboard info here:
http://www.execpc.com/~geezer/osd/kbd */
#define	RAW1_LEFT_CTRL		0x1D
#define	RAW1_RIGHT_CTRL		0x1D	/* same as left */
#define	RAW1_LEFT_SHIFT		0x2A
#define	RAW1_RIGHT_SHIFT	0x36
#define	RAW1_LEFT_ALT		0x38
#define	RAW1_RIGHT_ALT		0x38	/* same as left */
#define	RAW1_CAPS_LOCK		0x3A
#define	RAW1_F1			0x3B
#define	RAW1_F2			0x3C
#define	RAW1_F3			0x3D
#define	RAW1_F4			0x3E
#define	RAW1_F5			0x3F
#define	RAW1_F6			0x40
#define	RAW1_F7			0x41
#define	RAW1_F8			0x42
#define	RAW1_F9			0x43
#define	RAW1_F10		0x44
#define	RAW1_NUM_LOCK		0x45
#define	RAW1_SCROLL_LOCK	0x46
#define	RAW1_F11		0x57
#define	RAW1_F12		0x58

#define	KBD_BUF_SIZE		64

static queue_t _main_kbd_queue;
static wait_queue_t _keyboard_bh;
/*****************************************************************************
*****************************************************************************/
static void reboot(void)
{
	unsigned temp;

	disable();
/* flush the keyboard controller */
	do
	{
		temp = inportb(0x64);
		if((temp & 0x01) != 0)
		{
			(void)inportb(0x60);
			continue;
		}
	} while((temp & 0x02) != 0);
/* pulse the CPU reset line */
	outportb(0x64, 0xFE);
/* ...and if that didn't work, just halt */
	halt();
}
/*****************************************************************************
xxx - the queues need more thought -- code is ugly

	name:	inq
	action:	tries to add data (a byte) to queue
	returns:-1 if queue full
		0  if success
*****************************************************************************/
static int inq(queue_t *q, unsigned char data)
{
	unsigned temp;

	temp = q->in_ptr + 1;
	if(temp >= q->size)
		temp = 0;
/* if in_ptr reaches out_ptr, the queue is full */
	if(temp == q->out_ptr)
		return -1;
	q->data[q->in_ptr] = data;
	q->in_ptr = temp;
	return 0;
}
/*****************************************************************************
"unbuffered"; advances q->in_base in lockstep with q->in_ptr
*****************************************************************************/
static int ub_inq(queue_t *q, unsigned char data)
{
	unsigned temp;

	temp = q->in_ptr + 1;
	if(temp >= q->size)
		temp = 0;
/* if in_ptr reaches out_ptr, the queue is full */
	if(temp == q->out_ptr)
		return -1;
	q->data[q->in_ptr] = data;
	q->in_ptr = q->in_base = temp;
	return 0;
}
/*****************************************************************************
	name:	deq
	action:	tries to get byte from queue
	returns:-1 if queue empty
		0  if success (data set to value read from queue)
*****************************************************************************/
int deq(queue_t *q, unsigned char *data)
{
/* if out_ptr reaches in_base, the queue is empty */
	if(q->out_ptr == q->in_base)
		return -1;
	*data = q->data[q->out_ptr++];
	if(q->out_ptr >= q->size)
		q->out_ptr = 0;
	return 0;
}
/*****************************************************************************
	name:	write_kbd
	action:	writes data to 8048 keyboard MCU (adr=0x60)
		or 8042 controller MCU (adr=0x64)
*****************************************************************************/
static void write_kbd(unsigned adr, unsigned char data)
{
	unsigned long timeout;
	unsigned char stat;

/* Linux code didn't have a timeout here... */
	for(timeout = 500000L; timeout != 0; timeout--)
	{
		stat = inportb(0x64);
/* loop until 8042 input buffer empty */
		if((stat & 0x02) == 0)
			break;
	}
	if(timeout != 0)
		outportb(adr, data);
}
/*****************************************************************************
	name:	convert
	action:	converts raw set 1 scancodes to pseudo-ASCII
	returns:0 if nothing to return, else 8-bit "ASCII" value
*****************************************************************************/
static unsigned convert(unsigned key)
{
	static const unsigned char set1_map[] =
	{
/* 00 */0,	0x1B,	'1',	'2',	'3',	'4',	'5',	'6',
/* 08 */'7',	'8',	'9',	'0',	'-',	'=',	'\b',	'\t',
/* 10 */'q',	'w',	'e',	'r',	't',	'y',	'u',	'i',
/* 1Dh is left Ctrl */
/* 18 */'o',	'p',	'[',	']',	'\n',	0,	'a',	's',
/* 20 */'d',	'f',	'g',	'h',	'j',	'k',	'l',	';',
/* 2Ah is left Shift */
/* 28 */'\'',	'`',	0,	'\\',	'z',	'x',	'c',	'v',
/* 36h is right Shift */
/* 30 */'b',	'n',	'm',	',',	'.',	'/',	0,	0,
/* 38h is left Alt, 3Ah is Caps Lock */
/* 38 */0,	' ',	0,	KEY_F1,	KEY_F2,	KEY_F3,	KEY_F4,	KEY_F5,
/* 45h is Num Lock, 46h is Scroll Lock */
/* 40 */KEY_F6,	KEY_F7,	KEY_F8,	KEY_F9,	KEY_F10,0,	0,	KEY_HOME,
/* 48 */KEY_UP,	KEY_PGUP,'-',	KEY_LFT,'5',	KEY_RT,	'+',	KEY_END,
/* 50 */KEY_DN,	KEY_PGDN,KEY_INS,KEY_DEL,0,	0,	0,	KEY_F11,
/* 58 */KEY_F12
	};
	static unsigned short kbd_status, saw_break_code;
/**/
	unsigned short temp;

/* check for break key (i.e. a key is released) */
	if(key >= 0x80)
	{
		saw_break_code = 1;
		key &= 0x7F;
	}
/* the only break codes we're interested in are Shift, Ctrl, Alt */
	if(saw_break_code)
	{
		if(key == RAW1_LEFT_ALT || key == RAW1_RIGHT_ALT)
			kbd_status &= ~KBD_META_ALT;
		else if(key == RAW1_LEFT_CTRL || key == RAW1_RIGHT_CTRL)
			kbd_status &= ~KBD_META_CTRL;
		else if(key == RAW1_LEFT_SHIFT || key == RAW1_RIGHT_SHIFT)
			kbd_status &= ~KBD_META_SHIFT;
		saw_break_code = 0;
		return 0;
	}
/* it's a make key: check the "meta" keys, as above */
	if(key == RAW1_LEFT_ALT || key == RAW1_RIGHT_ALT)
	{
		kbd_status |= KBD_META_ALT;
		return 0;
	}
	if(key == RAW1_LEFT_CTRL || key == RAW1_RIGHT_CTRL)
	{
		kbd_status |= KBD_META_CTRL;
		return 0;
	}
	if(key == RAW1_LEFT_SHIFT || key == RAW1_RIGHT_SHIFT)
	{
		kbd_status |= KBD_META_SHIFT;
		return 0;
	}
/* Scroll Lock, Num Lock, and Caps Lock set the LEDs. These keys
have on-off (toggle or XOR) action, instead of momentary action */
	if(key == RAW1_SCROLL_LOCK)
	{
		kbd_status ^= KBD_META_SCRL;
		goto LEDS;
	}
	if(key == RAW1_NUM_LOCK)
	{
		kbd_status ^= KBD_META_NUM;
		goto LEDS;
	}
	if(key == RAW1_CAPS_LOCK)
	{
		kbd_status ^= KBD_META_CAPS;
LEDS:		write_kbd(0x60, 0xED);	/* "set LEDs" command */
		temp = 0;
		if(kbd_status & KBD_META_SCRL)
			temp |= 1;
		if(kbd_status & KBD_META_NUM)
			temp |= 2;
		if(kbd_status & KBD_META_CAPS)
			temp |= 4;
		write_kbd(0x60, temp);	/* bottom 3 bits set LEDs */
		return 0;
	}
/* ignore invalid scan codes */
	if(key >= sizeof(set1_map) / sizeof(set1_map[0]))
		return 0;
/* convert raw scancode in key to unshifted ASCII in temp */
	temp = set1_map[key];
/* defective keyboard? non-US keyboard? more than 104 keys? */
	if(temp == 0)
		return temp;
/* handle the three-finger salute */
	if((kbd_status & KBD_META_CTRL) && (kbd_status & KBD_META_ALT) &&
		(temp == KEY_DEL))
	{
		kprintf("\n""\x1B[42;37;1m""*** rebooting!");
		reboot();
	}
/* I really don't know what to do yet with Alt, Ctrl, etc. -- punt */
	return temp;
}
/*****************************************************************************
done:
	8=BS	13=CR
to do:
	4=EOF	7=BEL	9=TAB	10=LF	12=FF	?=CAN (erase line)
*****************************************************************************/
static void do_console(unsigned char key)
{
	unsigned temp;
	queue_t *q;

	q = &_curr_vc->keystrokes;
	if(_curr_vc->unbuffered)
	{
		if(ub_inq(q, key) != 0)
			/* full queue, beep or something */;
		wake_up(&_curr_vc->wait_queue);
	}
	else
	{
/* backspace */
		if(key == 8)
		{
/* if in_ptr equals in_base, the input side of the queue is empty
xxx - make a function to check if input side of queue is empty? */
			if(q->in_ptr != q->in_base)
			{
				if(q->in_ptr == 0)
					q->in_ptr = q->size - 1;
				else
					q->in_ptr--;
				putch(_curr_vc, key);
				putch(_curr_vc, ' ');
				putch(_curr_vc, key);
			}
		}
		else if(key == '\n')
		{
			if(ub_inq(q, key) || inq(q, '\0'))
				return;/* xxx - should not happen */
			putch(_curr_vc, key);
			putch(_curr_vc, '\r');
			wake_up(&_curr_vc->wait_queue);
		}
		else if(key >= ' ')
		{
/* need room for 3 bytes (key, and CR and NUL that are stored by Enter)
xxx - make a function to check if enough room in queue */
			temp = q->in_ptr + 1;
			if(temp >= q->size)
				temp = 0;
/* if in_ptr reaches out_ptr, the queue is full */
			if(temp == q->out_ptr)
				return;

			temp = q->in_ptr + 1;
			if(temp >= q->size)
				temp = 0;
			if(temp == q->out_ptr)
				return;

			temp = q->in_ptr + 1;
			if(temp >= q->size)
				temp = 0;
			if(temp == q->out_ptr)
				return;

			if(inq(q, key))
				return;/* xxx - should not happen */
			putch(_curr_vc, key);
		}
	}
}
/*****************************************************************************
*****************************************************************************/
void kbd_bh(void)
{
	unsigned char key;
	unsigned temp;

	while(1)
	{
		sleep_on(&_keyboard_bh, 0);
		if(deq(&_main_kbd_queue, &key) != 0)
			continue;
/* if it's F1, F2 etc. switch to the appropriate virtual console */
		switch(key)
		{
		case RAW1_F1:
			temp = 0;
			goto SWITCH_VC;
		case RAW1_F2:
			temp = 1;
			goto SWITCH_VC;
		case RAW1_F3:
			temp = 2;
			goto SWITCH_VC;
		case RAW1_F4:
			temp = 3;
			goto SWITCH_VC;
		case RAW1_F5:
			temp = 4;
			goto SWITCH_VC;
		case RAW1_F6:
			temp = 5;
			goto SWITCH_VC;
		case RAW1_F7:
			temp = 6;
			goto SWITCH_VC;
		case RAW1_F8:
			temp = 7;
			goto SWITCH_VC;
		case RAW1_F9:
			temp = 8;
			goto SWITCH_VC;
		case RAW1_F10:
			temp = 9;
			goto SWITCH_VC;
		case RAW1_F11:
			temp = 10;
			goto SWITCH_VC;
		case RAW1_F12:
			temp = 11;
SWITCH_VC:
			select_vc(temp);
			break;
		default:
			temp = convert(key);
			if(temp != 0)
				do_console(temp);
			break;
		}
	}
}
/*****************************************************************************
*****************************************************************************/
void kbd_irq(void)
{
	unsigned key;

/* get scancode from port 0x60 */
	key = inportb(0x60);
#if 1
/* stick it in the main queue */
	if(ub_inq(&_main_kbd_queue, key) != 0)
		/* full queue; beep or something */;
/* wake up the bottom-half handler */
	wake_up(&_keyboard_bh);
#else
/* if it's F1, F2 etc. switch to the appropriate virtual console */
		switch(key)
		{
			unsigned temp;

		case RAW1_F1:
			temp = 0;
			goto SWITCH_VC;
		case RAW1_F2:
			temp = 1;
			goto SWITCH_VC;
		case RAW1_F3:
			temp = 2;
			goto SWITCH_VC;
		case RAW1_F4:
			temp = 3;
			goto SWITCH_VC;
		case RAW1_F5:
			temp = 4;
			goto SWITCH_VC;
		case RAW1_F6:
			temp = 5;
			goto SWITCH_VC;
		case RAW1_F7:
			temp = 6;
			goto SWITCH_VC;
		case RAW1_F8:
			temp = 7;
			goto SWITCH_VC;
		case RAW1_F9:
			temp = 8;
			goto SWITCH_VC;
		case RAW1_F10:
			temp = 9;
			goto SWITCH_VC;
		case RAW1_F11:
			temp = 10;
			goto SWITCH_VC;
		case RAW1_F12:
			temp = 11;
SWITCH_VC:
			select_vc(temp);
			break;
		default:
			temp = convert(key);
			if(temp != 0)
				do_console(temp);
			break;
		}
#endif
}
/*****************************************************************************
*****************************************************************************/
DISCARDABLE_CODE(void init_kbd(void))
{
	static unsigned char buffers[KBD_BUF_SIZE * MAX_VC];
	static unsigned char main_buffer[1024];
/**/
	int temp;

	_main_kbd_queue.data = main_buffer;
	_main_kbd_queue.size = sizeof(main_buffer);
	for(temp = 0; temp < MAX_VC; temp++)
	{
		_vc[temp].keystrokes.data = buffers + KBD_BUF_SIZE * temp;
		_vc[temp].keystrokes.size = KBD_BUF_SIZE;
	}
	kprintf("init_kbd: %u buffers, %u bytes each\n",
		MAX_VC, KBD_BUF_SIZE);
}
