/*****************************************************************************
DEBUG ROUTINES

EXPORTS:
void dump_stack_frame(void);
void dump(unsigned char *data, unsigned count);
void dump_regs(regs_t *regs);
void dump_task(task_t *task);
void dump_page_tables(void);
*****************************************************************************/
#include <_krnl.h>

/* IMPORTS
from MAIN.C */
void kprintf(const char *fmt, ...);

/* from KSTART.ASM */
unsigned *get_page_dir(void);
unsigned get_page_fault_adr(void);

/* from TASKS.C */
extern task_t *_curr_task;
/*****************************************************************************
*****************************************************************************/
void dump_stack_frame(void)
{
	unsigned curr_ebp, *esp;

	kprintf("return addresses from stack frames:\n");
	__asm__ __volatile__(
		"movl %%ebp,%0"
		: "=m"(curr_ebp));
//	while(curr_ebp >= 0xC0000000 && curr_ebp < 0xC0010000)
	while(curr_ebp >= 0x2000 && curr_ebp < 0x3000)
	{
		esp = (unsigned *)curr_ebp;	/* mov esp,ebp */
		curr_ebp = *esp++;		/* pop ebp */
		kprintf("%08X  ", *esp);	/* return EIP at [esp + 0] */
		esp++;
	}
	kprintf("\n");
}
/*****************************************************************************
*****************************************************************************/
#define BPERL		16	/* byte/line for dump */

void dump(unsigned char *data, unsigned count)
{
	unsigned byte1, byte2;

	while(count != 0)
	{
		for(byte1 = 0; byte1 < BPERL; byte1++)
		{
			if(count == 0)
				break;
			kprintf("%02X ", data[byte1]);
			count--;
		}
		kprintf("\t");
		for(byte2 = 0; byte2 < byte1; byte2++)
		{
			if(data[byte2] < ' ')
				kprintf(".");
			else
				kprintf("%c", data[byte2]);
		}
		kprintf("\n");
		data += BPERL;
	}
}
/*****************************************************************************
*****************************************************************************/
void dump_regs(regs_t *regs)
{
	kprintf("EDI=%08X    ESI=%08X    EBP=%08X    ESP=%08X    EBX=%08X\n",
		regs->edi, regs->esi, regs->ebp, regs->esp, regs->ebx);
	kprintf("EDX=%08X    ECX=%08X    EAX=%08X     DS=%08X     ES=%08X\n",
		regs->edx, regs->ecx, regs->eax, regs->ds, regs->es);
	kprintf(" FS=%08X     GS=%08X intnum=%08X  error=%08X    EIP=%08X\n",
		regs->fs, regs->gs, regs->which_int, regs->err_code,
		regs->eip);
	kprintf(" CS=%08X EFLAGS=%08X   uESP=%08X    uSS=%08X\n",
		regs->cs, regs->eflags, regs->user_esp, regs->user_ss);
	kprintf("CR2=%08X    CR3=%08X\n", get_page_fault_adr(),
		_curr_task->page_dir);
}
/*****************************************************************************
*****************************************************************************/
void dump_task(task_t *task)
{
	dump_regs((regs_t *)task->krnl_esp);
	kprintf("CR3=%08X        init kernel ESP=%08X             "
		"kernel ESP=%08X\n", task->page_dir,
		task->init_krnl_esp, task->krnl_esp);
}
/*****************************************************************************
*****************************************************************************/
void dump_page_tables(void)
{
	unsigned *page_dir, temp, *page_tab;
	unsigned pde, pte;

	page_dir = get_page_dir();
	kprintf("page directory at 0x%8lX (physical), ", page_dir);
	page_dir = (unsigned *)PAGE_DIR_VA;
	kprintf("0x%8lX (self-mapped virtual)\npage tables at "
		"0x%8lX (virtual)\n", page_dir, PAGE_TABLES_VA);
	for(pde = 0; pde < 1024; pde++)
	{
		if(page_dir[pde] == 0)
			continue;
		temp = page_dir[pde];
/* show page directory entries in bright white */
		kprintf("\x1B[1m""page dir[%4u]=0x%8lX""\x1B[0m""\n",
			pde, temp);
		page_tab = (unsigned *)(PAGE_TABLES_VA + 4096 * pde);
/* for identity-mapped memory (the first 4 meg), show only
the first few and last entries */
		if(pde < 1)
		{
			for(pte = 0; pte < 6; pte++)
			{
				if(page_tab[pte] == 0)
					continue;
				temp = page_tab[pte];
				kprintf("    page tab[%4u]=0x%8lX\n",
					pte, temp);
			}
			pte = 1023;
		}
		else
			pte = 0;
		for(; pte < 1024; pte++)
		{
			if(page_tab[pte] == 0)
				continue;
			temp = page_tab[pte];
			kprintf("    page tab[%4u]=0x%8lX\n", pte, temp);
		}
	}
}
