/*----------------------------------------------------------------------------
TEXT VIDEO (AND SERIAL) OUTPUT ROUTINES

EXPORTS:
extern unsigned g_vc_height;

void blink(void);
void putch(unsigned c);
void init_video(void);
----------------------------------------------------------------------------*/
#include <string.h> /* NULL, memcpy(), memsetw() */
#include <stdint.h> /* uint8_t, uint16_t */
#include <system.h> /* outportb(), inportb() */
#include <ctype.h> /* isdigit() */

/* IMPORTS
from KRNL.C */
void kprintf(const char *fmt, ...);

/* from START.ASM */
extern unsigned g_kvirt_to_phys;

/* from VM86.C */
void pokeb(unsigned seg, unsigned off, unsigned val);

#define	VGA_MISC_READ	0x3CC

static uint16_t *g_vga_fb_adr;
static unsigned g_csr_x, g_csr_y, g_esc, g_esc1, g_esc2, g_esc3;
static unsigned g_crtc_io_adr, g_vc_width/*, g_vc_height*/, g_attrib = 0x0B;
unsigned g_vc_height;
/*****************************************************************************
polled serial output in case video is f*cked up
*****************************************************************************/
static void ser_putch(unsigned c)
{
	static unsigned io_adr;
/**/

	if(io_adr == 0)
	{
		io_adr = 0x3F8;	/* 3F8=COM1, 2F8=COM2, 3E8=COM3, 2E8=COM4 */
		outportb(io_adr + 3, 0x80);
/* 115200 baud */
		outportb(io_adr + 0, 1);
		outportb(io_adr + 1, 0);
/* 8N1 */
		outportb(io_adr + 3, 0x03);
/* all interrupts disabled */
		outportb(io_adr + 1, 0);
/* turn off FIFO, if any */
		outportb(io_adr + 2, 0);
/* loopback off, interrupts (Out2) off, Out1/RTS/DTR off */
		outportb(io_adr + 4, 0);
	}
/* wait for transmitter ready */
	while((inportb(io_adr + 5) & 0x40) == 0)
		/* nothing */;
/* send char */
	outportb(io_adr + 0, c);
}
/*****************************************************************************
*****************************************************************************/
void blink(void)
{
	(*(uint8_t *)g_vga_fb_adr)++;
}
/*****************************************************************************
*****************************************************************************/
static void scroll(void)
{
	unsigned blank, temp;

	blank = 0x20 | ((unsigned)g_attrib << 8);
/* scroll up */
	if(g_csr_y >= g_vc_height)
	{
		temp = g_csr_y - g_vc_height + 1;
		memcpy(g_vga_fb_adr, g_vga_fb_adr + temp * g_vc_width,
			(g_vc_height - temp) * g_vc_width * 2);
/* blank the bottom line of the screen */
		memsetw(g_vga_fb_adr + (g_vc_height - temp) * g_vc_width,
			blank, g_vc_width);
		g_csr_y = g_vc_height - 1;
	}
}
/*****************************************************************************
*****************************************************************************/
static void clear_screen(void)
{
	unsigned blank;

	blank = 0x20 | ((unsigned)g_attrib << 8);
	memsetw(g_vga_fb_adr, blank, g_vc_height * g_vc_width);
}
/*****************************************************************************
*****************************************************************************/
static void set_attrib(unsigned att)
{
	static const unsigned ansi_to_vga[] =
	{
		0, 4, 2, 6, 1, 5, 3, 7
	};
/**/
	unsigned new_att;

	new_att = g_attrib;
	if(att == 0)
		new_att &= ~0x08;		/* bold off */
	else if(att == 1)
		new_att |= 0x08;		/* bold on */
	else if(att >= 30 && att <= 37)
	{
		att = ansi_to_vga[att - 30];
		new_att = (new_att & ~0x07) | att;/* fg color */
	}
	else if(att >= 40 && att <= 47)
	{
		att = ansi_to_vga[att - 40] << 4;
		new_att = (new_att & ~0x70) | att;/* bg color */
	}
	g_attrib = new_att;
}
/*****************************************************************************
*****************************************************************************/
static void move_csr(void)
{
	unsigned temp;

	temp = g_csr_y * g_vc_width + g_csr_x;
	outportb(g_crtc_io_adr + 0, 14);
	outportb(g_crtc_io_adr + 1, temp >> 8);
	outportb(g_crtc_io_adr + 0, 15);
	outportb(g_crtc_io_adr + 1, temp);
/* tell the BIOS what we've done */
	pokeb(0x40, 0x50, g_csr_y);
	pokeb(0x40, 0x51, g_csr_y);
}
/*****************************************************************************
*****************************************************************************/
static int do_vt(unsigned c)
{
/* state machine to handle the escape sequences */
	switch(g_esc)
	{
	case 0:
/* ESC -- next state */
		if(c == 0x1B)
		{
			g_esc++;
			return 1; /* "I handled it" */
		}
		break;
/* ESC */
	case 1:
		if(c == '[')
		{
			g_esc++;
			g_esc1 = 0;
			return 1;
		}
		break;
/* ESC[ */
	case 2:
		if(isdigit(c))
		{
			g_esc1 = g_esc1 * 10 + c - '0';
			return 1;
		}
		else if(c == ';')
		{
			g_esc++;
			g_esc2 = 0;
			return 1;
		}
/* ESC[2J -- clear screen */
		else if(c == 'J')
		{
			if(g_esc1 == 2)
			{
				clear_screen();
				g_csr_x = g_csr_y = 0;
				g_esc = 0;
				return 1;
			}
		}
/* ESC[num1m -- set attribute num1 */
		else if(c == 'm')
		{
			set_attrib(g_esc1);
			g_esc = 0;
			return 1;
		}
		break;
/* ESC[num1; */
	case 3:
		if(isdigit(c))
		{
			g_esc2 = g_esc2 * 10 + c - '0';
			return 1;
		}
		else if(c == ';')
		{
			g_esc++;
			g_esc3 = 0;
			return 1;
		}
/* ESC[num1;num2H -- move cursor to num1,num2 */
		else if(c == 'H')
		{
			if(g_esc2 < g_vc_width)
				g_csr_x = g_esc2;
			if(g_esc1 < g_vc_height)
				g_csr_y = g_esc1;
			g_esc = 0;
			return 1;
		}
/* ESC[num1;num2m -- set attributes num1,num2 */
		else if(c == 'm')
		{
			set_attrib(g_esc1);
			set_attrib(g_esc2);
			g_esc = 0;
			return 1;
		}
		break;
/* ESC[num1;num2;num3 */
	case 4:
		if(isdigit(c))
		{
			g_esc3 = g_esc3 * 10 + c - '0';
			return 1;
		}
/* ESC[num1;num2;num3m -- set attributes num1,num2,num3 */
		else if(c == 'm')
		{
			set_attrib(g_esc1);
			set_attrib(g_esc2);
			set_attrib(g_esc3);
			g_esc = 0;
			return 1;
		}
		break;
/* invalid state; reset */
	default:
		g_esc = 0;
		break;
	}
	g_esc = 0;
	return 0; /* "YOU handle it" */
}
/*****************************************************************************
*****************************************************************************/
void putch(unsigned c)
{
	unsigned att;

#if 0
/* send kernel text out the serial port, too */
	ser_putch(c);
#endif
	att = (unsigned)g_attrib << 8;
/* handle ANSI/VT escape sequences */
	if(do_vt(c))
		return;
/* backspace */
	if(c == 0x08)
	{
		if(g_csr_x != 0)
			g_csr_x--;
	}
/* tab */
	else if(c == 0x09)
		g_csr_x = (g_csr_x + 8) & ~(8 - 1);
/* carriage return */
	else if(c == '\r')	/* 0x0D */
		g_csr_x = 0;
/* line feed */
//	else if(c == '\n')	/* 0x0A */
//		g_csr_y++;
/* CR/LF */
	else if(c == '\n')	/* ### - 0x0A again */
	{
		g_csr_x = 0;
		g_csr_y++;
	}
/* printable ASCII */
	else if(c >= ' ')
	{
		uint16_t *where;

		where = g_vga_fb_adr + (g_csr_y * g_vc_width + g_csr_x);
		*where = (c | att);
		g_csr_x++;
	}
	if(g_csr_x >= g_vc_width)
	{
		g_csr_x = 0;
		g_csr_y++;
	}
	scroll();
	move_csr();
}
/*****************************************************************************
*****************************************************************************/
void init_video(void)
{
	unsigned vga_fb_adr; /* physical address of framebuffer */

/* check for monochrome or color VGA emulation
I don't think new SVGA boards support mono emulation */
	if((inportb(VGA_MISC_READ) & 0x01) != 0)
	{
		vga_fb_adr = 0xB8000L;
		g_crtc_io_adr = 0x3D4;
	}
	else
	{
		vga_fb_adr = 0xB0000L;
		g_crtc_io_adr = 0x3B4;
	}
	g_vga_fb_adr = (uint16_t *) /* virtual address of FB */
		(vga_fb_adr - g_kvirt_to_phys);
/* read text-mode screen size from BIOS data segment */
	g_vc_width = *(uint16_t *)(0x44A - g_kvirt_to_phys);
	g_vc_height = *(uint8_t *)(0x484 - g_kvirt_to_phys) + 1;
	kprintf("\x1B[2J"
		"  init_video: %s emulation, %u x %u, framebuffer at "
		"0x%X\n", (g_crtc_io_adr == 0x3D4) ? "color" : "mono",
		g_vc_width, g_vc_height, vga_fb_adr);
}
