;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; EXPORTS
; extern unsigned g_kvirt_to_phys;
; extern unsigned char g_tss_iopb[], g_tss_end[];
;
; void start_v86(uregs_t *regs);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

%include "nasm.inc" ; IMP, EXP

; IMPORTS
; from KRNL.LD
IMP g_bss
IMP g_end

; from MAIN.C
IMP g_curr_thread
IMP fault
IMP main

SECTION .text
BITS 16

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; entry point -- execution starts here
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

GLOBAL entry
entry:

; check for DOS
	cmp word [0],20CDh ; "INT 20h"
	jne no_dos
	inc byte [_dos]
no_dos:

; check for 32-bit CPU
	pushf
		pushf
		pop bx		; old FLAGS -> BX
		mov ax,bx
		xor ah,70h	; try changing b14 (NT)...
		push ax		; ... or b13:b12 (IOPL)
		popf
		pushf
		pop ax		; new FLAGS -> AX
	popf
	xor ah,bh		; 32-bit CPU if we changed NT...
	and ah,70h		; ...or IOPL
	jne cpu_ok
	mov si,_cpu_msg
	jmp die
cpu_ok:

; check for virtual 8086 mode
	smsw ax
	test al,1		; look at PE bit of MSW (CR0)
	je no_v86
	mov si,_v86_msg
	jmp die
no_v86:

; save real-mode segment register values
	mov [_real_cs],cs

; get code segment base address
	xor ebp,ebp
	mov bp,cs
	shl ebp,4

; fix up code segment descriptors in the GDT
	mov eax,ebp
	shr eax,16
	mov [_gdt_krnl_cs + 2],bp
	mov [_gdt_krnl_cs + 4],al
	mov [_gdt_krnl_cs + 7],ah

	mov [_gdt_16bit_cs + 2],bp
	mov [_gdt_16bit_cs + 4],al
	;mov [_gdt_16bit_cs + 7],ah ; 16-bit descriptor

; get data segment base address (DS=CS for TINY-model .COM file)
	xor ebp,ebp
	mov bp,ds
	shl ebp,4
	mov [g_kvirt_to_phys],ebp

; fix up data segment descriptors in the GDT
	mov eax,ebp
	shr eax,16
	mov [_gdt_krnl_ds + 2],bp
	mov [_gdt_krnl_ds + 4],al
	mov [_gdt_krnl_ds + 7],ah

	mov [_gdt_16bit_ds + 2],bp
	mov [_gdt_16bit_ds + 4],al
	;mov [_gdt_16bit_ds + 7],ah ; 16-bit descriptor

; fix up TSS descriptor, and GDT and IDT 'pseudo-descriptors'
	lea eax,[ebp + _tss]
	mov [_gdt_tss + 2],ax
	shr eax,16
	mov [_gdt_tss + 4],al
	mov [_gdt_tss + 7],ah

	lea eax,[ebp + _gdt]
	mov [_gdt_ptr + 2],eax

	lea eax,[ebp + _idt]
	mov [_idt_ptr + 2],eax

; build IDT
	mov cx,(_idt_end - _idt) / 8 ; number of exception handlers
	mov bx,_idt
	mov edx,isr0
do_idt:
	mov eax,edx		; set entry point
	mov [bx],ax		; _idt[N] -> isrN
	shr eax,16
	mov [bx + 6],ax
	add bx,8		; 8 bytes per interrupt gate
	add edx,(isr1 - isr0)	; 9 bytes per stub (push+push+jmp)
	loop do_idt

; set up stack
	mov esp,_stack

; interrupts off; IOPL=0; NT=0
	push dword 2
	popfd

; enter pmode
	lgdt [_gdt_ptr]
	lidt [_idt_ptr]
	mov eax,cr0
	or al,1
	mov cr0,eax
	jmp KRNL_CODE_SEL:pmode

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			die
; action:		displays message and exits
; in:			pointer to 0-terminated message in SI
; out:			(DOES NOT RETURN)
; modifies:		(DOES NOT RETURN)
; minimum CPU:		8088
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

die:
	call puts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			exit
; action:		exits to DOS or reboots
; in:			[_dos] must be set
; out:			(DOES NOT RETURN)
; modifies:		(DOES NOT RETURN)
; minimum CPU:		8088
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

exit:
	test byte [_dos],0FFh
	je reboot
	mov ax,4C01h		; DOS terminate
	int 21h
reboot:
	mov ah,0		; await key pressed
	int 16h
	int 19h			; re-start the boot process

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			putc
; action:		displays one character on stdout
; in:			character in AL. [_dos] must be set
; out:			(nothing)
; modifies:		(nothing)
; minimum CPU:		8088
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

putc:
	push ax
		test byte [_dos],0FFh
		jne dos_putc
		push bx
			mov ah,0Eh	; INT 10h: teletype output
			xor bx,bx	; video page 0
			int 10h
		pop bx
		jmp short putc_2
dos_putc:
		push dx
			mov dl,al
			mov ah,2
			int 21h
		pop dx
putc_2:
	pop ax
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:			puts
; action:		displays text on screen or stdout
; in:			0-terminated string at DS:SI
; out:			(nothing)
; modifies:		(nothing)
; minimum CPU:		8088
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

puts:
	push si
	push ax
		cld
		jmp short puts_2
puts_1:
		call putc
puts_2:
		lodsb
		or al,al
		jne puts_1
	pop ax
	pop si
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; return to real mode procedure, modified from section 14.5 of 386INTEL.TXT
; (...continued from below...)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; 4. load SS with selector to descriptor that is "appropriate for real mode":
;	Limit = 64K (FFFFh)	Byte-granular (G=0)
;	Expand-up (E=0)		Writable (W=1)
;	Present (P=1)		Base address = any value
; You may leave the other data segment registers with limits >64K if you
; want 'unreal mode', otherwise load them with a similar selector.

pmode16:
	mov ax,_16BIT_DATA_SEL
	mov ss,ax
	mov ds,ax
	mov es,ax
	mov fs,ax
	mov gs,ax

; 5. Clear the PE bit in register CR0
	mov eax,cr0
	and al,0FEh
	mov cr0,eax

; 6. Jump to a 16:16 real-mode far address
	jmp far [_real_ip]
real:

; 7. Load all other segment registers (SS, DS, ES, FS, GS)
	mov ax,cs
	mov ss,ax
	mov ds,ax
	mov es,ax
	mov fs,ax
	mov gs,ax

; 8. Use the LIDT instruction to load an IDT appropriate for real mode,
; with base address = 0 and limit = 3FFh. Use the 32-bit operand size
; override prefix so all 32 bits of the IDT base are set (otherwise,
; only the bottom 24 bits will be set).
	o32 lidt [_real_idt]

; 9. Zero the high 16 bits of 32-bit registers. If the register value is
; not important, just zero the entire 32-bit register, otherwise use 'movzx'
	xor eax,eax
	xor ebx,ebx
	xor ecx,ecx
	xor edx,edx
	xor esi,esi
	xor edi,edi
	movzx ebp,bp
	movzx esp,sp

; 10. Enable interrupts
	sti
	jmp exit

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; pmode
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

ALIGN 512 ; so NDISASM knows where the 32-bit code starts

BITS 32
GLOBAL pmode ; so "objdump --source ..." can find it
pmode:
	mov ax,KRNL_DATA_SEL
	mov ds,ax
	mov ss,ax
	mov es,ax
	mov fs,ax
	mov gs,ax

; LTR is illegal in real mode, so do it now
	mov ax,TSS_SEL
	ltr ax

; zero the kernel BSS
	xor eax,eax
	mov edi,g_bss		; BSS starts here
	mov ecx,g_end
	sub ecx,edi		; BSS size is (g_end - g_bss) bytes
	cld
	rep stosb

; call C code
	call main

; return to real mode procedure, modified from section 14.5 of 386INTEL.TXT
; 1. Disable interrupts (already done)
; 2. If paging is enabled... (it isn't)
; 3. Jump to a segment with limit 64K-1. CS must have this limit
; before you return to real mode.

	jmp _16BIT_CODE_SEL:pmode16

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:		start_v86
; action:	enters virtual 8086 mode
; in:		args on stack per C prototype
; out:		(see end_v86, below)
; modifies:	(see end_v86, below)
; notes:	C prototype: void start_v86(uregs_t *regs);
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

EXP start_v86

; save registers used by C: EBP, EBX, ESI, EDI
; Actually, just use PUSHA to save them all
	pusha

; copy V86 mode registers to top of stack
	sub esp,92
		mov esi,[esp + 128] ; (uregs_t *regs)
		mov edi,esp
		mov ecx,92 ; sizeof(uregs_t)
		cld
		rep movsb

		jmp short to_user

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:		all_ints
; action:	unified IRQ/interrupt/exception handler
; in:		IRET frame, error code, and exception number
; 		all pushed on stack
; out:		(nothing)
; modifies:	(nothing)
; notes:	prototype for C-language interrupt handler:
; 		int fault(uregs_t *regs);
;
; *** WARNING: The order in which registers are pushed and popped here
; must agree with the layout of the uregs_t structure in the C code.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

all_ints:
	push gs			; push segment registers
	push fs
	push ds
	push es
	pusha			; push EDI...EAX (7 regs; 8 dwords)
		mov ax,KRNL_DATA_SEL
		mov ds,eax	; put known-good values in segment registers
		mov es,eax
		mov fs,eax
		mov gs,eax
		push esp	; push pointer to stacked registers
			call fault
		add esp,4
		or eax,eax
		jne end_v86

; *** FALL-THROUGH to to_user ***

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:		to_user
; action:	kernel exit code
; in:		uregs_t frame on kernel stack
; out:		(nothing)
; modifies:	(nothing)
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to_user:

; IRET to Ring 3 pmode will pop these values
; 8 GP regs, 4 seg regs, int_num/err_code, EIP/CS/EFLAGS/ESP/SS
; making the kernel stack 76 bytes smaller
		lea eax,[esp + 76]

; IRET to V86 mode...
		;test dword [esp + 64],20000h ; test VM bit in stacked EFLAGS
		test byte [esp + 66],2
		je set_tss_esp0

; ...will pop these values
; 8 GP regs, 4 seg regs, int_num/err_code, EIP/CS/EFLAGS/ESP/SS/ES/DS/FS/GS
; making the kernel stack 92 bytes smaller
		lea eax,[esp + 92]

; For IRET to Ring 0 pmode, _tss_esp0 is not used,
; so this value doesn't matter
set_tss_esp0:
		mov [_tss_esp0],eax
	popa			; pop EAX...EDI (7 regs; 8 dwords)
	pop es			; pop segment registers
	pop ds
	pop fs
	pop gs
	add esp,byte 8		; drop int_num and err_code
	iret			; pop EIP...GS (9 regs)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; name:		end_v86
; action:	cleans up after virtual 8086 mode session;
;		returns to pmode kernel
; in:		(see start_v86, above)
; out:		registers in uregs_t may be modified
; modifies:	registers in uregs_t may be modified
; notes:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

end_v86:

; get modified V86 mode registers
		mov esi,esp
		mov edi,[esp + 128]
		mov ecx,92
		cld
		rep movsb

; remove V86 registers from stack
		add esp,92

; restore C registers and return to pmode kernel
	popa
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; interrupt stubs
;
; *** WARNING: These stubs MUST be consecutive,
; and each stub MUST occupy the same number of bytes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

isr0:
	push byte 0		; push fake error code
	push byte 0		; push interrupt number

; Use 'dword' to force a 5-byte JMP here. Otherwise, depending on what
; version of NASM you use, you might get a mix of short (2-byte)
; and long (5-byte) JMPs, which screws up the code at 'do_idt'
	jmp dword all_ints	; zero divide (fault)
isr1:
	push byte 0
	push byte 1
	jmp dword all_ints	; debug/single step
isr2:
	push byte 0
	push byte 2
	jmp dword all_ints	; non-maskable interrupt
isr3:
	push byte 0
	push byte 3
	jmp dword all_ints	; INT3
isr4:
	push byte 0
	push byte 4
	jmp dword all_ints	; INTO
isr5:
	push byte 0
	push byte 5
	jmp dword all_ints	; BOUND
isr6:
	push byte 0
	push byte 6
	jmp dword all_ints	; invalid opcode
isr7:
	push byte 0
	push byte 7
	jmp dword all_ints	; coprocessor not available
isr8:
	nop			; fake error code not needed
	nop
	push byte 8
	jmp dword all_ints	; double fault
isr9:
	push byte 0
	push byte 9
	jmp dword all_ints	; coproc segment overrun
isr0A:
	nop
	nop
	push byte 0Ah
	jmp dword all_ints	; bad TSS
isr0B:
	nop
	nop
	push byte 0Bh
	jmp dword all_ints	; segment not present
isr0C:
	nop
	nop
	push byte 0Ch
	jmp dword all_ints	; stack fault
isr0D:
	nop
	nop
	push byte 0Dh
	jmp dword all_ints	; GPF
isr0E:
	nop
	nop
	push byte 0Eh
	jmp dword all_ints	; page fault
isr0F:
	push byte 0
	push byte 0Fh
	jmp dword all_ints	; reserved
isr10:
	push byte 0
	push byte 10h
	jmp dword all_ints	; FP exception/coprocessor error
isr11:
	push byte 0
	push byte 11h
	jmp dword all_ints	; alignment check
isr12:
	push byte 0
	push byte 12h
	jmp dword all_ints	; machine check
isr13:
	push byte 0
	push byte 13h
	jmp dword all_ints
isr14:
	push byte 0
	push byte 14h
	jmp dword all_ints
isr15:
	push byte 0
	push byte 15h
	jmp dword all_ints
isr16:
	push byte 0
	push byte 16h
	jmp dword all_ints
isr17:
	push byte 0
	push byte 17h
	jmp dword all_ints
isr18:
	push byte 0
	push byte 18h
	jmp dword all_ints
isr19:
	push byte 0
	push byte 19h
	jmp dword all_ints
isr1A:
	push byte 0
	push byte 1Ah
	jmp dword all_ints
isr1B:
	push byte 0
	push byte 1Bh
	jmp dword all_ints
isr1C:
	push byte 0
	push byte 1Ch
	jmp dword all_ints
isr1D:
	push byte 0
	push byte 1Dh
	jmp dword all_ints
isr1E:
	push byte 0
	push byte 1Eh
	jmp dword all_ints
isr1F:
	push byte 0
	push byte 1Fh
	jmp dword all_ints
isr20:
	push byte 0
	push byte 20h
	jmp dword all_ints
isr21:
	push byte 0
	push byte 21h
	jmp dword all_ints
isr22:
	push byte 0
	push byte 22h
	jmp dword all_ints
isr23:
	push byte 0
	push byte 23h
	jmp dword all_ints
isr24:
	push byte 0
	push byte 24h
	jmp dword all_ints
isr25:
	push byte 0
	push byte 25h
	jmp dword all_ints
isr26:
	push byte 0
	push byte 26h
	jmp dword all_ints
isr27:
	push byte 0
	push byte 27h
	jmp dword all_ints
isr28:
	push byte 0
	push byte 28h
	jmp dword all_ints
isr29:
	push byte 0
	push byte 29h
	jmp dword all_ints
isr2A:
	push byte 0
	push byte 2Ah
	jmp dword all_ints
isr2B:
	push byte 0
	push byte 2Bh
	jmp dword all_ints
isr2C:
	push byte 0
	push byte 2Ch
	jmp dword all_ints
isr2D:
	push byte 0
	push byte 2Dh
	jmp dword all_ints
isr2E:
	push byte 0
	push byte 2Eh
	jmp dword all_ints
isr2F:
	push byte 0
	push byte 2Fh
	jmp dword all_ints
isr30:
	push byte 0
	push byte 30h
	jmp dword all_ints

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; data
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

SECTION .data

_cpu_msg:
	db "32-bit CPU required", 10, 13, 0

_v86_msg:
	db "CPU already in Virtual-8086 mode "
	db "(Windows DOS box or EMM386 loaded?)"
	db 10, 13, 0

_real_idt:
	dw 1023
	dd 0
_dos:
	db 0

; 16:16 far address for return to real mode. Don't change the order
; of '_real_ip' and '_real_cs', and don't put anything between them,
; because these two values are the operand of a far JMP
_real_ip:
	dw real
_real_cs:
	dw 0

EXP g_kvirt_to_phys
	dd 0

;;;;;;;;;;;;;
;;;; TSS ;;;;
;;;;;;;;;;;;;

	ALIGN 4

; most of the TSS is unused
; I need only ESP0, SS0, and the I/O permission bitmap
_tss:
	dw 0, 0			; back link
_tss_esp0:
	dd 0			; ESP0
	dw KRNL_DATA_SEL, 0	; SS0, reserved

	dd 0			; ESP1
	dw 0, 0			; SS1, reserved

	dd 0			; ESP2
	dw 0, 0			; SS2, reserved

	dd 0			; CR3
	dd 0, 0			; EIP, EFLAGS
	dd 0, 0, 0, 0		; EAX, ECX, EDX, EBX
	dd 0, 0, 0, 0		; ESP, EBP, ESI, EDI
	dw 0, 0			; ES, reserved
	dw 0, 0			; CS, reserved
	dw 0, 0			; SS, reserved
	dw 0, 0			; DS, reserved
	dw 0, 0			; FS, reserved
	dw 0, 0			; GS, reserved
	dw 0, 0			; LDT, reserved
	dw 0, g_tss_iopb - _tss	; debug, IO permission bitmap base
EXP g_tss_iopb

; no I/O permitted
	times 8192 db 0FFh

; The TSS notes in section 12.5.2 of volume 1 of
; "IA-32 Intel Architecture Software Developer's Manual"
; are confusing as hell. I think they mean simply that the IOPB
; must contain an even number of bytes; so pad here if necessary.
	;dw 0FFFFh
EXP g_tss_end

;;;;;;;;;;;;;
;;;; GDT ;;;;
;;;;;;;;;;;;;

_gdt:
	dd 0
	dd 0

TSS_SEL		EQU ($ - _gdt)
_gdt_tss:
	dw g_tss_end - _tss - 1
	dw 0
	db 0
	db 89h			; ring 0 available 32-bit TSS
	db 0
	db 0

KRNL_DATA_SEL	EQU ($ - _gdt)
_gdt_krnl_ds:
	dw 0FFFFh
	dw 0			; base; gets set above
	db 0
	db 92h			; present, ring 0, data, expand-up, writable
	db 0CFh
	db 0

KRNL_CODE_SEL	EQU ($ - _gdt)
_gdt_krnl_cs:
	dw 0FFFFh
	dw 0			; base; gets set above
	db 0
	db 9Ah			; present, ring 0, code, non-conforming, readable
	db 0CFh			; 32-bit
	db 0

USER_DATA_SEL	EQU (($ - _gdt) | 3) ; RPL=3 (ring 3)
_gdt_user_ds:
	dw 0FFFFh
	dw 0			; base
	db 0
	db 0F2h			; present, ring 3, data, expand-up, writable
	db 0CFh
	db 0

USER_CODE_SEL	EQU (($ - _gdt) | 3)
_gdt_user_cs:
	dw 0FFFFh
	dw 0			; base; gets set above
	db 0
	db 0FAh			; present, ring 3, code, non-conforming, readable
	db 0CFh			; 32-bit
	db 0

LINEAR_DATA_SEL	EQU ($ - _gdt)
	dw 0FFFFh
	dw 0
	db 0
	db 92h			; present, ring 0, data, expand-up, writable
	db 0CFh
	db 0

LINEAR_CODE_SEL	EQU ($ - _gdt)
	dw 0FFFFh
	dw 0
	db 0
	db 9Ah			; present, ring 0, code, non-conforming, readable
	db 0CFh
	db 0

_16BIT_DATA_SEL	EQU ($ - _gdt)
_gdt_16bit_ds:
	dw 0FFFFh
	dw 0			; base; gets set above
	db 0
	db 92h			; present, ring 0, data, expand-up, writable
	db 0
	db 0

_16BIT_CODE_SEL	EQU ($ - _gdt)
_gdt_16bit_cs:
	dw 0FFFFh
	dw 0			; base; gets set above
	db 0
	db 9Ah			; present, ring 0, code, non-conforming, readable
	db 0			; 16-bit
	db 0
_gdt_end:

_gdt_ptr:
	dw _gdt_end - _gdt - 1	; GDT limit
	dd _gdt			; linear adr of GDT; gets set above

;;;;;;;;;;;;;
;;;; IDT ;;;;
;;;;;;;;;;;;;

_idt:
; 48 ring 0 interrupt gates
	times 48	dw 0, KRNL_CODE_SEL, 8E00h, 0

; one ring 3 interrupt gate for syscalls (INT 30h)
	dw 0			; offset 15:0
	dw KRNL_CODE_SEL	; selector
	db 0			; (always 0 for interrupt gates)
	db 0EEh			; present, ring 3, '386 interrupt gate
	dw 0			; offset 31:16
_idt_end:

_idt_ptr:
	dw _idt_end - _idt - 1	; IDT limit
	dd _idt			; linear, physical address of IDT (patched)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; BSS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

SECTION .bss

	resb 16384
_stack:
