/*----------------------------------------------------------------------------
32-BIT PMODE CODE CALLING 16-BIT REAL-MODE PnP BIOS

EXPORTS:
int scan_pnp16(void);
----------------------------------------------------------------------------*/
#include <stdint.h>
#include <stdio.h>
#include "defs.h"

/* IMPORTS
from VM86.C */
unsigned peekb(unsigned seg, unsigned off);
unsigned peekw(unsigned seg, unsigned off);
void v86_push16(uregs_t *regs, unsigned value);

/* from KRNL.C */
void kprintf(const char *fmt, ...);
void init_v86_regs(uregs_t *regs);

/* from START.ASM */
extern unsigned g_kvirt_to_phys;

void start_v86(uregs_t *regs);

static const char g_crash_vm86[] =
{
	0xBC, 0xFF, 0xFF,/* mov sp,0FFFFh */
	0xC3		/* ret */
};
static uint16_t g_pnp_cs, g_pnp_ip, g_pnp_ds;
/*****************************************************************************
When called, the PnP BIOS expects things to be pushed onto the stack.
This means we can't use RET or RETF from an empty stack to end the
V86 session, as we do with interrupts (IRET from empty stack).

Solution: push the far address of g_crash_vm86[]; which is a short
segment of 16-bit code that causes the necessary stack fault.
*****************************************************************************/
static int fn0(uint16_t *node_size, uint16_t *num_nodes)
{
	unsigned long adr;
	uregs_t regs;

/* init regs AND real-mode stack */
	init_v86_regs(&regs);
/* push onto real-mode stack...
...PNP BIOS DS */
	v86_push16(&regs, g_pnp_ds);
/* ...far address (offset:segment) of node_size */
	adr = (unsigned)node_size + g_kvirt_to_phys;
	v86_push16(&regs, adr >> 4);
	v86_push16(&regs, adr & 0x0F);
/* ...far address (offset:segment) of num_nodes */
	adr = (unsigned)num_nodes + g_kvirt_to_phys;
	v86_push16(&regs, adr >> 4);
	v86_push16(&regs, adr & 0x0F);
/* ...function number */
	v86_push16(&regs, 0);
/* ...far return address (offset:segment) of g_crash_vm86 */
	adr = (unsigned)&g_crash_vm86 + g_kvirt_to_phys;
	v86_push16(&regs, adr >> 4);
	v86_push16(&regs, adr & 0x0F);
/* SIMULATE V86 FAR CALL */
	regs.cs = g_pnp_cs;
	regs.eip = g_pnp_ip;
/* do it */
	start_v86(&regs);
	return regs.eax & 0xFFFF;
}
/*****************************************************************************
*****************************************************************************/
static int fn1(unsigned char *node_num, unsigned char *node_info)
{
	enum
	{
		THIS_BOOT = 1, NEXT_BOOT = 2
	} boot_type = THIS_BOOT;
	unsigned long adr;
	uregs_t regs;

	init_v86_regs(&regs);
/* push onto real-mode stack...
...PNP BIOS DS */
	v86_push16(&regs, g_pnp_ds);
/* ...boot_type */
	v86_push16(&regs, boot_type);
/* ...far address (offset:segment) of node_info */
	adr = (unsigned)node_info + g_kvirt_to_phys;
	v86_push16(&regs, adr >> 4);
	v86_push16(&regs, adr & 0x0F);
/* ...far address (offset:segment) of node_num */
	adr = (unsigned)node_num + g_kvirt_to_phys;
	v86_push16(&regs, adr >> 4);
	v86_push16(&regs, adr & 0x0F);
/* ...function number */
	v86_push16(&regs, 1);
/* ...far return address (offset:segment) of g_crash_vm86 */
	adr = (unsigned)&g_crash_vm86 + g_kvirt_to_phys;
	v86_push16(&regs, adr >> 4);
	v86_push16(&regs, adr & 0x0F);
/* SIMULATE V86 FAR CALL */
	regs.cs = g_pnp_cs;
	regs.eip = g_pnp_ip;
/* do it */
	start_v86(&regs);
	return regs.eax & 0xFFFF;
}
/*****************************************************************************
converts 5-character "compressed ASCII" PnP ID at src
to 7-character normal ASCII at dst
*****************************************************************************/
static void decode_id(char *dst, unsigned char *src)
{
	unsigned i;

	i = src[2];
	i <<= 8;
	i |= src[3];
	sprintf(dst + 3, "%04X", i);

	i = src[0];
	i <<= 8;
	i |= src[1];
	dst[2] = '@' + (i & 0x1F);
	i >>= 5;
	dst[1] = '@' + (i & 0x1F);
	i >>= 5;
	dst[0] = '@' + (i & 0x1F);
}
/*****************************************************************************
*****************************************************************************/
static int pnp16(unsigned seg, unsigned off)
{
	unsigned char node_num, next_node_num, buf[128];
	uint16_t node_size, num_nodes;
	char id[16];
	int i;

/* get data segment and real-mode entry point */
	kprintf("found, version is %u.%u\n",
		peekb(seg, off + 4) >> 4,
		peekb(seg, off + 4) & 0x0F);
	g_pnp_ip = peekw(seg, off + 13);
	g_pnp_cs = peekw(seg, off + 15);
	g_pnp_ds = peekw(seg, off + 27);
	kprintf("16-bit PnP BIOS entry point is %04X:%04X, "
		"data segment is %04X\n",
		g_pnp_cs, g_pnp_ip, g_pnp_ds);
/* get node size and count */
	i = fn0(&node_size, &num_nodes);
	if(i != 0)
	{
		kprintf("*** PnP function 0 returned %d\n", i);
		return -1;
	}
//	num_nodes &= 0xFF;
	kprintf("node_size = %u, num_nodes = %u\n", node_size, num_nodes);
/* read nodes */
	node_num = next_node_num = 0;
	do
	{
		i = fn1(&next_node_num, buf);
		if(i != 0)
		{
			kprintf("PnP function 1 returned %d\n", i);
			return -1;
		}
/* bytes 3-6 are the device ID */
		decode_id(id, buf + 3);
		kprintf("device %2u: %s\n", node_num, id);
		node_num = next_node_num;
	} while(next_node_num != 0xFF);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
int scan_pnp16(void)
{
	unsigned seg = 0xF000, off, size, i;
	unsigned char csum;

/* look for 16-bit PnP BIOS signature */
	kprintf("Scanning for 16-bit PnP BIOS...");
	for(off = 0; off < 0xFFF0; off += 16)
	{
		if(peekb(seg, off + 0) != '$')
			continue;
		if(peekb(seg, off + 1) != 'P')
			continue;
		if(peekb(seg, off + 2) != 'n')
			continue;
		if(peekb(seg, off + 3) != 'P')
			continue;
/* verify checksum */
		size = peekb(seg, off + 5);
		csum = 0;
		for(i = 0; i < size; i++)
			csum += peekb(seg, off + i);
		if(csum != 0)
			continue;
/* found it! */
		return pnp16(seg, off);
	}
	kprintf("not found\n");
	return 1;
}
