/*----------------------------------------------------------------------------
PMODE KERNEL FOR V86 DEMO

EXPORTS:
void kprintf(const char *fmt, ...);
int fault(uregs_t *regs);
void init_v86_regs(uregs_t *regs);
int main(void);
----------------------------------------------------------------------------*/
#include <_printf.h> /* do_printf() */
#include <stdarg.h> /* va_list, va_start(), va_end() */
#include <stdint.h> /* uint8_t, uint16_t, uint32_t */
#include <setjmp.h> /* jmp_buf, setjmp(), longjmp() */
#include <string.h> /* NULL, memset(), strcpy() */
#include <system.h> /* disable(), outportb(), inportb() */
#include "defs.h" /* uregs_t */

/* IMPORTS
from DEBUG.C */
void dump(void *data_p, unsigned count);
void dump_iopb(void);
void dump_uregs(uregs_t *regs);

/* from VM86.C */
unsigned peekw(unsigned seg, unsigned off);
void v86_int(uregs_t *regs, unsigned int_num);
int v86_emulate(uregs_t *regs);

/* from VIDEO.C */
extern unsigned g_vc_height;

void blink(void);
void putch(unsigned c);
void init_video(void);

/* from START.ASM */
extern unsigned char g_tss_iopb[];
extern unsigned char g_tss_end[];
extern unsigned g_kvirt_to_phys;

void start_v86(uregs_t *regs);

/* from VGA.C */
void set_text_mode(void);

/* from PNP.C */
int scan_pnp16(void);

#define	KRNL_CODE_SEL	0x18

static jmp_buf g_oops;
/*****************************************************************************
*****************************************************************************/
static int kprintf_help(unsigned c, void **ptr_UNUSED)
{
	putch(c);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
void kprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	do_printf(fmt, args, kprintf_help, NULL);
	va_end(args);
}
/*****************************************************************************
*****************************************************************************/
int fault(uregs_t *regs)
{
	static const char *msg[] =
	{
		"divide error", "debug exception", "NMI", "INT3",
		"INTO", "BOUND exception", "invalid opcode", "no coprocessor",
		"double fault", "coprocessor segment overrun",
			"bad TSS", "segment not present",
		"stack fault", "GPF", "page fault", "??",
		"coprocessor error", "alignment check", "??", "??",
		"??", "??", "??", "??",
		"??", "??", "??", "??",
		"??", "??", "??", "??",
		"IRQ0", "IRQ1", "IRQ2", "IRQ3",
		"IRQ4", "IRQ5", "IRQ6", "IRQ7",
		"IRQ8", "IRQ9", "IRQ10", "IRQ11",
		"IRQ12", "IRQ13", "IRQ14", "IRQ15",
		"syscall"
	};
/**/
	int i;

/* "reflect" hardware interrupts (IRQs) to V86 mode */
	if(regs->which_int >= 0x20 && regs->which_int < 0x30)
	{
/* for timer interrupt, blink character in upper left corner of screen */
		if(regs->which_int == 0x20)
			blink();
/* IRQs 0-7 -> pmode exception numbers 20h-27h -> real-mode vectors 8-15 */
		if(regs->which_int < 0x28)
			i = (regs->which_int - 0x20) + 8;
/* IRQs 8-15 -> pmode exception numbers 28h-2Fh -> real-mode vectors 70h-77h */
		else
			i = (regs->which_int - 0x28) + 0x70;
		v86_int(regs, i);
		return 0;
	}
/* stack fault from V86 mode with SP=0xFFFF means we're done */
	else if(regs->which_int == 0x0C)
	{
		if((regs->eflags & 0x20000uL) && regs->user_esp == 0xFFFF)
			return 1;
	}
/* GPF from V86 mode: emulate instructions */
	else if(regs->which_int == 0x0D)
	{
		if(regs->eflags & 0x20000uL)
		{
			i = v86_emulate(regs);
/* error */
			if(i < 0)
				longjmp(g_oops, 1);
/* pop from empty stack: end V86 session */
			else if(i > 0)
				return 1;
			return 0;
		}
	}
/* for anything else, display exception number,
exception name, and CPU mode */
	kprintf("*** Exception #%u ", regs->which_int);
	if(regs->which_int <= sizeof(msg) / sizeof(char *))
		kprintf("(%s) ", msg[regs->which_int]);
	if(regs->eflags & 0x20000uL)
		kprintf("in V86");
	else if(regs->cs & 0x03)
		kprintf("in user");
	else
		kprintf("in kernel");
	kprintf(" mode ***\n");
/* dump registers and longjmp() back to main() */
	dump_uregs(regs);
	longjmp(g_oops, 1);
/* not reached */
	return -1;
}
/*****************************************************************************
Sets up V86 mode stack and initializes some registers
*****************************************************************************/
void init_v86_regs(uregs_t *regs)
{
/* V86 mode stack. This MUST be in conventional memory (below 1 meg)
but still be accessible to the pmode kernel: */
	static char _v86_stack[4096];
/**/
	char *v86_stack;
	unsigned s;

/* start with all registers zeroed */
	memset(regs, 0, sizeof(uregs_t));
/* point to BOTTOM (highest addess) of stack */
	v86_stack = _v86_stack + sizeof(_v86_stack);
/* v86_stack is virtual address. Convert it to physical address
and align it so bottom 4 bits == 0x0F
's' must be >= 0x10000 so we can set ESP=0xFFFF, below. */
	s = ((unsigned)v86_stack + g_kvirt_to_phys - 16) | 0x0F;
/* init stack */
	regs->user_esp = 0xFFFF;
	regs->user_ss = (s - 0xFFFF) / 16;
/* init kernel data segment registers */
	regs->ds = regs->es = regs->fs = regs->gs = KRNL_CODE_SEL;
/* init EFLAGS: set RF=0, NT=0, IF=0, and reserved bits */
#if 0
	regs->eflags = 0x00020002L; /* VM=1, IOPL=0 */
#else
	regs->eflags = 0x00023002L; /* VM=1, IOPL=3 */
#endif
}
/*****************************************************************************
Call real-mode interrupt handler in V86 mode
*****************************************************************************/
static void do_v86_int(uregs_t *regs, unsigned int_num)
{
	unsigned ivt_off;

/* convert int_num to IVT index */
	ivt_off = (int_num & 0xFF) * 4;
/* fetch CS:IP of real-mode interrupt handler */
	regs->cs = peekw(0, ivt_off + 2);
	regs->eip = peekw(0, ivt_off + 0);
/* do it */
	start_v86(regs);
}
/*****************************************************************************
*****************************************************************************/
static void init_8259s(char pmode)
{
	static unsigned pic0mask, pic1mask;
/**/

	if(pmode)
	{
		outportb(0x20, 0x11); /* ICW1: edge-triggered IRQs... */
		outportb(0xA0, 0x11); /* ...cascade mode, need ICW4 */

		outportb(0x21, 0x20); /* ICW2: route IRQs 0...7 to INTs 0x20-0x27 */
		outportb(0xA1, 0x27); /* ...IRQs 8...15 to INTs 0x28-0x2F */

		outportb(0x21, 0x04); /* ICW3 master: slave 8259 on IRQ2 */
		outportb(0xA1, 0x02); /* ICW3 slave: I am slave on IRQ2 */

		outportb(0x21, 0x01); /* ICW4 */
		outportb(0xA1, 0x01);

		pic0mask = inportb(0x21);
		pic1mask = inportb(0xA1);
/* disable interrupts */
		outportb(0x21, ~0x00);
		outportb(0xA1, ~0x00);
	}
	else
	{
		outportb(0x20, 0x11); /* ICW1: edge-triggered IRQs... */
		outportb(0xA0, 0x11); /* ...cascade mode, need ICW4 */

		outportb(0x21, 0x08); /* ICW2: route IRQs 0...7 to INTs 0x08-0x0F */
		outportb(0xA1, 0x70); /* ...IRQs 8...15 to INTs 0x70-0x77 */

		outportb(0x21, 0x04); /* ICW3 master: slave 8259 on IRQ2 */
		outportb(0xA1, 0x02); /* ICW3 slave: I am slave on IRQ2 */

		outportb(0x21, 0x01); /* ICW4 */
		outportb(0xA1, 0x01);

		outportb(0x21, pic0mask);
		outportb(0xA1, pic1mask);
	}
}
/*****************************************************************************
TEXT OUTPUT USING INT 10h AH=0Eh
*****************************************************************************/
static void demo1(void)
{
	static const char v86_msg[] = "Hello from V86 mode";
/**/
	const char *s;
	uregs_t regs;

	kprintf("Demo 1: INT 10h AH=0Eh (display text character)\n");
	init_v86_regs(&regs);
	for(s = v86_msg; *s != '\0'; s++)
	{
		regs.eax = 0x0E00 | *(unsigned char *)s;
		regs.ebx = 0x0000;
		do_v86_int(&regs, 0x10);
	}
/* INT 10h AH=0Eh doesn't handle newlines, so use kprintf() to
leave a blank line (so we can see the message printed in V86 mode) */
	kprintf("\n");
}
/*****************************************************************************
LOCATE ROM FONT VIA INT 10h AX=1130h
*****************************************************************************/
static void demo2(void)
{
	uregs_t regs;

	kprintf("Demo 2: INT 10h AX=1130h (locate ROM font)\n");
	init_v86_regs(&regs);
	regs.eax = 0x1130;
	regs.ebx = 0x0300;	/* BH=03h (8x8 font) */
	do_v86_int(&regs, 0x10);
	kprintf("\tFont #3 (8x8) is at real-mode address %04X:%04X\n",
		regs.v_es, regs.ebp & 0xFFFF);
}
/*****************************************************************************
VGA MODE-SET USING INT 10h AH=00h
*****************************************************************************/
static void demo3(void)
{
	uregs_t regs;

	kprintf("Demo 3: INT 10h AH=00h (VGA mode-set; 80x50 text)\n");
	init_v86_regs(&regs);
/* set 80x25 text mode */
	regs.eax = 0x0003;
	do_v86_int(&regs, 0x10);
/* set 8x8 font for 80x50 text mode */
	regs.eax = 0x1112;
	regs.ebx = 0;
	do_v86_int(&regs, 0x10);
/* */
	g_vc_height = 50;
}
/*****************************************************************************
VBE FUNCTIONS USING INT 10h AH=4Fh
*****************************************************************************/
#pragma pack(1)
typedef struct
{
	char sig[4];
	uint8_t ver_minor, ver_major;
	uint16_t oem_name_off, oem_name_seg;
	uint32_t capabilities;	 /* b1=1 for non-VGA board */
	uint16_t mode_list_off, mode_list_seg;
	uint16_t vid_mem_size; /* in units of 64K */
	char reserved[492];
} vbe_info_t;

#pragma pack(1)
typedef struct
{
	uint16_t mode_attrib;	 /* b5=1 for non-VGA mode */
	uint8_t win_a_attrib, win_b_attrib;
	uint16_t k_per_gran;
	uint16_t win_size;
	uint16_t win_a_seg;
	uint16_t win_b_seg;
	char reserved1[4];
	uint16_t bytes_per_row;
	char reserved2[494];
} vbe_mode_t;

static unsigned long g_bytes_per_row;
static unsigned g_gran_per_64k;
static char g_use_win_a, *g_fb;

static void set_bank(unsigned b)
{
	static unsigned curr_bank = -1u;
	static uregs_t regs;
/**/

	if(curr_bank == -1u)
		init_v86_regs(&regs);
	if(b == curr_bank)
		return;
	curr_bank = b;
	regs.eax = 0x4F05;
/* g_use_win_a and g_gran_per_64k set by INT 10h AX=4F01h */
	regs.ebx = g_use_win_a ? 0x0000 : 0x0001;
	regs.edx = b * g_gran_per_64k;
	do_v86_int(&regs, 0x10);
}
/*****************************************************************************
*****************************************************************************/
static void write_pixel8b(unsigned x, unsigned y, unsigned c)
{
	unsigned long off;

	off = g_bytes_per_row * y + x;
	set_bank((unsigned)(off >> 16));
	g_fb[off & 0xFFFF] = c;
}
/*****************************************************************************
*****************************************************************************/
static char g_screen_may_be_fubar;

static int demo4(void)
{
	unsigned i, vbe_ver, wd, ht;
	vbe_info_t *vbe_info;
	vbe_mode_t *mode_info;
/* 'buf' must be in conventional memory (below 1 meg)
but still be accessible to the pmode code in this function */
	char buf[512], *s;
	uregs_t regs;

	kprintf("Demo 4: INT 10h AX=4Fxxh (VBE graphics)\n");
	init_v86_regs(&regs);
/* get physical address of buf and round up to paragraph
(16-byte) boundary */
	i = ((unsigned)buf + g_kvirt_to_phys + 15) & ~0x0F;
/* convert aligned value back to near address */
	s = (char *)(i - g_kvirt_to_phys);
	vbe_info = (vbe_info_t *)s;
	mode_info = (vbe_mode_t *)s;
/* convert physical address to real-mode segment value */
	i /= 16;
/* get VBE info */
	strcpy(vbe_info->sig, "VBE2");
	regs.eax = 0x4F00;
	regs.edi = 0;
	regs.v_es = i;
	do_v86_int(&regs, 0x10);
	if((regs.eax & 0xFFFF) != 0x004F || memcmp(vbe_info->sig, "VESA", 4))
	{
		kprintf("\t*** No VBE video BIOS\n");
		return -1;
	}
	kprintf("VBE version %u.%u detected; %luK video memory\n",
		vbe_info->ver_major, vbe_info->ver_minor,
		vbe_info->vid_mem_size * 64);
	vbe_ver = vbe_info->ver_major;
/* get mode info for modes 0x100 (640x400x256) or 0x101 (640x480x256) */
	for(i = 0x100; i < 0x102; i++)
	{
		regs.eax = 0x4F01;
		regs.ecx = i;
		do_v86_int(&regs, 0x10);
		if((regs.eax & 0xFFFF) != 0x004F)
			continue;	/* BIOS call failed */
		if((mode_info->mode_attrib & 0x0001) == 0)
			continue;	/* mode not supported */
		if(vbe_ver >= 2 && (mode_info->mode_attrib & 0x0040) == 0)
			continue;	/* no banked framebuffer */
		goto OK;
	}
	kprintf("\t*** VBE modes 0x100 and 0x101 not supported\n");
	return -1;
OK:
/* find readable/writable "window" */
	if((mode_info->win_a_attrib & 0x07) == 0x07)
	{
		g_fb = (char *)(mode_info->win_a_seg * 16L - g_kvirt_to_phys);
		g_use_win_a = 1;
	}
	else if((mode_info->win_b_attrib & 0x07) == 0x07)
	{
		g_fb = (char *)(mode_info->win_b_seg * 16L - g_kvirt_to_phys);
		g_use_win_a = 0;
	}
	else
	{
		kprintf("\t*** Error getting VBE window address\n");
		return -1;
	}
	g_gran_per_64k = 64 / mode_info->k_per_gran;
	g_bytes_per_row = mode_info->bytes_per_row;
	wd = 640;
	ht = (i == 0x100) ? 400 : 480;
/* the moment of truth... */
	g_screen_may_be_fubar = 1;
/* switch to graphics mode */
	regs.eax = 0x4F02;
	regs.ebx = i;
	do_v86_int(&regs, 0x10);
	if((regs.eax & 0xFFFF) != 0x004F)
	{
		kprintf("\t*** Error setting graphics mode 0x%X\n", i);
		return -1;
	}
/* draw 2-color 'X' */
	for(i = 0; i < ht; i++)
	{
		write_pixel8b((wd - ht) / 2 + i,	i,	1);
		write_pixel8b((ht + wd) / 2 - i,	i,	2);
	}
	g_screen_may_be_fubar = 0;
/* delay */
#if 0
	for(i = 0x7FFFFFFuL; i != 0; i--)
		/* nothing */;
#else
/* await key pressed */
	outportb(0x21, ~0x02);
	regs.eflags |= 0x200;	/* ENable interrupts */
	regs.eax = 0;
	do_v86_int(&regs, 0x16);
	outportb(0x21, ~0);
#endif
	return 0;
}
/*****************************************************************************
TIMER INTERRUPT
xxx - this doesn't work -- you don't call do_v86_int()
so the V86 mode stack is undefined!
*****************************************************************************/
static void demo5(void)
{
	unsigned long i;
	uregs_t regs;

	kprintf("Demo 5: timer interrupt\n");
	init_v86_regs(&regs);
/* enable IRQ 0 (timer) interrupt at 8259 chips */
	outportb(0x21, ~0x01);
/* enable interrupts at CPU */
	enable();
/* sit a spell */
	for(i = 0xFFFFFFuL; i != 0; i--)
		/* nothing */;
	disable();
/* clear dangling interrupts */
	outportb(0xA0, 0x20);
	outportb(0x20, 0x20);
	(void)inportb(0x60); /* keyboard */
}
/*****************************************************************************
KEYBOARD HARDWARE (IRQ 1) AND BIOS (INT 16h) INTERRUPTS
AND TEXT OUTPUT USING INT 10h AH=0Eh
*****************************************************************************/
static void demo6(void)
{
	uregs_t regs;
	unsigned i;

	kprintf("Demo 6: INT 16h AH=00h (get key). Press Esc to quit.\n");
	init_v86_regs(&regs);
/* enable IRQ 1 (keyboard) interrupt */
	outportb(0x21, ~0x02);
/* read and write characters until Esc */
	do
	{
/* use INT 16h AH=00h to get key */
		regs.eax = 0x0000;
		regs.eflags |= 0x200;	/* ENable interrupts */
		do_v86_int(&regs, 0x16);
		i = regs.eax & 0xFF;
/* use INT 10h AH=0Eh to display it */
		regs.eax = 0x0E00 | i;
		regs.ebx = 0x0000;
		regs.eflags &= ~0x200;	/* disable interrupts */
		do_v86_int(&regs, 0x10);
	} while(i != 27);
/* leave a blank line */
	kprintf("\n");
}
/*****************************************************************************
INT 13h FLOPPY DISK ACCESS
*****************************************************************************/
static void demo7(void)
{
	unsigned i, j;
	uregs_t regs;

	kprintf("Demo 7: INT 13h AH=02h floppy disk read\n");
	init_v86_regs(&regs);
#if 1
/* enable IRQ 6 (floppy) interrupt -- this works for me */
	outportb(0x21, ~0x40);
#else
/* enable IRQ 6 (floppy) and IRQ 0 (timer) interrupts */
	outportb(0x21, ~0x41);
#endif
	regs.ecx = 0x0001; /* cylinder 0, sector 1 */
	regs.edx = 0x0000; /* head 0, drive 0 */
/* this loads sector 0 of the floppy to 9000:0000
xxx - allocate proper conventional memory buffer here
(maybe this is why the 486 system hangs?) */
	regs.ebx = 0;
	regs.v_es = 0x9000;
	regs.eflags |= 0x200;	/* enable interrupts */
/* make 3 attempts */
	for(i = 2; /*i != 0*/; i--)
	{
		regs.eax = 0x0201; /* read; 1 sector */
		do_v86_int(&regs, 0x13);
		j = (regs.eax >> 8) & 0xFF;
		if(j == 0 || i == 0)
			break;
		regs.eax = 0x0000; /* AH=0: reset drive */
		do_v86_int(&regs, 0x13);
	}
/* if success, dump buffer, else display INT 13h error number */
	if(j == 0)
	{
		kprintf("\tDump of floppy bootsector:\n");
		dump((char *)(0x90000L - g_kvirt_to_phys), 64);
	}
	else
		kprintf("\t*** Error 0x%02X from INT 13h ***\n", j);
}
/*****************************************************************************
*****************************************************************************/
int main(void)
{
	int i;

/* get pmode video working */
	init_video();
/* say hello */
	kprintf("Hello from 32-bit pmode\n");
/* set up 8259 programmable interrupt controller (PIC) chips
for protected mode, and disable interrupts at the chips */
	kprintf("Programming interrupt controller chips for pmode...\n");
	init_8259s(1);
#if 1
/* disallow all I/O */
	kprintf("DISABLING all I/O access in the IOPB...\n");
	memset(g_tss_iopb, 0xFF, g_tss_end - g_tss_iopb);
#else
/* allow all I/O */
	kprintf("Enabling all I/O access in the IOPB...\n");
	memset(g_tss_iopb, 0, g_tss_end - g_tss_iopb);
#endif
	kprintf("\n");
/* set up longjmp() error trapping */
	i = setjmp(g_oops);
	if(i == 0)
	{
		if(demo4() == 0)	/* call VBE BIOS to set graphics mode */
		{
kprintf("*** demo4: success ***\n");
			demo3();	/* call VGA BIOS to set text mode */
kprintf("*** demo3: success ***\n");
		}
		demo1();		/* use VGA BIOS to display text */
kprintf("*** demo1: success ***\n");
		demo2();		/* get address of font in VGA BIOS */
kprintf("*** demo2: success ***\n");
		demo6();		/* keyboard input */
kprintf("*** demo6: success ***\n");
		demo7();		/* read floppy disk bootsector */
kprintf("*** demo7: success ***\n");
		scan_pnp16();
// xxx - need this to prevent system hang after returning to DOS
// not any more??? doesn't help on 486SX system
//longjmp(g_oops, 1);
	}
/* switch to text mode by writing VGA registers directly.
Use this if you crash while trying call the video BIOS mode-set
interrupt via V86 mode */
	if(g_screen_may_be_fubar)
		set_text_mode();
/* dump I/O permission bitmap to see which I/O ports were used */
	kprintf("\n");
	dump_iopb();
	kprintf("Programming interrupt controller chips for real mode...\n");
	init_8259s(0);
// doesn't help on 486SX system
//outportb(0x21, ~0x47); /* enable IRQ 6 (floppy), 2 (cascade), 1 (kbd), 0 (timer) */
//outportb(0xA1, ~0x61); /* enable IRQ 14 (IDE), 13 (i387), 8 (RTC) */
	kprintf("Hope that worked. Returning to DOS...\n");
/* clear dangling interrupts */
// doesn't help on 486SX system
//outportb(0xA0, 0x20);
//outportb(0x20, 0x20);
	(void)inportb(0x60); /* keyboard */
	return 0;
}
